#include <unistd.h>
#include <stdlib.h>
#include <pthread.h>
#include <netdb.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>

#include <sys/types.h>
#include <sys/socket.h>

#include "list.h"
#include "gnet_lib.h"
#include "gnet_channel.h"
#include "gnet_engine.h"
#include "gnet_proto.h"
#include "gnet_msg.h"
#include "gnet_host.h"

static char CONNECT_STRING[] = "GNUTELLA CONNECT/0.6\r\nUser-Agent: lufs\r\nBye-Packet: 0.1\r\n\r\n";
static char CONFIRM_STRING[] = "GNUTELLA/0.6 200 OK\r\n\r\n";

struct channel*
gnet_channel_create(struct gnet *gnet){
    struct channel *chan;

    if(!(chan = malloc(sizeof(struct channel))))
	return NULL;

    memset(chan, 0, sizeof(struct channel));

    INIT_LIST_HEAD(&chan->c_out);
    INIT_LIST_HEAD(&chan->c_guids);
    chan->c_engine = gnet;

    return chan;
}

void
gnet_channel_destroy(struct gnet *gnet, struct channel *chan){
    struct list_head *p, *tmp;
    struct msgref *ref;
    struct bnode *node;

    list_for_each_safe(p, tmp, &chan->c_out){
	ref = list_entry(p, struct msgref, list);

	list_del(&ref->list);
	gnet_delete_message(ref->msg);
	free(ref);
    }

    list_for_each_safe(p, tmp, &chan->c_guids){
	node = list_entry(p, struct bnode, list);

	if(gnet_delete_guid(gnet, node->guid) < 0)
	    WARN("could not delete guid?!");       
    }

    free(chan);    
}

int
gnet_channel_connect(struct channel *chan){
    unsigned long flags;
    struct sockaddr_in addr;

    memset(&addr, 0, sizeof(struct sockaddr_in));
    addr.sin_family = AF_INET;
    addr.sin_port = htons(chan->c_port);
    memcpy(&addr.sin_addr.s_addr, chan->c_addr, 4);

    chan->c_state = CHANNEL_STATE_CONNECTING;
    chan->c_substate = CONNECT_STATE_START;
    chan->c_testwr = 1;
    chan->c_stamp = time(NULL);

    if((chan->c_fd = socket(PF_INET, SOCK_STREAM, 0)) < 0)
	return -1;

    if(fcntl(chan->c_fd, F_GETFL, &flags) < 0){
	close(chan->c_fd);
	return -1;
    }
    
    flags |= O_NONBLOCK;

    if(fcntl(chan->c_fd, F_SETFL, &flags) < 0){
	close(chan->c_fd);
	return -1;
    }

    fcntl(chan->c_fd, F_GETFL, &flags);
    if(!(flags & O_NONBLOCK)){
	ERROR("BLOCKING SOCKET!!!");
    }

    if(connect(chan->c_fd, (struct sockaddr*)&addr, sizeof(struct sockaddr_in)) < 0){
	if(errno != EINPROGRESS){
	    close(chan->c_fd);
	    return -1;
	}
//	TRACE("connection initiated");
    }else{
	WARN("connection completed!?!?!?!");
    }
	    
    return 0;
}

int
gnet_add_peer(struct gnet *gnet, char *host, unsigned short port){
    struct hostent *hst;
    struct channel *chan;

    if(!(hst = gethostbyname(host))){
	WARN("could not resolve host %s: %s", host, strerror(errno));
	return -1;
    }

    if(!(chan = gnet_channel_create(gnet)))
	return -1;

    memcpy(chan->c_addr, hst->h_addr_list[0], 4);
    chan->c_port = port;

    if(gnet_channel_connect(chan) < 0){
	WARN("could not initiate connection: %s", strerror(errno));
	gnet_channel_destroy(gnet, chan);
	return -1;
    }

    pthread_mutex_lock(&gnet->g_channels_lock);

    list_add(&chan->c_list, &gnet->g_channels);
    gnet->g_connecting_peers++;

    pthread_mutex_unlock(&gnet->g_channels_lock);

    gnet_engine_signal(gnet, 0);

//    TRACE("new peer added to list...");

    return 0;
}

void
gnet_test_rd(struct gnet *gnet, struct channel *chan){

    chan->c_testrd = 1;

    FD_SET(chan->c_fd, &gnet->g_rdset);
    FD_SET(chan->c_fd, &gnet->g_exset);

    if(chan->c_fd > gnet->g_maxfd)
	gnet->g_maxfd = chan->c_fd;
}

void
gnet_test_wr(struct gnet *gnet, struct channel *chan){

    chan->c_testwr = 1;

    FD_SET(chan->c_fd, &gnet->g_wrset);
    FD_SET(chan->c_fd, &gnet->g_exset);

    if(chan->c_fd > gnet->g_maxfd)
	gnet->g_maxfd = chan->c_fd;
}

void
gnet_untest_rd(struct gnet *gnet, struct channel *chan){
    
    chan->c_testrd = 0;

    FD_CLR(chan->c_fd, &gnet->g_rdset);

    if(!FD_ISSET(chan->c_fd, &gnet->g_wrset)){
	FD_CLR(chan->c_fd, &gnet->g_exset);
	if(chan->c_fd == gnet->g_maxfd)
	    gnet->g_maxfd--;
    }
}

void
gnet_untest_wr(struct gnet *gnet, struct channel *chan){

    chan->c_testwr = 0;

    FD_CLR(chan->c_fd, &gnet->g_wrset);

    if(!FD_ISSET(chan->c_fd, &gnet->g_rdset)){
	FD_CLR(chan->c_fd, &gnet->g_exset);
	if(chan->c_fd == gnet->g_maxfd)
	    gnet->g_maxfd--;
    }
}

void
gnet_drop_channel(struct gnet *gnet, struct channel *chan){
    

    TRACE("dropping channel %d", chan->c_fd);
    
    if(CONNECTED(chan))
	gnet->g_connected_peers--;
    else
	gnet->g_connecting_peers--;

    gnet_untest_rd(gnet, chan);
    gnet_untest_wr(gnet, chan);

    close(chan->c_fd);

    list_del(&chan->c_list);
    gnet_channel_destroy(gnet, chan);

    TRACE("peers left: %d", gnet->g_connected_peers);

    if(gnet->g_connected_peers < gnet->g_cfg->keep_peers){
//	TRACE("trying to open some more connections");
	gnet_check_peers(gnet);
    }    
}

static void
handshake(struct gnet *gnet, struct channel *chan, int op){
    int res;

//    TRACE("substate: %d, op: %d", chan->c_substate, op);

    chan->c_stamp = time(NULL);

    switch(chan->c_substate){
    case CONNECT_STATE_START:
	if(op == GNET_OP_WRITE){
	    socklen_t len;
	    
	    len = sizeof(int);
	    if(getsockopt(chan->c_fd, SOL_SOCKET, SO_ERROR, &res, &len) < 0){
		WARN("could not get sock options!");
		gnet_drop_channel(gnet, chan);
		break;
	    }

	    if(res){
		WARN("connect failed: %s", strerror(res));
		gnet_drop_channel(gnet, chan);
		break;
	    }

//	    TRACE("connected...");

	    chan->c_substate = CONNECT_STATE_HANDSHAKE1;
	    chan->c_out_index = 0;
	}
	break;

    case CONNECT_STATE_HANDSHAKE1:
    {
	int to_send;

	to_send = strlen(CONNECT_STRING) - chan->c_out_index;

	if((res = write(chan->c_fd, CONNECT_STRING + chan->c_out_index, to_send)) <= 0){
	    WARN("write failed: %s", strerror(errno));
	    gnet_drop_channel(gnet, chan);
	    break;
	}

	chan->c_out_index += res;

	if(res == to_send){
//	    TRACE("connect string sent...");

	    chan->c_substate = CONNECT_STATE_HANDSHAKE2;
	    chan->c_in_index = 0;
	    gnet_test_rd(gnet, chan);
	    gnet_untest_wr(gnet, chan);
	}
    }    
	break;

    case CONNECT_STATE_HANDSHAKE2:
	if((res = read(chan->c_fd, chan->c_buf + chan->c_in_index, MAX_BUFFER - chan->c_in_index)) <= 0){
	    WARN("read failed: %s", strerror(errno));
	    gnet_drop_channel(gnet, chan);
	    break;
	}

//	TRACE("read %d bytes", res);

	if((chan->c_in_index += res) >= 4){
	    if(!strncmp(chan->c_buf + chan->c_in_index - 4, "\r\n\r\n", 4)){
//		TRACE("connect string received...");

		if(sscanf(chan->c_buf, "GNUTELLA/0.%*c %d ", &res) != 1){
		    WARN("could not read status code!");
		    gnet_drop_channel(gnet, chan);
		    break;
		}
		
		if(res != 200){
		    WARN("connection rejected: %d", res);
		    gnet_get_hosts(gnet, chan->c_buf);
		    gnet_drop_channel(gnet, chan);
		    break;
		}

		chan->c_substate = CONNECT_STATE_HANDSHAKE3;
		chan->c_out_index = 0;
		gnet_test_wr(gnet, chan);
		gnet_untest_rd(gnet, chan);
	    }	
	}

	break;

    case CONNECT_STATE_HANDSHAKE3:
    {
	int to_send;

	to_send = strlen(CONFIRM_STRING) - chan->c_out_index;

	if((res = write(chan->c_fd, CONFIRM_STRING + chan->c_out_index, to_send)) <= 0){
	    WARN("write failed: %s", strerror(errno));
	    gnet_drop_channel(gnet, chan);
	    break;
	}

	chan->c_out_index += res;

	if(res == to_send){
	    TRACE("channel connected :)");

	    chan->c_state = CHANNEL_STATE_CONNECTED;
	    chan->c_in_index = 0;
	    chan->c_out_index = 0;
	    gnet_test_rd(gnet, chan);
	    gnet_untest_wr(gnet, chan);

	    gnet->g_connecting_peers--;
	    gnet->g_connected_peers++;
	}
    }        
    break;

    default:
	WARN("unknown state %d", chan->c_substate);
	gnet_drop_channel(gnet, chan);
    }

}

static int
io_read(struct gnet *gnet, struct channel *chan){
    int res, len;

    if(chan->c_in_index < GNET_HDR_SIZE){
//	TRACE("reading header...");
	
	len = GNET_HDR_SIZE - chan->c_in_index;

    }else{
//	TRACE("reading message...");
	if(GNET_SIZE(chan->c_buf) >= MAX_BUFFER - GNET_HDR_SIZE){
	    ERROR("!!!message too long: %d!!!", GNET_SIZE(chan->c_buf));
	    return 0;
	}

//	TRACE("msg size: %d", GNET_SIZE(chan->c_buf));

	len = GNET_HDR_SIZE + GNET_SIZE(chan->c_buf) - chan->c_in_index;	
    }

    if((res = read(chan->c_fd, chan->c_buf + chan->c_in_index, len)) <= 0)
	return res;

//    TRACE("read %d bytes", res);

    chan->c_in_index += res;
    chan->c_stamp = time(NULL);
    
    return res;
}

static int
io_write(struct gnet *gnet, struct channel *chan){
    struct msgref *ref;
    int len, res;
    
    ref = list_entry(chan->c_out.next, struct msgref, list);
    len = GNET_HDR_SIZE + GNET_SIZE(ref->msg->m_data) - chan->c_out_index;
    
//    TRACE("trying to write %d bytes", len);
	
    if((res = write(chan->c_fd, ref->msg->m_data + chan->c_out_index, len)) <= 0)
	return res;
    
//    TRACE("wrote %d bytes", res);
	
    chan->c_out_index += res;
    chan->c_stamp = time(NULL);

    return res;
}

void
gnet_channel_io(struct gnet *gnet, struct channel *chan, int op){
    struct msgref *ref;
    int res;

//    TRACE("fd: %d, op: %d", chan->c_fd, op);

    switch(chan->c_state){
    case CHANNEL_STATE_CONNECTING:
	handshake(gnet, chan, op);
	break;

    case CHANNEL_STATE_CONNECTED:
	if(op & GNET_OP_READ){
	    if((res = io_read(gnet, chan)) <= 0){
		if(res < 0)
		    WARN("read failed: %d(%s)", errno, strerror(errno));
		else
		    WARN("connection closed by peer");

		gnet_drop_channel(gnet, chan);
		break;
	    }
		
	    if(chan->c_in_index == GNET_HDR_SIZE + GNET_SIZE(chan->c_buf)){
//		TRACE("read a whole message");
		
//		gnet_trace_message(chan->c_buf);

		if(gnet_handle_message(gnet, chan) < 0){
		    WARN("processing failed!");
		    gnet_drop_channel(gnet, chan);
		    break;
		}
		
		chan->c_in_index = 0;
	    }
	}

	if((op & GNET_OP_WRITE) && (!list_empty(&chan->c_out))){
	    ref = list_entry(chan->c_out.next, struct msgref, list);

	    if((res = io_write(gnet, chan)) <= 0){
		if(res < 0)
		    WARN("write failed: %d(%s)", errno, strerror(errno));
		gnet_drop_channel(gnet, chan);
		break;
	    }

	    if(chan->c_out_index == GNET_HDR_SIZE + GNET_SIZE(ref->msg->m_data)){
//		TRACE("wrote a whole message...");
		    
//		    gnet_trace_message(ref->msg->m_data);

		chan->c_out_index = 0;
		list_del(&ref->list);
		gnet_delete_message(ref->msg);
		free(ref);

		if(list_empty(&chan->c_out)){
//		    TRACE("no more messages in queue.");
		    gnet_untest_wr(gnet, chan);
		}
	    }
	}
	

	break;
	
    default:
	WARN("unknown state: %d", chan->c_state);
	gnet_drop_channel(gnet, chan);
    }
}
