/*
 * locasefs.cpp
 *
 * This file is part of LUFS, a free userspace filesystem implementation.
 * See http://lufs.sourceforge.net/ for updates.
 *
 * LUFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * LUFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 TTimo: filename casing operations
 Use case:
   Win32 filesystem is basically case insensitive.
	 Casing in file names is supported, but not relevant when opening files, a case insensitive matching is performed.
	 If working with source code obtained from win32 on *nix systems, you often have to manually convert into a lowercase-only tree to be safe
	 Getting into a more complicated situation where the *nix version has to interoperate with the win32 source, without
	 the ability to change all the casing to be correctly lowercase in the win32 master source.
 Objective:
   Provide a lowercase mapping of a filesystem (read/write).
 Discussion:
   There are probably several operating options, such as supporting a case-sensitive filesystem a-la-win32, mapping to uppercase, mapping to lowercase ..	 
 Implementation:
   get the lowercase mount as a flag over regular localfs?
   NOTE: what happens if root has something like: File and file in same dir? first found will be selected
*/
 

#include <unistd.h>
#include <dirent.h>
#include <fcntl.h>
#include <stdio.h>
#include <utime.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <lufs/proto.h>
#include <lufs/fs.h>

#include "locasefs.h"

extern "C"{

void*
locasefs_init(struct list_head *cfg, struct dir_cache *cache, struct credentials *cred, void **global_ctx){
    return (void*)new LocaseFS(cfg, cache, cred);
}

void
locasefs_free(void *ctx){
    LocaseFS *p = (LocaseFS*)ctx;

    delete p;
}

int 	
locasefs_mount(void *ctx){
    return ((LocaseFS*)ctx)->do_mount();
}

void 	
locasefs_umount(void *ctx){
//    return ((LocaseFS*)ctx)->do_umount();
}

int 	
locasefs_readdir(void *ctx, char *dir_name, struct directory *dir){
    return ((LocaseFS*)ctx)->do_readdir(dir_name, dir);
}

int 	
locasefs_stat(void *ctx, char *name, struct lufs_fattr *fattr){
    return ((LocaseFS*)ctx)->do_stat(name, fattr);
}

int 	
locasefs_mkdir(void *ctx, char *dir, int mode){
    return ((LocaseFS*)ctx)->do_mkdir(dir, mode);
}

int 	
locasefs_rmdir(void *ctx, char *dir){
    return ((LocaseFS*)ctx)->do_rmdir(dir);
}

int 	
locasefs_create(void *ctx, char *file, int mode){
    return ((LocaseFS*)ctx)->do_create(file, mode);
}

int 	
locasefs_unlink(void *ctx, char *file){
    return ((LocaseFS*)ctx)->do_unlink(file);
}

int 	
locasefs_rename(void *ctx, char *old_name, char *new_name){
    return ((LocaseFS*)ctx)->do_rename(old_name, new_name);
}

int 	
locasefs_open(void *ctx, char *file, unsigned mode){
    return ((LocaseFS*)ctx)->do_open(file, mode);
}

int 	
locasefs_release(void *ctx, char *file){
    return ((LocaseFS*)ctx)->do_release(file);
}

int 	
locasefs_read(void *ctx, char *file, long long offset, unsigned long count, char *buf){
    return ((LocaseFS*)ctx)->do_read(file, offset, count, buf);
}

int	
locasefs_write(void *ctx, char *file, long long offset, unsigned long count, char *buf){
    return ((LocaseFS*)ctx)->do_write(file, offset, count, buf);
}

int 	
locasefs_readlink(void *ctx, char *link, char *buf, int buflen){
    return ((LocaseFS*)ctx)->do_readlink(link, buf, buflen);
}

int 	
locasefs_link(void *ctx, char *target, char *link){
    return ((LocaseFS*)ctx)->do_link(target, link);
}

int 	
locasefs_symlink(void *ctx, char *target, char *link){
    return ((LocaseFS*)ctx)->do_symlink(target, link);
}

int 	
locasefs_setattr(void *ctx, char *file, struct lufs_fattr *fattr){
    return ((LocaseFS*)ctx)->do_setattr(file, fattr);
}

}

/***** Some generic code *****/

/* Constructor.
 * Importance: 10
 * Nothing interesting here, just call the superclass' constructor
 */
LocaseFS::LocaseFS(struct list_head *c, struct dir_cache *cache, struct credentials *cred){
    const char *root;

    TRACE("in LocaseFS constructor");

    cfg = c;
    this->cache = cache;
    this->cred = cred;

    strcpy(root_dir, "/");
    if((root = lu_opt_getchar(c, "MOUNT", "root")))
	strcpy(root_dir, root);

    if(!root_dir[0])
	strcpy(root_dir, "/");

    if (root_dir[strlen(root_dir)-1] == '/')
        root_dir[strlen(root_dir)-1] = '\0';

    TRACE("root dir: " << root_dir);
}

/* Destructor.
 * Importance: 10
 * Not too intresting either...
 */
LocaseFS::~LocaseFS(){
    TRACE("in LocaseFS destructor");
}


/***** Utilities *****/

/* get_reference_path
 * Retrieve the reference path from a local dir
 * depending on lowercase, it's a simple copy or a lookup
 * works on dirs and files
 */
bool
LocaseFS::get_reference_path(const char *d, string &ref_d)
{
    char *p;

    TRACE("path: "<<d);

    p = strstr(d, root_dir); 
    if (!p)
	{
	    WARN("Couldn't extract root path from: " << d);
	    return false;
	}
    ref_d = root_dir; // the root is unchanged
    char *local_p = p + strlen(root_dir);
    TRACE("local_p: "<<local_p);

    if (local_p[0]=='/')
	local_p++;
    if (local_p[0]=='\0'){
	TRACE("early exit");
	return true; // this is the root of the mount, early exit
    }
    // from there we have to walk down and find the correct uppercase versions..
    // we know we have at least one item to match
    char match[PATH_MAX];
    char *match_p,*match_cut;
    bool bEndLoop = false; // set to true when we are on the last match
    strcpy(match, local_p);
    match_p = match;
    match_cut = strchr(match_p, '/');
    if (match_cut)
    {
	if (match_cut[1]=='\0')
	    bEndLoop = true; // stop after this, it's only a trailing /
	match_cut[0] = '\0';
    }
    else
	bEndLoop = true;

    TRACE("ref_d: "<<ref_d);

    // start reading
    // we use the current ref_d we have been building (it grows as we descend dirs)
    DIR *dir = ::opendir(ref_d.c_str());
    struct dirent *dent;
    if (!dir)
    {
	WARN("could not open directory!" << ref_d.c_str());
	return false;
    }
    while((dent = ::readdir(dir))){
	if (!strcasecmp(match_p, dent->d_name))
	{
	    // got a match, go down one level
	    ref_d += '/';
	    ref_d += dent->d_name;
	    ::closedir(dir);
	    if (bEndLoop)
		break; // we are done
	    // new match
	    match_p = match_cut; match_p++;
	    match_cut = strchr(match_p, '/');
	    if (match_cut)
	    {
		if (match_cut[1]=='\0')
		    bEndLoop = true; // stop after this, it's only a trailing /
		match_cut[0] = '\0';
	    }
	    else
		bEndLoop = true;
	    // setup the next loop
	    dir = ::opendir(ref_d.c_str());
	    if (!dir)
	    {
		WARN("could not open directory!" << ref_d.c_str());
		return false;		    
	    }
	}
    }
    if (!dent)
    {
	// we ran through the dir and didn't find any match
	WARN("Didn't find case match for " << match_p);
	::closedir(dir);
	return false;
    }
    TRACE(d << " matched to " << ref_d);
    return true;
}

/* get_top_reference_path
 * get the reference path of the top directory (for a file or directory)
 * leave the file unchanged (also works for dirs)
 * this is used for dir/file creations
 */
bool
LocaseFS::get_top_reference_path(const char *d, string &ref_top_d)
{
    unsigned i;
    string s_dir = d;
    i = s_dir.find_last_of('/');
    if (i == string::npos){
	WARN("Could not isolate top directory from: " << d);
	return -1;
    }
    string top_dir = (i == 0) ? string("/") : s_dir.substr(0, i);
    string file = s_dir.substr(i + 1, s_dir.length() - i - 1);	
    if(!get_reference_path(top_dir.c_str(), ref_top_d))
	return false;
    ref_top_d += string("/");
    ref_top_d += file;
    TRACE("matched up to :" << ref_top_d.c_str());	
    return true;
}

/* int_do_stat: stat a file.
 * Importance: 10
 * Will read a file/dir/link's attributes and store them in fattr.
 *
 */
int
LocaseFS::int_do_stat(const char *nm, struct lufs_fattr *fattr){
    struct stat stat;
    int res;
    
    if((res = ::lstat(nm, &stat)) < 0)
	return res;

    fattr->f_mode = stat.st_mode;
    fattr->f_nlink = stat.st_nlink;
    fattr->f_uid = (getuid() == stat.st_uid) ? 1 : 0;
    fattr->f_gid = (getgid() == stat.st_gid) ? 1 : 0;
    fattr->f_size = stat.st_size;
    fattr->f_atime = stat.st_atime;
    fattr->f_mtime = stat.st_mtime;
    fattr->f_ctime = stat.st_ctime;

    return 0;
}


/***** Next come the callbacks, associated with their respective VFS operations *****/

/* do_mount: mount the filesystem
 * Importance: 10
 * This will initialize the filesystem (create connections,
 * authenticate users, etc).
 * For localfs there's nothing to initialize...
 */
int
LocaseFS::do_mount(){
    TRACE("mounting a localfs.");
    return 1;
}

/* do_readdir: read a directory in cache.
 * Importance: 10
 * go through all the entries in the given dir and add them to 
 * the directory structure (lu_cache_add2dir(ddir, filename, link, fattr)).
 * 
 * fields of fattr that must be filled:
 * * f_mode
 * * f_nlink
 * * f_uid - should be != 0 if we have owner permissions on the entry, 0 otherwise.
 * * f_gid - should be != 0 if we have group permissions on the entry, 0 otherwise.
 * * f_size
 * * f_atime
 * * f_ctime
 * * f_mtime

 * TTimo - bLowCase
 *   remap all the names to lowercase
 */ 

int
LocaseFS::do_readdir(char* d, struct directory *ddir){
    DIR *dir;
    struct lufs_fattr fattr;
    int res;
    string s;
    string ref_d; // reference directory name

    TRACE("dir: "<< d);

    if (!get_reference_path(d, ref_d))
	return -1;
    
    if(!(dir = ::opendir(ref_d.c_str()))){
	WARN("could not open directory!");
	return -1;
    }

    while(struct dirent *dent = ::readdir(dir)){
	// directly convert d_name to lowercase
	// do we have any interest in keep the reverse mapping?
	// (for faster lookups probably?)
	// depends .. if we look into this systematically when we open the files.. then we should do this
	
	char d_name_low[PATH_MAX];
	strcpy(d_name_low, dent->d_name);
	
	char *p = d_name_low;
	while(*p) { *p = tolower(*p); p++; }
	
	TRACE("adding direntry " << dent->d_name << " as " << d_name_low);
	
	// ref_d never has trailing slash
	s = ref_d + '/' + dent->d_name;
 
	// here we need to stat the reference file

	TRACE("stating file " << s.c_str());
	
	if((res = int_do_stat((char*)s.c_str(), &fattr)) < 0){
	    WARN("could not stat file!");
	    ::closedir(dir);
	    return res;
	}
	
	// here we should be adding the lowercase version
	lu_cache_add2dir(ddir, d_name_low, NULL, &fattr);
	
    }

    ::closedir(dir);

    return 0;
}

/* do_stat: stat a file.
 * Importance: 10
 * Will read a file/dir/link's attributes and store them in fattr.
 * lowercase: will lookup the stat on the reference case version
 */
int
LocaseFS::do_stat(char *nm, struct lufs_fattr *fattr){
    string ref_nm;
    if (!get_reference_path(nm, ref_nm))
	return -1;
    
    return int_do_stat(ref_nm.c_str(), fattr);
}

/* do_mkdir: create a directory.
 * Importance: 5
 * Will create the named directory with the given permissions.
 * lowercase: we have to do some lookup to matchup the top dir
 */
int
LocaseFS::do_mkdir(char *dir, int mode){
    string ref_top_dir;
    if (!get_top_reference_path(dir, ref_top_dir))
	return -1;
    
    return ::mkdir(ref_top_dir.c_str(), mode);
}

/* do_rmdir: erase a directory.
 * Importance: 5
 * Will erase the named directory.
 * lowercase: lookup first
 */
int
LocaseFS::do_rmdir(char *dir){
    string ref_dir;
    if (!get_reference_path(dir, ref_dir))
	return -1;    

    return ::rmdir(ref_dir.c_str());
}

/* do_create: create a file.
 * Importance: 6
 * Will create the named file with the given permissions.
 * lowercase: lookup the top dir of the file to create
 */
int
LocaseFS::do_create(char *file_, int mode){
    string ref_top_file;
    if (!get_top_reference_path(file_, ref_top_file))
	return -1;	
    
    TRACE("create "<<ref_top_file.c_str()<<",mode: "<<std::oct<<mode<<std::hex);
    return ::mknod(ref_top_file.c_str(), mode, 0);
}

/* do_unlink: delete a file.
 * Importance: 6
 * Will delete the named file.
 * lowercase: lookup
 */
int
LocaseFS::do_unlink(char *file){
    string ref_file;
    if (!get_reference_path(file, ref_file))
	return -1;    
    
    return ::unlink(ref_file.c_str());
}

/* do_rename: rename a file/dir.
 * Importance: 4
 * Rename the file/dir from the old to the new name.
 * lowercase: regular lookup on old, top match on nnew
 */
int
LocaseFS::do_rename(char *old, char *nnew){
    string ref_old;
    if (!get_reference_path(old, ref_old))
	return -1;    
    string ref_top_nnew;
    if (!get_top_reference_path(nnew, ref_top_nnew))
	return -1;

    return ::rename(ref_old.c_str(), ref_top_nnew.c_str());
}

/* do_open: open a file
 * Importance: 1
 * This will be called *after* the actual generic file opening,
 * just in case your filesystem needs to do some work here.
 * The return value is ignored by the module.
 * lowercase: lookup the full file, and lookup the top also
 */
int
LocaseFS::do_open(char *file, unsigned mode){
    TRACE("open "<<file<<" mode="<<std::oct<<mode<<std::hex); 

    return 1;
}

/* do_release: close a file.
 * Importance: 8
 * Will close an open file.
 * For localfs, nothig to do here.
 */
int
LocaseFS::do_release(char *file){
    TRACE("release "<<file);

    return 1;
}

/* do_read: read from a file.
 * Importance: 9
 * Read "count" bytes from "file" at "offset", in buffer "buf".
 * For localfs we need to open the file too...
 */
int
LocaseFS::do_read(char *file_, long long offset, unsigned long count, char *buf){
    FILE *f;
    int res;
    string ref_file;
    if (!get_reference_path(file_, ref_file))
	{
	    return -1;
	}
    
    if((f = fopen(ref_file.c_str(), "rb")) == NULL){
	TRACE("fopen failed");
	return -1;
    }

    if(fseek(f, offset, SEEK_SET) < 0){
	fclose(f);
	TRACE("fseek failed");
	return -1;
    }

    res = fread(buf, 1, count, f);

    TRACE("read: "<<res);

    fclose(f);
    return res;
}

/* do_write: write to a file.
 * Importance: 5
 * Write "count" bytes from "buf" to "file", at "offset".
 * For localfs we need to open the file too...
 * lowercase: we lookup, but it's not critical if we don't find (we created)
 * when we create it's possible that we create case sensitive
 */
int
LocaseFS::do_write(char *file_, long long offset, unsigned long count, char *buf){
    FILE *fd;
    string ref_file;
    if (!get_reference_path(file_, ref_file))
	{
	    TRACE("failed case lookup, create");
	    if (!get_top_reference_path(file_, ref_file))
		return -1; // if this failed too, it's bad
	}
    
    if((fd = fopen(ref_file.c_str(), "r+b")) == NULL){
	TRACE("open failed");
	return -1;
    }

    if(fseek(fd, offset, SEEK_SET) < 0){
	fclose(fd);
	TRACE("lseek failed");
	return -1;
    }

    if(fwrite(buf, count, 1, fd) < 0){
	fclose(fd);
	TRACE("write failed");
	return -1;
    }

    fclose(fd);
    return 1;
}

/* do_readlink: resolve a link.
 * Importance: 7
 * Resolve the link and place the target in "buf".
 * lowercase: lookup
 */
int
LocaseFS::do_readlink(char *link, char *buf, int buflen){
    string ref_link;
    if (!get_reference_path(link, ref_link))
	return -1;
    
    return ::readlink(ref_link.c_str(), buf, buflen);
}

/* do_link: create link.
 * Importance: 3
 * Create a link from "old" to "nnew".
 * lowercase: lookup
 */
int
LocaseFS::do_link(char *old, char *nnew){
    string ref_old;
    if (!get_reference_path(old, ref_old))
		return -1;
	string ref_nnew;
	if (!get_top_reference_path(nnew, ref_nnew))
		return -1;
    
    return ::link(ref_old.c_str(), ref_nnew.c_str());
}

/* do_symlink: create a symlink.
 * Importance: 3
 * Create a symlink from "old" to "nnew".
 * lowercase: lookup
 */
int
LocaseFS::do_symlink(char *old, char *nnew){
    string ref_old;
    if (!get_reference_path(old, ref_old))
	return -1;
    string ref_nnew;
    if (!get_top_reference_path(nnew, ref_nnew))
	return -1;
    
    TRACE("symlink "<<ref_old.c_str()<<" <=> "<<ref_nnew.c_str());
    return ::symlink(ref_old.c_str(), ref_nnew.c_str());
}

/* do_setattr: set file/dir attributes.
 * Importance: 8
 * Set the file/dir attributes stored in "fattr".
 * lowercase: lookup
 */
int
LocaseFS::do_setattr(char *f_, struct lufs_fattr *fattr){
    struct stat stat;
    int res;

    string ref_f;
    if (!get_reference_path(f_, ref_f))
	return -1;

    if((res = ::lstat(ref_f.c_str(), &stat)) < 0)
	goto out;

    if(stat.st_mode != fattr->f_mode){
	TRACE("set mode "<<fattr->f_mode<<" old="<<stat.st_mode);
	if((res = chmod(ref_f.c_str(), fattr->f_mode)) < 0)
	    goto out;
    }

    if((stat.st_atime != (time_t)fattr->f_atime) || (stat.st_mtime != (time_t)fattr->f_mtime)){
	struct utimbuf utim = {fattr->f_atime, fattr->f_mtime};

	TRACE("set times "<<fattr->f_atime<<","<<fattr->f_mtime<<" old="<<stat.st_atime<<","<<stat.st_mtime);
	if((res = utime(ref_f.c_str(), &utim)) < 0)
	    goto out;
    }

  out:
    return res;    
}
