#line 2 "alloc.c"
/* 
 * This file is almost exactly the same as great Apache httpd's alloc.c code. 
 * As it's not wise to use two separate memory handling mechanisms, I
 * decided to adopt Apache's one. Below is theirs copyright. The code
 * is changed just a bit - I've deleted a few thing I don't need and
 * changed function definitions to be K&R C compatible.
 *
 * Jaromir Dolecek <dolecek@ics.muni.cz> 
 */

/* ====================================================================
 * Copyright (c) 1995-1997 The Apache Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by the Apache Group
 *    for use in the Apache HTTP server project (http://www.apache.org/)."
 *
 * 4. The names "Apache Server" and "Apache Group" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission.
 *
 * 5. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by the Apache Group
 *    for use in the Apache HTTP server project (http://www.apache.org/)."
 *
 * THIS SOFTWARE IS PROVIDED BY THE APACHE GROUP ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Group and was originally based
 * on public domain software written at the National Center for
 * Supercomputing Applications, University of Illinois, Urbana-Champaign.
 * For more information on the Apache Group and the Apache HTTP server
 * project, please see <http://www.apache.org/>.
 *
 */

/* $Id: alloc.c,v 1.30 2000/01/18 22:31:04 dolecek Exp $ */

#include "csacek.h"

#ifdef HAVE_FCNTL_H
#  include <fcntl.h>
#endif

#ifdef CSACEK

#define BLOCK_MINFREE	512

/*
 * Local functions.
 */
static union block_hdr *malloc_block __P((int));
static void free_blocks __P((union block_hdr *));
static union block_hdr *new_block __P((int));
static void file_cleanup __P((void *));
static void file_child_cleanup __P((void *));

/*
 * Resource allocation code... the code here is responsible for making
 * sure that nothing leaks.
 *
 * rst --- 4/95 --- 6/95
 */

/*****************************************************************
 *
 * Managing free storage blocks...
 */

struct pool {
  union block_hdr *first;
  union block_hdr *last;
  struct cleanup *cleanups;
  struct process_chain *subprocesses;
  struct pool *sub_pools;
  struct pool *sub_next;
  struct pool *sub_prev;
  struct pool *parent;
  char *free_first_avail;
};

union align
{
  /* Types which are likely to have the longest RELEVANT alignment
   * restrictions...
   */

  char *cp;
  void (*f)__P((void));
  long l;
  FILE *fp;
  double d;
};

#define CLICK_CZ (sizeof(union align))

union block_hdr
{
  union align a;

  /* Actual header... */

  struct {
    char *endp;
    union block_hdr *next;
    char *first_avail;
  } h;
};
#define CLICK_CZ (sizeof(union align))

static union block_hdr *csacek_block_freelist = NULL;
static struct pool *permanent_pool;

#ifdef CSA_THREADSAFE
static csa_mutex *alloc_mutex=NULL;
#endif

/* this should be called on startup only */
struct pool *
ap_init_alloc()
{
#ifdef CSA_THREADSAFE
	alloc_mutex = csa_create_mutex(NULL);
#endif
	permanent_pool = ap_make_sub_pool(NULL);
	return permanent_pool;
}

/* this should be called when we want to free all allocated memory 
 * caller has to ensure it's safe to do so, i.e. block threads and be sure
 * the memory is not referenced anywhere */
void
csa_done_alloc()
{
	union block_hdr *ob, *nb;
	if (!permanent_pool) return;

	ap_destroy_pool(permanent_pool);
	ob = csacek_block_freelist;
	while(ob) {
		nb = ob->h.next;
		free((void *)ob);
		ob = nb;
	}
#ifdef CSA_THREADSAFE
	csa_release_mutex(alloc_mutex);
#endif
	csacek_block_freelist = NULL;
}

/* Get a completely new block from the system pool. Note that we rely on
malloc() to provide aligned memory. */

static union block_hdr *
malloc_block (size)
  int size;
{
  union block_hdr *blok =
    (union block_hdr *)malloc(size + sizeof(union block_hdr));

  if (blok == NULL) csa_alloc_fail();

  blok->h.next = NULL;
  blok->h.first_avail = (void *)(blok + 1);
  blok->h.endp = size + blok->h.first_avail;
  
  return blok;
}



/* Free a chain of blocks --- must be called with alarms blocked. */

static void 
free_blocks (blok)
  union block_hdr *blok;
{
  /* First, put new blocks at the head of the free list ---
   * we'll eventually bash the 'next' pointer of the last block
   * in the chain to point to the free blocks we already had.
   */
  
  union block_hdr *old_free_list;

  if (blok == NULL) return;	/* Sanity check --- freeing empty pool? */
  
  csa_acquire_mutex(alloc_mutex);
  old_free_list = csacek_block_freelist;
  csacek_block_freelist = blok;
  
  /*
   * Next, adjust first_avail pointers of each block --- have to do it
   * sooner or later, and it simplifies the search in new_block to do it
   * now.
   */

  while (blok->h.next != NULL) {
    blok->h.first_avail = (void *)(blok + 1);
    blok = blok->h.next;
  }

  blok->h.first_avail = (void *)(blok + 1);

  /* Finally, reset next pointer to get the old free blocks back */

  blok->h.next = old_free_list;
  csa_release_mutex(alloc_mutex);
}




/* Get a new block, from our own free list if possible, from the system
 * if necessary.  Must be called with alarms blocked.
 */

static union block_hdr *
new_block (min_size)
  int min_size;
{
  union block_hdr **lastptr = &csacek_block_freelist;
  union block_hdr *blok = csacek_block_freelist;
  
  /* First, see if we have anything of the required size
   * on the free list...
   */

  min_size += BLOCK_MINFREE;

  while (blok != NULL) {
    if (min_size <= blok->h.endp - blok->h.first_avail) {
      *lastptr = blok->h.next;
      blok->h.next = NULL;
      return blok;
    }
    else {
      lastptr = &blok->h.next;
      blok = blok->h.next;
    }
  }

  /* Nope. */

  return malloc_block (min_size);
}


/*****************************************************************
 *
 * Pool internals and management...
 * NB that subprocesses are not handled by the generic cleanup code,
 * basically because we don't want cleanups for multiple subprocesses
 * to result in multiple three-second pauses.
 */

struct process_chain;
struct cleanup;

static void run_cleanups __P((struct cleanup *));

/* Each pool structure is allocated in the start of its own first block,
 * so we need to know how many bytes that is (once properly aligned...).
 * This also means that when a pool's sub-pool is destroyed, the storage
 * associated with it is *completely* gone, so we have to make sure it
 * gets taken off the parent's sub-pool list...
 */

#define POOL_HDR_CLICKS (1 + ((sizeof(struct pool) - 1) / CLICK_CZ))
#define POOL_HDR_BYTES (POOL_HDR_CLICKS * CLICK_CZ)			 

struct pool *
ap_make_sub_pool (p)
  struct pool *p;
{
  union block_hdr *blok;
  struct pool *new_pool;

  csa_acquire_mutex(alloc_mutex);

  blok = new_block (0);
  /* LINTED */ /* lint: possible pointer alignment problem, it's okay here */
  new_pool = (struct pool *)blok->h.first_avail;
  blok->h.first_avail += POOL_HDR_BYTES;

  memset((void *)new_pool, 0, sizeof (struct pool));
  new_pool->free_first_avail = blok->h.first_avail;
  new_pool->first = new_pool->last = blok;
    
  if (p) {
    new_pool->parent = p;
    new_pool->sub_next = p->sub_pools;
    if (new_pool->sub_next) new_pool->sub_next->sub_prev = new_pool;
    p->sub_pools = new_pool;
  }

  csa_release_mutex(alloc_mutex);

  return new_pool;
}

void 
ap_clear_pool (a)
  struct pool *a;
{
  csa_acquire_mutex(alloc_mutex);
  while (a->sub_pools)
    ap_destroy_pool (a->sub_pools);
  a->sub_pools = NULL;

  csa_release_mutex(alloc_mutex);
  
  /* don't hold the mutex during cleanup */
  run_cleanups (a->cleanups);        a->cleanups = NULL;
  free_blocks (a->first->h.next);    a->first->h.next = NULL;

  a->last = a->first;
  a->first->h.first_avail = a->free_first_avail;
}

void 
ap_destroy_pool (a)
  struct pool *a;
{
  ap_clear_pool (a);

  csa_acquire_mutex(alloc_mutex);

  if (a->parent) {
    if (a->parent->sub_pools == a) a->parent->sub_pools = a->sub_next;
    if (a->sub_prev) a->sub_prev->sub_next = a->sub_next;
    if (a->sub_next) a->sub_next->sub_prev = a->sub_prev;
  }
  csa_release_mutex(alloc_mutex);
  
  free_blocks (a->first);
}

/*****************************************************************
 *
 * Allocating stuff...
 */


void *
ap_palloc (a,reqsize)
  struct pool *a;
  int reqsize;
{
  /* Round up requested size to an even number of alignment units (core clicks)
   */
  
  int nclicks = 1 + ((reqsize - 1) / CLICK_CZ);
  int size = nclicks * CLICK_CZ;

  /* First, see if we have space in the block most recently
   * allocated to this pool
   */
  
  union block_hdr *blok = a->last; 
  char *first_avail = blok->h.first_avail;
  char *new_first_avail;

  if(reqsize <= 0)
      return NULL;

  new_first_avail = first_avail + size;
  
  if (new_first_avail <= blok->h.endp) {
    blok->h.first_avail = new_first_avail;
    return (void *)first_avail;
  }

  /* Nope --- get a new one that's guaranteed to be big enough */
  
  csa_acquire_mutex(alloc_mutex);

  blok = new_block (size);
  a->last->h.next = blok;
  a->last = blok;

  first_avail = blok->h.first_avail;
  blok->h.first_avail += size;

  csa_release_mutex(alloc_mutex);

  return (void *)first_avail;
}

void *
ap_pcalloc(a, size)
  struct pool *a;
  int size;
{
  void *res = ap_palloc (a, size);
  /* LINTED */ /* 3rd arg is size_t on NetBSD */
  memset(res, 0, size);
  return res;
}

char *
ap_pstrdup(a, s)
  struct pool *a;
  const char *s;
{
  char *res;
  if (s == NULL) return NULL;
  res = ap_palloc (a, (int) strlen(s) + 1);
  strcpy (res, s);
  return res;
}

char *
ap_pstrndup(a, s, n)
  struct pool *a;
  const char *s;
  int n;
{
  char *res;
  if (s == NULL) return NULL;
  res = ap_palloc (a, n + 1);
  /* LINTED */ /* 3rd arg is size_t on NetBSD */
  strncpy (res, s, n);
  res[n] = '\0';
  return res;
}


/*****************************************************************
 *
 * Managing generic cleanups.  
 */

struct cleanup {
  void *data;
  void (*plain_cleanup) __P((void *));
  void (*child_cleanup) __P((void *));
  struct cleanup *next;
};

void 
ap_register_cleanup (p, data, plain_cleanup, child_cleanup)
  struct pool *p;
  void *data;
  void (*plain_cleanup) __P((void *));
  void (*child_cleanup) __P((void *));
{
  struct cleanup *c = (struct cleanup *)ap_palloc(p, sizeof (struct cleanup));
  c->data = data;
  c->plain_cleanup = plain_cleanup;
  c->child_cleanup = child_cleanup;
  c->next = p->cleanups;
  p->cleanups = c;
}

static void 
run_cleanups (c)
  struct cleanup *c;
{
  while (c) {
    (*c->plain_cleanup)(c->data);
    c = c->next;
  }
}

/* Note that we have separate plain_ and child_ cleanups for FILE *s,
 * since fclose() would flush I/O buffers, which is extremely undesirable;
 * we just close the descriptor.
 */

static void file_cleanup (fpv)
  void *fpv;
{ fclose ((FILE *)fpv); }

static void file_child_cleanup (fpv)
  void *fpv;
{ close (fileno ((FILE *)fpv)); }

void ap_note_cleanups_for_file (p, fp)
  struct pool *p;
  FILE *fp;
{
  ap_register_cleanup (p, (void *)fp,
#ifdef __MSWIN__
	/* need to cast the function pointers, MS VC++ 5.0 doesn't like
	 * it, for whatever reason */
	(void *)file_cleanup, (void *)file_child_cleanup
#else
	file_cleanup, file_child_cleanup
#endif
	);
}

#endif /* CSACEK */
