/*
  mod_tsunami 2.0

  mod_tsunami is an apache module which dynamically limits the number
  of httpd slots used per vhosts. It is useful if you provide home
  pages hosting, and don't want one site to use all the ressources.

  Copyright (C) 2001-2003 Bertrand Demiddelaer / Libertysurf Telecom

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "httpd.h"
#include "http_config.h"
#include "http_conf_globals.h"
#include "scoreboard.h"

#define TSUNAMI_VERSION "2.0"

module MODULE_VAR_EXPORT tsunami_module;

typedef struct tsunami_conf {
  int active;
  int active_set;
  int trigger;
  int trigger_set;
  int max_connections;
  int max_connections_set;
} tsunami_conf;

static void init_tsunami(server_rec *s, pool *p) {
  ap_add_version_component("mod_tsunami/" TSUNAMI_VERSION);
}

static void *create_tsunami_server_config(pool *p, server_rec *dummy) {
  tsunami_conf *newcfg=(tsunami_conf *)ap_pcalloc(p, sizeof(tsunami_conf));
  newcfg->active=0;
  newcfg->active_set=0;
  newcfg->trigger=0;
  newcfg->trigger_set=0;
  newcfg->max_connections=0;
  newcfg->max_connections_set=0;
  return (void *)newcfg;
}

static void *merge_tsunami_server_config(pool *p, void *parent, void *child) {
  tsunami_conf *parent_cfg;
  tsunami_conf *child_cfg;
  tsunami_conf *conf;

  parent_cfg=(tsunami_conf *)parent;
  child_cfg=(tsunami_conf *)child;
  conf = (tsunami_conf *)ap_pcalloc(p, sizeof(tsunami_conf));

  conf->active=child_cfg->active_set?child_cfg->active:parent_cfg->active;
  conf->trigger=child_cfg->trigger_set?child_cfg->trigger:parent_cfg->trigger;
  conf->max_connections=child_cfg->max_connections_set?child_cfg->max_connections:parent_cfg->max_connections;

  return conf;
}

static const char *set_tsunami_status(cmd_parms *cmd, void *dummy, int arg) {
  tsunami_conf *conf=(tsunami_conf *)ap_get_module_config (cmd->server->module_config, &tsunami_module);

  conf->active_set=1;
  if (arg!=0) {
    if (ap_extended_status==0) {
      return "Tsunami can only be activated if ExtendedStatus is 'On'";
    }
    conf->active=1;
    return NULL;
  } else {
    conf->active=0;
    return NULL;
  }
}

static const char *set_tsunami_trigger(cmd_parms *cmd, void *dummy, char *arg) {
  tsunami_conf *conf=(tsunami_conf *)ap_get_module_config (cmd->server->module_config, &tsunami_module);

  conf->trigger_set=1;
  conf->trigger=atoi(arg);
  if (conf->trigger<0) {
    return "TsunamiTrigger argument must be greater or equal to 0";
  }
  return NULL;
}

static const char *set_tsunami_max(cmd_parms *cmd, void *dummy, char *arg) {
  tsunami_conf *conf=(tsunami_conf *)ap_get_module_config (cmd->server->module_config, &tsunami_module);

  conf->max_connections_set=1;
  conf->max_connections=atoi(arg);
  if (conf->max_connections<0) {
    return "TsunamiMaxConnections argument must be greater or equal to 0";
  }
  return NULL;
}

static const command_rec tsunami_cmds[] = {
  {"TsunamiActive", set_tsunami_status, NULL, RSRC_CONF, FLAG,
  "activate or desactivate tsunami protection"},
  {"TsunamiTrigger", set_tsunami_trigger, NULL, RSRC_CONF, TAKE1,
   "number of total simultaneous requests to start tsunami protection"},
  {"TsunamiMaxConnections", set_tsunami_max, NULL, RSRC_CONF, TAKE1,
   "number of maximum simultaneous requests allowed per vhost"},
  {NULL}
};

static int tsunami_filter(request_rec *r) {
  tsunami_conf *conf=(tsunami_conf *)ap_get_module_config(r->server->module_config, &tsunami_module);

  int i;
  unsigned char status;
  int total_count;
  int server_count;

  /*
    if tsunami is not active, or if there is no scoreboard, don't process
  */
  if ((conf->active>0)&&
      (conf->max_connections>0)&&
      (ap_exists_scoreboard_image())) {

    /*
      count the number of requests active for this account
    */
    ap_sync_scoreboard_image();
    for (i=0, total_count=0, server_count=0;
	 i<ap_daemons_limit;
	 i++) {
      
      status=ap_scoreboard_image->servers[i].status;
      if (status == SERVER_BUSY_READ ||
	  status == SERVER_BUSY_WRITE ||
	  status == SERVER_BUSY_KEEPALIVE ||
	  status == SERVER_BUSY_DNS) {

	/* update counters */
	total_count++;
	if (r->server == ap_scoreboard_image->servers[i].vhostrec) {
	  server_count++;

	  /* tsunami ? */
	  if (total_count > conf->trigger &&
	      server_count > conf->max_connections)
	    return HTTP_SERVICE_UNAVAILABLE;
	}
      }
    }
  }

  /*
    no tsunami detected
  */
  return DECLINED;
}

module MODULE_VAR_EXPORT tsunami_module =
{
    STANDARD_MODULE_STUFF,
    init_tsunami,                 /* initializer */
    NULL,                         /* dir config creater */
    NULL,                         /* merge dir configs */
    create_tsunami_server_config, /* server config */
    merge_tsunami_server_config,  /* merge server configs */
    tsunami_cmds,                 /* command table */
    NULL,                         /* handlers */
    NULL,                         /* filename translation */
    NULL,                         /* check_user_id */
    NULL,                         /* check auth */
    NULL,                         /* check access */
    NULL,                         /* type_checker */
    tsunami_filter,               /* fixups */
    NULL,                         /* logger */
    NULL,                         /* header parser */
    NULL,                         /* child_init */
    NULL,                         /* child_exit */
    NULL                          /* post read-request */
};
