/**
 *
 * mod_ifier
 *  Simple filtering module for Apache 2.x which allows you to deny
 * incoming connections based upon semi-arbitary criterea.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 * Furthermore, Steve Kemp gives permission to link mod_ifier
 * with OpenSSL and Apache Licensed works as well as distributing the 
 * linked executables. 
 *
 *  Steve Kemp
 *  ---
 *  http://www.steve.org.uk/
 *
 */


/* Apache Headers */
#include "ap_compat.h"
#include "apr.h"
#include "apr_compat.h"
#include "apr_file_io.h"
#include "apr_hooks.h"
#include "apr_lib.h"
#include "apr_strings.h"
#include "apr_tables.h"
#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_protocol.h"
#include "http_request.h"
#include "http_log.h"

/* Our module header */
#include "mod_ifier.h"

/* Our match code. */
#include "match.c"


/*
 * Links to use in the status handler.
 */
#define HOMEPAGE  "http://www.steve.org.uk/Software/mod_ifier"
#define RULE_BASE "http://www.steve.org.uk/Software/mod_ifier/rules.html"


/**
 * Debugging routine which writes printf-type formatted strings
 * to a file under /tmp.
 *
 * Only enabled in Debug builds.
 *
 */
void debug_printf( const char *fmt, ... )
{
#ifdef _DEBUG
    FILE *log;
    char log_line[1024];
    
    va_list ap;

    /* Open logfile, abort if that fails. */
    log = fopen( "/tmp/modifier.log","a" );
    if ( !log ) 
        return; 

    /* Format line. */
    va_start( ap, fmt );
    memset(log_line, '\0', sizeof(log_line) );
    vsnprintf(log_line, sizeof(log_line) -1, fmt, ap);
    
    /* Write it out */
    fprintf( log, "%s\n", log_line );
    fclose( log );

    /* Finished */
    va_end(ap);
#endif
}








/**
 * A simple table of the configuration directives this module accepts.
 */
static const command_rec mod_ifier_cmds[] =
{
    AP_INIT_TAKE1("DropAction",
                  add_drop_action,  NULL,  RSRC_CONF,
                  "the action to take on a successful match"),
    AP_INIT_TAKE12("DropAgent",
                  add_drop_agent,  NULL,  RSRC_CONF,
                  "a user agent to drop"),
    // TODO: RULE??!??
    AP_INIT_TAKE1("DropBlacklist",
                  add_drop_blacklist, NULL, RSRC_CONF,
                  "an IP address/range to blacklist"),
    AP_INIT_TAKE23("DropHeader",
                  add_drop_header,  NULL,  RSRC_CONF,
                  "drop a match against an arbitary header"),
    AP_INIT_TAKE12("DropMethod",
                  add_drop_method,  NULL,  RSRC_CONF,
                  "drop a match against an incoming request method"),
    AP_INIT_TAKE1("DropLog",
                  add_drop_log,  NULL,  RSRC_CONF,
                  "a logfile to write drops to"),
    AP_INIT_TAKE1("DropLogHeaders",
                  add_drop_log_headers,  NULL,  RSRC_CONF,
                  "a logfile to write headers to"),
    // TODO: RULE?!?
    AP_INIT_TAKE1("DropParam",
                  add_drop_param,  NULL,  RSRC_CONF,
                  "a CGI parameter to drop"),
    AP_INIT_TAKE23("DropParamValue",
                  add_drop_param_value,  NULL,  RSRC_CONF,
                  "a CGI parameter and value to drop"),
    AP_INIT_TAKE12("DropParamValues",
                  add_drop_param_values,  NULL,  RSRC_CONF,
                  "a CGI parameter value to drop"),
    AP_INIT_TAKE12("DropPath",
                  add_drop_path,  NULL,  RSRC_CONF,
                  "a request pattern to drop"),
    AP_INIT_TAKE12("DropReferer",
                  add_drop_referer,  NULL,  RSRC_CONF,
                  "a referer to drop"),
    // TODO: RULE??!??
    AP_INIT_TAKE1("DropWhitelist",
                  add_drop_whitelist, NULL, RSRC_CONF,
                  "an IP address/range to whitelist"),
    { NULL }
};



/**
 *  Tell Apache which functions this module will provide.
 */
module AP_MODULE_DECLARE_DATA mod_ifier_module =
{
    STANDARD20_MODULE_STUFF,
    NULL,
    NULL,
    mod_ifier_create_server_config, /* server config */
    NULL,
    mod_ifier_cmds,                 /* configuration commands. */
    mod_ifier_register_hooks,       /* callback for registering hooks */
};



/**
 * Return the current date & time, suitable for writing to our logfile.
 */
char *current_time(request_rec *r) 
{
    apr_time_exp_t t;
    char tstr[100];
    apr_size_t len;
  
    apr_time_exp_lt(&t, apr_time_now());
    memset(tstr, '\0', sizeof(tstr) );
    apr_strftime(tstr, &len, sizeof(tstr)-1, "%d/%b/%Y %H:%M:%S", &t);
    return apr_pstrdup(r->pool, tstr);
}


/**
 *  Write a message to our logfile.
 */
static void modifier_write_log( request_rec *r, const char *text, ...)
{
    char str1[1024] = "";
    char str2[1256] = "";
    int rc;
    apr_file_t  *fd;
    apr_size_t nbytes, nbytes_written;
    va_list ap;
    
    mod_ifier_config *conf =
        (mod_ifier_config *) ap_get_module_config(r->server->module_config, &mod_ifier_module);

    /* If we don't have a logfile defined then we return. */
    const char *filename = conf->logfile;
    if ( filename == NULL )
        return;


    rc = apr_file_open(&fd, filename,
                       APR_WRITE | APR_APPEND | APR_CREATE | APR_BINARY,
                       ( APR_UREAD | APR_UWRITE | APR_GREAD ),
                       r->pool);
    if (rc != APR_SUCCESS) 
        return;

    va_start(ap, text);
    apr_vsnprintf(str1, sizeof(str1), text, ap);
    apr_snprintf(str2, sizeof(str2), "%s %s %s - %s", 
                 current_time(r),
                 ap_get_server_name(r),
                 r->connection->remote_ip, 
                 str1);
    
    nbytes = strlen(str2);
    apr_file_write_full(fd, str2, nbytes, &nbytes_written);  
    apr_file_close(fd);
    va_end(ap);
}


/**
 *  Write headers to the logfile.
 * 
 * NOTE:
 *
 *  Although this function is documented it isn't really intended for
 * end-users, instead it is useful for Steve.
 *
 */
static void log_headers( request_rec *req, const char *path )
{
    apr_file_t  *fd;
    apr_size_t nbytes_written;

    mod_ifier_config *conf =
        (mod_ifier_config *) ap_get_module_config(req->server->module_config, &mod_ifier_module);

    /* If we don't have a logfile then use the global one */
    if ( path == NULL )
        path = conf->logfile_headers;


    /* If the global path wasn't set then return */
    if ( path == NULL )
        return;

    apr_status_t rc = apr_file_open(&fd, path,
                       APR_WRITE | APR_APPEND | APR_CREATE | APR_BINARY,
                       ( APR_UREAD | APR_UWRITE | APR_GREAD ),
                       req->pool);
    if (rc != APR_SUCCESS) 
        return;

    
    /* The IP */
    apr_file_write_full(fd, req->connection->remote_ip, strlen( req->connection->remote_ip ), &nbytes_written);  
    apr_file_write_full(fd, "\n", 1, &nbytes_written );

    /* The request */
    apr_file_write_full(fd, req->the_request, strlen( req->the_request ), &nbytes_written);  
    apr_file_write_full(fd, "\n", 1, &nbytes_written );

    /* The headers */
    {
        const apr_array_header_t *arr;
        apr_table_entry_t *te;
        int i;

        arr = apr_table_elts(req->headers_in);
        te = (apr_table_entry_t *)arr->elts;
        for (i = 0; i < arr->nelts; i++) 
        {
            char buffer[2048];
            memset(buffer, '\0', sizeof(buffer) );
            snprintf( buffer,sizeof(buffer),"%s: %s\n", te[i].key, te[i].val );
            apr_file_write_full(fd, buffer, strlen(buffer), &nbytes_written );
        }
    }
    apr_file_write_full(fd, "\n\n", 2, &nbytes_written );


    /* Any CGI parameters. */
    {
        /* TODO */
    }

    apr_file_close(fd);
}

/*
 * Fill out an action record according the to the given string.
 * 
 * Valid argument strings are:
 *
 *    execute=/x/x - Execute a process.
 *    logfile=/x/y - Write to a logfile.
 *    redirect=xxx - Redirect matching clients to the given URL.
 *    status=NNN   - Return the HTTP status code NNN to matching clients.
 *
 * Multiple values are allowed, seperate them with ","s.
 *
 */
static void setup_action_record( cmd_parms *cmd, mod_ifier_action *act, const char *action )
{
    char *start;

    /* By default we do nothing */
    act->uri      = NULL;
    act->response = 0;
    act->exec     = NULL;

    /* Execute a command */
    if ( ( start = strstr( action, "execute=" ) ) != NULL )
    {
        int len             = 0;
        const char *command = start + strlen( "execute=" );

        while( ( command[len] != '\0' ) &&
               ( command[len] != ',' ) )
            len++;

        act->exec = apr_pcalloc(cmd->pool, len + 1 );
        strncpy( (char *)act->exec, command, len );
    }

    /* Per-action logfile. */
    if ( ( start = strstr( action, "logfile=" ) ) != NULL )
    {
        int len          = 0;
        const char *path = start + strlen( "logfile=" );

        while( ( path[len] != '\0' ) &&
               ( path[len] != ',' ) )
            len++;

        act->path = apr_pcalloc(cmd->pool, len + 1 );
        strncpy( (char *)act->path, path, len );
    }

    /* Look for a redirect */
    if ( ( start = strstr( action, "redirect=" ) )!= NULL )
    {
        int len         = 0;
        const char *uri = start + strlen( "redirect=" );

        while( ( uri[len] != '\0' ) &&
               ( uri[len] != ',' ) )
            len++;

        act->uri = apr_pcalloc(cmd->pool, len + 1 );
        strncpy( (char *)act->uri, uri, len );
    }    

    /* Look for a HTTP status code */
    if ( ( start = strstr( action, "status=" ) ) != NULL )
    {
        const char *code = start + strlen( "status=" );
        act->response    = atoi( code );
    }
}


/**
 * Process a "DropAction" request in the Apache configuration file.
 */
static const char * add_drop_action(cmd_parms *cmd, void *dummy, const char *action )
{
    server_rec *s = cmd->server;

    /* Server configuration */
    mod_ifier_config *conf =
    (mod_ifier_config *) ap_get_module_config(s->module_config, &mod_ifier_module);

    /* Action object. */
    mod_ifier_action *act = apr_pcalloc(cmd->pool, sizeof(mod_ifier_action));

    /* Fill it out */
    setup_action_record( cmd, act, action );

    conf->defaultAction = act;

    return NULL;
}


/**
 * Process a "DropAgent" request in the Apache configuration file.
 */
static const char * add_drop_agent(cmd_parms *cmd, void *dummy, const char *term, const char *action )
{
    return( filter_header( cmd, "User-Agent", term, action ) );
}


/**
 * Add a new blacklist/whitelist entry to our list of rules.
 *
 *  Valid arguments are:
 *
 *    xx.xx.xx.xx:  The single IP address.
 *    xx.xx.xx/xx:  A CIDR range.
 */
static const char * filter_ip(cmd_parms *cmd, const char *ip, mod_ifier_rule_type type )
{
    server_rec *srv = cmd->server;
    char *s;

    mod_ifier_config *conf =
    (mod_ifier_config *) ap_get_module_config(srv->module_config, &mod_ifier_module);

    /* Add a new entry to the global rule list */
    mod_ifier_rule *new;
    new = (mod_ifier_rule *)apr_array_push(conf->rules);

    /* This rule will describe an IP blacklist/whitelist whichever was given */
    new->rule_type = type;
    new->val_text  = apr_pstrdup(cmd->pool, ip);
    
    /* Range vs. IP */
    if ((s = strchr(ip, '/'))) 
    {
        *s++ = '\0';
        apr_ipsubnet_create(&new->ip, ip, s, cmd->pool);
    }
    else
    {
        apr_ipsubnet_create(&new->ip, ip, NULL, cmd->pool);
    }
    return NULL;

}

/**
 * Process a "DropBlacklist" request in the Apache configuration file.
 */
static const char * add_drop_blacklist(cmd_parms *cmd, void *dummy, const char *ip )
{
    return( filter_ip( cmd, ip, BLACKLIST ) );
}


/**
 * Process a "DropHeader" request in the Apache configuration file.
 */
static const char * add_drop_header(cmd_parms *cmd, void *dummy, const char *header, const char *term, const char *action )
{
    return( filter_header( cmd, header, term, action ) );
}


/**
 * Process a "DropMethod" request in the Apache configuration file.
 */
static const char * add_drop_method(cmd_parms *cmd, void *dummy, const char *type, const char *action )
{
    server_rec *s = cmd->server;
    mod_ifier_config *conf =
        (mod_ifier_config *) ap_get_module_config(s->module_config, &mod_ifier_module);

    /* Add a new entry to the global matching list */
    mod_ifier_rule *new;
    new = (mod_ifier_rule *)apr_array_push(conf->rules);

    /* This rule describes a path match. */
    new->rule_type = METHOD_MATCH;

    /* If we have an action set it up */
    if ( action != NULL )
    {
        /* Action object. */
        mod_ifier_action *act = apr_pcalloc(cmd->pool, sizeof(mod_ifier_action));
        /* Fill it out */
        setup_action_record( cmd, act, action );

        /* Store it */
        new->action = act;

    }

    /* Store the path. */
    new->val_text = apr_pstrdup(cmd->pool, type);
    new->value    = ap_pregcomp(cmd->pool, type, AP_REG_EXTENDED|AP_REG_NOSUB);     
    return NULL;
}


/**
 * Process a "DropLog" request in the Apache configuration file.
 */
static const char * add_drop_log(cmd_parms *cmd, void *dummy, const char *file )
{
    server_rec *s = cmd->server;

    mod_ifier_config *conf =
    (mod_ifier_config *) ap_get_module_config(s->module_config, &mod_ifier_module);
    conf->logfile = ap_server_root_relative(cmd->pool, file);
    return NULL;
}


/**
 * Process a "DropLogHeaders" request in the Apache configuration file.
 */
static const char * add_drop_log_headers(cmd_parms *cmd, void *dummy, const char *file )
{
    server_rec *s = cmd->server;

    mod_ifier_config *conf =
    (mod_ifier_config *) ap_get_module_config(s->module_config, &mod_ifier_module);
    conf->logfile_headers = ap_server_root_relative(cmd->pool, file);
    return NULL;
}


/**
 * Process a "DropParam" request in the Apache configuration file.
 */
static const char * add_drop_param(cmd_parms *cmd, void *dummy, const char *parameter )
{
    filter_parameter( cmd, parameter, NULL , NULL);
    return NULL;
}


/**
 * Process a "DropParamValue" request in the Apache configuration file.
 */
static const char * add_drop_param_value(cmd_parms *cmd, void *dummy, const char *parameter, const char *value, const char *action )
{
    filter_parameter( cmd, parameter, value, action );
    return NULL;
}


/**
 * Process a "DropParamValues" request in the Apache configuration file.
 */
static const char * add_drop_param_values(cmd_parms *cmd, void *dummy, const char *value, const char *action )
{
    filter_parameter( cmd, NULL, value, action );
    return NULL;
}


/**
 * Process a "DropPath" request in the Apache configuration file.
 */
static const char * add_drop_path(cmd_parms *cmd, void *dummy, const char *path , const char *action)
{
    server_rec *s = cmd->server;
    mod_ifier_config *conf =
        (mod_ifier_config *) ap_get_module_config(s->module_config, &mod_ifier_module);

    /* Add a new entry to the global matching list */
    mod_ifier_rule *new;
    new = (mod_ifier_rule *)apr_array_push(conf->rules);

    /* This rule describes a path match. */
    new->rule_type = PATH_MATCH;

    /* If we have an action set it up */
    if ( action != NULL )
    {
        /* Action object. */
        mod_ifier_action *act = apr_pcalloc(cmd->pool, sizeof(mod_ifier_action));
        /* Fill it out */
        setup_action_record( cmd, act, action );

        /* Store it */
        new->action = act;

    }

    /* Store the path. */
    new->name     = NULL;
    new->val_text = apr_pstrdup(cmd->pool,path);
    new->value    = ap_pregcomp(cmd->pool,path, AP_REG_EXTENDED|AP_REG_NOSUB); 
    
    return NULL;
}


/**
 * Process a "DropReferer" request in the Apache configuration file.
 */
static const char * add_drop_referer(cmd_parms *cmd, void *dummy, const char *term, const char *action )
{
    return( filter_header( cmd, "Referer", term, action ) );
}


/**
 * Process a "DropWhitelist" request in the Apache configuration file.
 */
static const char * add_drop_whitelist(cmd_parms *cmd, void *dummy, const char *ip )
{
    return( filter_ip( cmd, ip, WHITELIST ) );
}


/**
 * Iterate over all our rules and see if we can match the client IP address
 * against any whitelist or a blacklist entries we might have.
 */
static int mod_ifier_match_ip( request_rec *req, mod_ifier_rule_type type )
{
    mod_ifier_config *conf =
        (mod_ifier_config *) ap_get_module_config(req->server->module_config, &mod_ifier_module);

    /**
     * Iterate over our rules and return 1 if the client address is
     * matched against the relevent blacklist/whitelist value.
     */
    if ( conf->rules->nelts )
    {
        int i;
        mod_ifier_rule *rules;
        rules = (mod_ifier_rule *) conf->rules->elts;

        for( i = 0; i < conf->rules->nelts; i++ )
        {
            /* Something we're matching against. */
            mod_ifier_rule *r  = &rules[i];

            if ( r->rule_type == type )
            {
                if ( apr_ipsubnet_test( r->ip, req->connection->remote_addr ) )
                    return 1;
            }
        }
    }

    return 0;
}


/**
 * Is the connection coming from an IP address we have whitelisted?
 */
static int mod_ifier_is_ip_blacklisted( request_rec *req )
{
    return( mod_ifier_match_ip( req, BLACKLIST ) );
}


/**
 * Is the connection coming from an IP address we have whitelisted?
 */
static int mod_ifier_is_ip_whitelisted( request_rec *req )
{
    return( mod_ifier_match_ip( req, WHITELIST ) );
}


/**
 * Add a new header to filter against, here we compile the regular expression
 * we're given immediately as a speed optimization.
 */
static const char * filter_header(cmd_parms *cmd, const char *header, const char *pattern, const char *action )
{
   server_rec *s = cmd->server;
    mod_ifier_config *conf =
        (mod_ifier_config *) ap_get_module_config(s->module_config, &mod_ifier_module);

    /* Add a new entry to the global matching list */
    mod_ifier_rule *new;
    new = (mod_ifier_rule *)apr_array_push(conf->rules);

    /* This rule describes a header value. */
    new->rule_type = HEADER_MATCH;

    /* If we have an action set it up */
    if ( action != NULL )
    {
        /* Action object. */
        mod_ifier_action *act = apr_pcalloc(cmd->pool, sizeof(mod_ifier_action));
        /* Fill it out */
        setup_action_record( cmd, act, action );

        /* Store it */
        new->action = act;

    }

    /* Compile the regular expression */
    new->name     = apr_pstrdup(cmd->pool,header);
    new->val_text = apr_pstrdup(cmd->pool,pattern);
    new->value    = ap_pregcomp(cmd->pool,pattern, AP_REG_EXTENDED|AP_REG_NOSUB); 
    
    return NULL;
}



/**
 * Add a new CGI parameter to filter against, here we compile the regular 
 * expression we're given immediately as a speed optimization.
 */
static const char * filter_parameter(cmd_parms *cmd, const char *parameter, const char *pattern, const char *action )
{
    server_rec *s = cmd->server;

    mod_ifier_config *conf =
    (mod_ifier_config *) ap_get_module_config(s->module_config, &mod_ifier_module);

    /* Add a new entry to the global matching list */
    mod_ifier_rule *new;
    new = (mod_ifier_rule *)apr_array_push(conf->rules);

    /* This rule describes a CGI parameter match. */
    new->rule_type = CGI_MATCH;

    /* If we have an action set it up */
    if ( action != NULL )
    {
        /* Action object. */
        mod_ifier_action *act = apr_pcalloc(cmd->pool, sizeof(mod_ifier_action));
        /* Fill it out */
        setup_action_record( cmd, act, action );

        /* Store it */
        new->action = act;

    }

    /* Add the parameter */
    if ( parameter ) 
        new->name = apr_pstrdup(cmd->pool,parameter);
    else
        new->name = NULL;

    /* And the optional value. */
    if ( pattern == NULL )
    {
        new->value     = NULL;
        new->val_text = NULL;
    }
    else
    {
        new->value     = ap_pregcomp(cmd->pool,pattern, AP_REG_EXTENDED|AP_REG_NOSUB); 
        new->val_text = apr_pstrdup(cmd->pool,pattern);
    }
    return NULL;
}


/**
 * Create a server configuration - ie. read all the options in the
 * configuration file, and store them as entries in our configuration.
 */
static void *mod_ifier_create_server_config(apr_pool_t *p, server_rec *dummy)
{
    /* Server configuration */
    mod_ifier_config *cfg = apr_pcalloc(p, sizeof(mod_ifier_config));

    /* Default action to take. */
    mod_ifier_action *act = apr_pcalloc(p, sizeof(mod_ifier_action));

    /* Create a new array of rules for this server. */
    cfg->rules = apr_array_make(p, 1, sizeof(mod_ifier_rule));

    /* The default action to take, deny with 403. */
    act->action   = HTTP_RESPONSE;
    act->response = 403;
    act->uri      = NULL;
    act->exec     = NULL;

    /* Save this default action away. */
    cfg->defaultAction = act;

    /* Logfile */
    cfg->logfile         = NULL;
    cfg->logfile_headers = NULL;

    return cfg;
}




/**
 * Iterate over all loaded rules and display them to the client.
 */
static int mod_ifier_handler(request_rec *req)
{
    mod_ifier_config *conf;
    conf = ap_get_module_config(req->server->module_config, &mod_ifier_module);

    /* Buffer for per-rule actions. */
    char buffer[1024];

    /* Make sure this request was for us */
    if ( req->handler == NULL )
        return DECLINED;
    if ( ( strcmp( req->handler, "mod_ifier-handler" ) != 0 ) &&
         ( strcmp( req->handler, "mod-ifier-handler" ) != 0 ) )
        return DECLINED;
    /**
     * Set the content type and return a header.
     */
    req->content_type = "text/html";
    ap_send_http_header(req);
    
    ap_rputs("<html><head><title>mod_ifier - Loaded rules</title></head><body>\n" , req);
    ap_rputs( "<body><center><h1>mod_ifier - Loaded rules</h1></center>\n" , req);
    ap_rputs( "<h2>Default Options</h2>", req );
    ap_rputs( "<p>These are the default server-wide options:</p>\n", req );
    ap_rputs( "<blockquote>\n", req );
    if ( conf->defaultAction )
    {
        ap_rputs( "<p><b>DropAction</b>: ", req );
        if ( conf->defaultAction->response )
        {
            memset(buffer, '\0', sizeof(buffer));
            apr_snprintf(buffer, sizeof(buffer)-1, "HTTP Code %d", conf->defaultAction->response );
        
            ap_rputs( buffer, req );
        }
        if ( conf->defaultAction->uri )
        {
            memset(buffer, '\0', sizeof(buffer));
            apr_snprintf(buffer, sizeof(buffer)-1, "Redirect <a href=\"%s\">%s</a>", conf->defaultAction->uri, conf->defaultAction->uri );
            
            ap_rputs( buffer, req );
        }
        if ( conf->defaultAction->exec )
        {
            memset(buffer, '\0', sizeof(buffer));
            apr_snprintf(buffer, sizeof(buffer)-1, "Execute %s", conf->defaultAction->exec );
            
            ap_rputs( buffer, req );
        }
        ap_rputs( "</p>", req );
    }

    if ( conf->logfile )
    {
        ap_rputs( "<p><b>DropLog</b>: ", req );
        ap_rputs( conf->logfile, req );
        ap_rputs( "</p>\n", req );
    }

    if ( conf->logfile_headers )
    {
        ap_rputs( "<p><b>DropLogHeaders</b>: ", req );
        ap_rputs( conf->logfile_headers, req );
        ap_rputs( "</p>\n", req );
    }
    ap_rputs( "</blockquote>\n", req );

    ap_rputs( "<h2>Loaded Rules</h2>", req );
    ap_rputs( "<p>These are the rules we have loaded.</p>\n", req );

    ap_rputs( "<blockquote><table>\n" , req);
    ap_rputs( "<tr><th><b>Drop Type</b></th><th><b>Parameters</b></th><th><b>Action</b></tr>\n" , req);
    
    /**
     * Iterate over our rules and return 1 if the client address is
     * matched against the relevent blacklist/whitelist value.
     */
    if ( conf->rules->nelts )
    {
        mod_ifier_rule *rules = (mod_ifier_rule *) conf->rules->elts;
        int i;

        /**
         * Loop over each rule.
         *
         */
        for( i = 0 ; i < conf->rules->nelts ; i++ )
        {
            /* The rule we're going to match against. */
            mod_ifier_rule *rule  = &rules[i];

            switch( rule->rule_type )
            {
            case WHITELIST:
                ap_rputs( "<tr><td><a href=\"" RULE_BASE "#DropWhitelist\">DropWhiteList</a></td><td>&quot;<tt>", req );
                ap_rputs( rule->val_text, req );
                ap_rputs( "</tt>&quot;</td>\n", req );
                break;
                
            case BLACKLIST:
                ap_rputs( "<tr><td><a href=\"" RULE_BASE "#DropBlacklist\">DropBlacklist</a></td><td>&quot;<tt>", req );
                ap_rputs( rule->val_text, req );
                ap_rputs( "</tt>&quot;</td>\n", req );
                break;

            case CGI_MATCH:
                if ( ( rule->name != NULL ) && ( rule->val_text != NULL ) )
                {
                    ap_rputs( "<tr><td><a href=\"" RULE_BASE "#DropParamValue\">DropParamValue</a></td><td>&quot;<tt>", req );
                    ap_rputs( rule->name, req );
                    ap_rputs( "</tt>&quot; &quot;<tt>", req );
                    ap_rputs( rule->val_text, req );
                    ap_rputs( "</tt>&quot;</td>", req );
                }
                else
                {
                    if ( rule->name == NULL )
                    {
                        ap_rputs( "<tr><td><a href=\"" RULE_BASE "#DropParamValues\">DropParamValues</a></td><td>&quot;<tt>", req );
                        ap_rputs( rule->val_text, req );
                        ap_rputs( "</tt>&quot;</td>", req );
                    }
                    else
                    {
                        ap_rputs( "<tr><td><a href=\"" RULE_BASE "#DropParam\">DropParam</a></td><td>&quot;<tt>", req );
                        ap_rputs( rule->name, req );
                        ap_rputs( "</tt>&quot;</td>", req );
                    }
                }
                break;

            case METHOD_MATCH:
                ap_rputs( "<tr><td><a href=\"" RULE_BASE "#DropMethod\">DropMethod</a></td><td>&quot;<tt>", req );
                ap_rputs( rule->val_text, req );
                ap_rputs( "</tt>&quot;</td>\n", req );
                break;

            case HEADER_MATCH:
                ap_rputs( "<tr><td><a href=\"" RULE_BASE "#DropHeader\">DropHeader</a></td><td>&quot;<tt>", req );
                ap_rputs( rule->name, req );
                ap_rputs( "</tt>&quot; &quot;<tt>", req );
                ap_rputs( rule->val_text, req );
                ap_rputs( "</tt>&quot;</td>\n", req );
                break;

            case PATH_MATCH:
                ap_rputs( "<tr><td><a href=\"" RULE_BASE "#DropPath\">DropPath</a></td><td>&quot;<tt>", req );
                ap_rputs( rule->val_text, req );
                ap_rputs( "</tt>&quot;</td>\n", req );
                break;

            default:
                ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, 
                             req->server, "Unknown rule type - %d", rule->rule_type );
                break;
            }

            /*
             * Add in the per-rule action if present.
             */
            if ( rule->action != NULL )
            {
                ap_rputs( "<td>\n", req );

                if ( rule->action->exec )
                {
                    memset(buffer, '\0', sizeof(buffer));
                    apr_snprintf(buffer, sizeof(buffer)-1, "Execute %s", rule->action->exec );

                    ap_rputs( buffer, req );
                }
                if ( rule->action->path )
                {
                    memset(buffer, '\0', sizeof(buffer));
                    apr_snprintf(buffer, sizeof(buffer)-1, "Logfile %s", rule->action->path );

                    ap_rputs( buffer, req );
                }
                if ( rule->action->response )
                {
                    memset(buffer, '\0', sizeof(buffer));
                    apr_snprintf(buffer, sizeof(buffer)-1, "HTTP Code %d", rule->action->response );

                    ap_rputs( buffer, req );
                }
                if ( rule->action->uri )
                {
                    memset(buffer, '\0', sizeof(buffer));
                    apr_snprintf(buffer, sizeof(buffer)-1, "Redirect <a href=\"%s\">%s</a>", rule->action->uri,rule->action->uri );

                    ap_rputs( buffer, req );
                }
                ap_rputs( "</td></tr>\n", req );

            }
            else
            {
                ap_rputs( "<td>Default.</td></tr>\n", req );
            }
        }
    }

    ap_rputs( "</table></blockquote>\n" , req);
    
    memset(buffer, '\0', sizeof(buffer));
    apr_snprintf(buffer, sizeof(buffer)-1, "<p>%d rules loaded.</p>", conf->rules->nelts );
    ap_rputs( buffer, req );

    ap_rputs( "<hr />\n<p style=\"text-align: right;\"><a href=\"" HOMEPAGE "\">mod_ifier v" VERSION "</a> by <a href=\"http://www.steve.org.uk\">Steve Kemp</a>.</p></body></html>\n" , req);
    

    return OK;
}


/**
 * Iterate over every rule we currently have defined and test it against
 * the request we've received.
 *
 */
static int mod_ifier_test_incoming_request(request_rec *req) 
{  
    mod_ifier_config *conf;
    conf = ap_get_module_config(req->server->module_config, &mod_ifier_module);

    /* Holder for the CGI arguments. */
    apr_array_header_t *cgi = NULL;

    /*
     * Is the IP address whitelisted/blacklisted? 
     *
     * This iterates over all our loaded rules once.
     *
     */
    if ( mod_ifier_is_ip_whitelisted( req ) )
        return OK;

    /*
     * Parse any CGI arguments if there were any.
     */
    if ( (req->args) || ( req->method_number == M_POST ) )
        cgi = mod_ifier_parse_cgi_parameters( req );
    else
        debug_printf("Not a CGI request\n" );

    /**
     * Iterate over our rules and return 1 if the client address is
     * matched against the relevent blacklist/whitelist value.
     */
    if ( conf->rules->nelts )
    {
        int blocked = 0;
        mod_ifier_rule *rules = (mod_ifier_rule *) conf->rules->elts;
        int i;

        /**
         * Loop over each rule.
         *
         * NOTE:  As an optimization we cease the loop if
         *       we mark an entry as a matching.  Hence the
         *       "nonstandard" loop.
         */
        for( i = 0 ;
             ( ( i < conf->rules->nelts ) &&  ( blocked == 0 ) ) ;
              i++ )
        {
            /* The rule we're going to match against. */
            mod_ifier_rule *rule  = &rules[i];

            debug_printf("Processing rule %d\n", i );

            switch( rule->rule_type )
            {
            case WHITELIST:
                /* NOP */
                break;
                
            case BLACKLIST:
                blocked = mod_ifier_is_ip_blacklisted( req );
                break;

            case CGI_MATCH:
                blocked = match_cgi( req, rule, cgi );
                break;

            case METHOD_MATCH:
                blocked = match_method( req, rule );
                break;

            case HEADER_MATCH:
                blocked = match_header_rule( req, rule );
                break;

            case PATH_MATCH:
                blocked = match_path( req, rule );
                break;

            default:
                ap_log_error(APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, 0, 
                             req->server, "Unknown rule type - %d", rule->rule_type );
                break;
            }

            /**
             * If we have a match then process it 
             */
            if ( blocked )
            {
                debug_printf("Rule %d matched\n", i );

                /* The action to take for this particular rule */
                mod_ifier_action *act = rule->action;

                /* No action for this rule?  Use the default. */
                if ( act == NULL )
                    act = conf->defaultAction;

                /* Do any global logging if we're supposed to. */
                log_headers( req, NULL );

                /* Execute */
                if ( act->exec != NULL )
                    run_external_command( req, act->exec );

                /* Per-rule Logfile */
                if ( act->path != NULL )
                    log_headers( req, act->path );

                /* HTTP status code */
                if ( act->response != 0 )
                    return( act->response );

                /* Redirect */
                if ( act->uri != NULL )
                {
                    apr_table_setn(req->headers_out, "Location", act->uri );
                    return(HTTP_MOVED_TEMPORARILY);
                }

            }
            else
                debug_printf("Rule %d did not match\n", i );
        }
    }

    /*
     * We didn't have a match - so the response should be allowed.
     */
    return OK;
}




/**
 * This function is a callback and it declares which other functions should
 * be called for request processing and configuration requests.
 */
static void mod_ifier_register_hooks (apr_pool_t * p)
{
    /* Called to filter the request */
    ap_hook_fixups(mod_ifier_test_incoming_request, NULL, NULL, APR_HOOK_FIRST);

    /* Called to show our rules */
    ap_hook_handler(mod_ifier_handler, NULL, NULL, APR_HOOK_MIDDLE);
}


/**
 * Run the user-specified command with the IP address of the matching
 * client connection as a single parameter.
 *
 * TODO: Perhaps fork() + exec()?
 *
 */
static int run_external_command( request_rec *req, const char *cmd )
{
    char buf[1024];

    /**
     * If we don't have a command defined then we return.
     */
    const char *ip = req->connection->remote_ip;
    if ( ( cmd == NULL ) || ( ip == NULL ) )
        return 0;

    /* build the comand */
    apr_snprintf(buf, sizeof(buf), "%s %s", cmd, ip );

    /* execute it */
    return( system( buf ) );

}


/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * End:
 * vim600: sw=4 ts=4 fdm=marker
 * vim<600: sw=4 ts=4
 */
