/**
 *
 * mod_ifier
 *  Simple filtering module for Apache 2.x which allows you to deny
 * incoming connections based upon semi-arbitary criterea.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 * Furthermore, Steve Kemp gives permission to link mod_ifier
 * with OpenSSL and Apache Licensed works as well as distributing the 
 * linked executables. 
 *
 *  Steve Kemp
 *  ---
 *  http://www.steve.org.uk/
 *
 */


/**
 ** This file contains the structures we define and use within our
 ** module.
 ** 
 ** Each ENUM value starts from 1.  This is so that we can be sure
 ** that we're not using an uninitialised variable when testing.
 **
 * eg.
 *
 *   typedef enum { A, B, C } steve_enum;
 *
 *   if ( i == A ) { .... ?? ... }
 *
 **
 **  In the code snippet above we cannot be sure we matched 'i' against
 ** A - or if it just happened to be zero, an uninitialised variable.
 **
 **
 **/


#ifndef _H_STRUCTS_H 
#define _H_STRUCTS_H 1




/**
 * A structure to hold CGI parameters parsed from a request.
 * Both GET and POST submissions can be parsed, but we never care
 * which it was that the request came from.
 */
typedef struct
{
    /**
     * The name of the submitted CGI parameter.
     */
    char * name;

    /**
     * The contents of the submitted CGI parameter.
     */
    char * value;

} mod_ifier_cgi_arguments;



/**
 *  This structure is used to store an action to take when
 * a rule matches.
 * 
 *  A rule may contain its own action structure, or we may fall
 * back to the server-wide default.
 *
 */
typedef struct
{
    /**
     * What are we going to do?
     */
    enum mod_ifier_action_type
    {
        EXECUTE = 1,
        HTTP_REDIRECT,
        HTTP_RESPONSE
    } action;

    /**
     * If we're executing a program then here we specify the program.
     */
    const char *exec;

    /**
     * Per-rule logfile.
     */
    const char *path;

    /**
     * If this is to return a HTTP code here it is.
     */
    int  response;
    
    /**
     * If we're redirecting then here we specify the URL,
     */
    const char *uri;

    
} mod_ifier_action ;



/**
 * What type of rule is this?
 *
 * 1.  Header match.
 * 2.  CGI Parameter match.
 * 3.  Method match
 * 4.  Request pattern match.
 * 5.  IP whitelist.
 * 6.  IP blacklist.
 *
 */
typedef enum 
{
    HEADER_MATCH = 1,  /* Make sure we don't start at zero. */
    CGI_MATCH,
    METHOD_MATCH,
    PATH_MATCH,
    WHITELIST,
    BLACKLIST

} mod_ifier_rule_type;




/**
 * This structure is the key to our module.
 *
 * Each rule which is parsed from the Apache configuration file
 * is converted into a copy of this structure.  The server-wide configuration
 * structure contains an array of these objects.
 *
 */
typedef struct
{
    /**
     * What action should we conduct if this rule matches?
     *
     * If this is NULL then we'll use the global action.
     */
    mod_ifier_action *action;

    /**
     * The type of rule this is.
     */
    mod_ifier_rule_type  rule_type;    

    /**
     **  Now we have per-type match data.
     **/

    /**
     * Name, used for method matching, header matching, CGI param matching.
     */
    const char *name;

    /**
     * Value used for header value, CGI param value, and path matching.
     *
     * We store both the text + regexp so we can use the text for logging.
     */
    ap_regex_t *value;

    /**
     * Value used for an IP blacklist/whitelist match.
     */
    apr_ipsubnet_t* ip;

    /**
     * In the case of a compiled regular expression or IP address
     * match we store the plain-text so that we can use it in 
     * the output logs.
     */
    const char *val_text;
    
} mod_ifier_rule ;



/**
 * 
 * This is the configuration structure for our module, it contains an
 * array of rules to match against along with some other configuration
 * settings such as logfile names.
 *
 */
typedef struct
{
    /**
     * The collection of rules we're going to match against.
     */
    apr_array_header_t *rules;

    /**
     * The default action to take if we've matched.
     * This can be overridden on a per-rule basis.
     */
    mod_ifier_action *defaultAction;

    /**
     * The logfile(s) we should write to.
     */
    char * logfile;
    char * logfile_headers;


} mod_ifier_config;



#endif /* _H_STRUCTS_H */


/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * End:
 * vim600: sw=4 ts=4 fdm=marker
 * vim<600: sw=4 ts=4
 */

