/***************************************************************************
 $RCSfile: kbanking.cpp,v $
                             -------------------
    cvs         : $Id: kbanking.cpp,v 1.12 2005/02/20 23:18:54 aquamaniac Exp $
    begin       : Mon Mar 01 2004
    copyright   : (C) 2004 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *          Please see toplevel file COPYING for license details           *
 ***************************************************************************/


#ifdef HAVE_CONFIG_H
# include <config.h>
#endif


#include "kbanking_p.h"
#include "progress.h"
#include "simplebox.h"
#include "inputbox.h"

#include <assert.h>
#include <qstring.h>
#include <qmessagebox.h>
#include <qinputdialog.h>
#include <qapplication.h>
#include <qdatetime.h>

#include <gwenhywfar/debug.h>



KBanking::KBanking(const char *appname,
                   const char *fname,
                   QObject *parent,
                   const char *name)
:QObject(parent, name), AB::Banking(appname, fname)
,_parentWidget(0)
,_lastWidgetId(0)
,_logLevel(AB_Banking_LogLevelInfo){
}



KBanking::~KBanking(){
}


int KBanking::_extractHTML(const char *text, GWEN_BUFFER *tbuf) {
  GWEN_BUFFEREDIO *bio;
  GWEN_XMLNODE *xmlNode;
  GWEN_BUFFER *buf;
  int rv;

  buf=GWEN_Buffer_new(0, 256, 0, 1);
  GWEN_Buffer_AppendString(buf, text);
  GWEN_Buffer_Rewind(buf);
  bio=GWEN_BufferedIO_Buffer2_new(buf, 1);
  GWEN_BufferedIO_SetReadBuffer(bio, 0, 256);
  xmlNode=GWEN_XMLNode_new(GWEN_XMLNodeTypeTag, "html");
  rv=GWEN_XML_Parse(xmlNode, bio,
		    GWEN_XML_FLAGS_DEFAULT |
                    GWEN_XML_FLAGS_HANDLE_OPEN_HTMLTAGS);
  GWEN_BufferedIO_Close(bio);
  GWEN_BufferedIO_free(bio);
  if (rv) {
    DBG_NOTICE(0, "here");
    GWEN_XMLNode_free(xmlNode);
    return -1;
  }
  else {
    GWEN_XMLNODE *nn;

    nn=GWEN_XMLNode_FindFirstTag(xmlNode, "html", 0, 0);
    if (nn) {
      GWEN_XMLNODE *on, *onn;

      on=GWEN_XMLNode_new(GWEN_XMLNodeTypeTag, "root");
      onn=GWEN_XMLNode_new(GWEN_XMLNodeTypeTag, "qt");
      GWEN_XMLNode_AddChild(on, onn);
      GWEN_XMLNode_AddChildrenOnly(onn, nn, 1);

        /* text contains HTML tag, take it */
      bio=GWEN_BufferedIO_Buffer2_new(tbuf, 0);
      GWEN_BufferedIO_SetWriteBuffer(bio, 0, 256);
      rv=GWEN_XMLNode_WriteToStream(on, bio, GWEN_XML_FLAGS_DEFAULT);
      GWEN_XMLNode_free(on);
      if (rv) {
	DBG_ERROR(AQBANKING_LOGDOMAIN, "Error writing data to stream");
	GWEN_BufferedIO_Abandon(bio);
	GWEN_BufferedIO_free(bio);
        GWEN_XMLNode_free(xmlNode);
	return -1;
      }
      else {
	rv=GWEN_BufferedIO_Close(bio);
	GWEN_BufferedIO_free(bio);
        if (rv) {
          GWEN_XMLNode_free(xmlNode);
          return -1;
        }
      }
    }
    else {
      GWEN_XMLNode_free(xmlNode);
      return 1;
    }
  }
  GWEN_XMLNode_free(xmlNode);
  return 0;
}



int KBanking::messageBox(GWEN_TYPE_UINT32 flags,
                         const char *title,
                         const char *text,
                         const char *b1,
                         const char *b2,
                         const char *b3){
  int rv;
  GWEN_BUFFER *buf;

  buf=GWEN_Buffer_new(0, strlen(text), 0, 1);
  if (!_extractHTML(text, buf)) {
    text=GWEN_Buffer_GetStart(buf);
  }

  switch(flags & AB_BANKING_MSG_FLAGS_TYPE_MASK) {
  case AB_BANKING_MSG_FLAGS_TYPE_WARN:
    rv=QMessageBox::warning(0, QString::fromUtf8(title), QString::fromUtf8(text), 
			    b1 ? QString::fromUtf8(b1) : QString::null,
			    b2 ? QString::fromUtf8(b2) : QString::null,
			    b3 ? QString::fromUtf8(b3) : QString::null);
    break;
  case AB_BANKING_MSG_FLAGS_TYPE_ERROR:
    rv=QMessageBox::critical(0, QString::fromUtf8(title), QString::fromUtf8(text), 
			     b1 ? QString::fromUtf8(b1) : QString::null,
			     b2 ? QString::fromUtf8(b2) : QString::null,
			     b3 ? QString::fromUtf8(b3) : QString::null);
    break;
  case AB_BANKING_MSG_FLAGS_TYPE_INFO:
  default:
    rv=QMessageBox::information(0, QString::fromUtf8(title), QString::fromUtf8(text),
				b1 ? QString::fromUtf8(b1) : QString::null,
				b2 ? QString::fromUtf8(b2) : QString::null,
				b3 ? QString::fromUtf8(b3) : QString::null);
    break;
  }
  rv++;
  GWEN_Buffer_free(buf);
  return rv;
}



int KBanking::inputBox(GWEN_TYPE_UINT32 flags,
                       const char *title,
                       const char *text,
                       char *buffer,
                       int minLen,
                       int maxLen){
  GWEN_BUFFER *buf;

  buf=GWEN_Buffer_new(0, strlen(text), 0, 1);
  if (!_extractHTML(text, buf)) {
    text=GWEN_Buffer_GetStart(buf);
  }

  KBInputBox ib(QString::fromUtf8(title), QString::fromUtf8(text),
		flags, minLen, maxLen, 0, "InputBox", true);
  GWEN_Buffer_free(buf);
  if (ib.exec()==QDialog::Accepted) {
    QString s;
    int l;

    s=ib.getInput();
    l=s.length();
    if (l && l<maxLen-1) {
      memmove(buffer, s.latin1(), l);
      buffer[l]=0;
    }
    else {
      DBG_ERROR(0, "Bad pin length");
      return AB_ERROR_INVALID;
    }
    return 0;
  }
  else {
    DBG_WARN(0, "Aborted by user");
    return AB_ERROR_USER_ABORT;
  }
}



GWEN_TYPE_UINT32 KBanking::showBox(GWEN_TYPE_UINT32 flags,
                                   const char *title,
                                   const char *text){
  GWEN_TYPE_UINT32 id;
  KBSimpleBox *b;
  QWidget *w;
  GWEN_BUFFER *buf;

  buf=GWEN_Buffer_new(0, strlen(text), 0, 1);
  if (!_extractHTML(text, buf)) {
    text=GWEN_Buffer_GetStart(buf);
  }

  w=_findProgressWidget(0);
  id=++_lastWidgetId;
  b=new KBSimpleBox(id, QString::fromUtf8(title),
		    QString::fromUtf8(text),
		    w, "SimpleBox",
		    Qt::WType_TopLevel | Qt::WType_Dialog | Qt::WShowModal);
  GWEN_Buffer_free(buf);
  //b->setModal(true);
  if (flags & AB_BANKING_SHOWBOX_FLAGS_BEEP)
    QApplication::beep();

  b->show();
  _simpleBoxWidgets.push_front(b);
  qApp->processEvents();
  return id;
}



void KBanking::hideBox(GWEN_TYPE_UINT32 id){
  if (_simpleBoxWidgets.size()==0) {
    DBG_WARN(0, "No simpleBox widgets");
    return;
  }
  if (id==0) {
    KBSimpleBox *b;

    b=_simpleBoxWidgets.front();
    b->close(true);
    _simpleBoxWidgets.pop_front();
  }
  else {
    std::list<KBSimpleBox*>::iterator it;
    for (it=_simpleBoxWidgets.begin(); it!=_simpleBoxWidgets.end(); it++) {
      if ((*it)->getId()==id) {
        (*it)->close(true);
        _simpleBoxWidgets.erase(it);
        break;
      }
    }
  }
  qApp->processEvents();
}



GWEN_TYPE_UINT32 KBanking::progressStart(const char *title,
                                         const char *text,
                                         GWEN_TYPE_UINT32 total){
  GWEN_TYPE_UINT32 id;
  KBProgress *pr;

  _cleanupProgressWidgets();
  id=++_lastWidgetId;
  pr=new KBProgress(id, QString::fromUtf8(title), 
		    QString::fromUtf8(text), 
		    _parentWidget, "ProgressWidget",
                    Qt::WType_Dialog | Qt::WShowModal);
  if (pr->start(total)) {
    DBG_ERROR(0, "Could not start progress dialog");
    delete pr;
    return 0;
  }
  pr->show();
  _progressWidgets.push_front(pr);
  return id;
}



KBProgress *KBanking::_findProgressWidget(GWEN_TYPE_UINT32 id) {
  std::list<KBProgress*>::iterator it;

  if (_progressWidgets.size()==0) {
    DBG_DEBUG(0, "No progress widgets");
    return 0;
  }
  if (id==0)
    return _progressWidgets.front();
  for (it=_progressWidgets.begin(); it!=_progressWidgets.end(); it++) {
    if ((*it)->getId()==id)
      return *it;
  }

  DBG_WARN(0, "Progress widget \"%08x\" not found", (unsigned int)id);
  return 0;
}



void KBanking::_cleanupProgressWidgets() {
  std::list<KBProgress*>::iterator it;

  while(1) {
    bool del;

    del=false;
    for (it=_progressWidgets.begin(); it!=_progressWidgets.end(); it++) {
      if ((*it)->isClosed()) {
        delete *it;
        _progressWidgets.erase(it);
        del=true;
        break;
      }
    } /* for */
    if (!del)
      break;
  } /* while */
}



int KBanking::progressAdvance(GWEN_TYPE_UINT32 id,
                              GWEN_TYPE_UINT32 progress){
  KBProgress *pr;

  pr=_findProgressWidget(id);
  if (pr) {
    return pr->advance(progress);
  }
  else
    return AB_ERROR_NOT_FOUND;
}



int KBanking::progressLog(GWEN_TYPE_UINT32 id,
                          AB_BANKING_LOGLEVEL level,
                          const char *chartext){
  KBProgress *pr;
  QString text(QString::fromUtf8(chartext));

  // Necessary when passing this QString into the macros
  const char *latin1text = text.latin1();

  if (level>_logLevel) {
    DBG_NOTICE(0, "Not logging this: %02d: %s (we are at %d)",
               level, latin1text, _logLevel);
    /* don't log this */
    return 0;
  }

  DBG_INFO(0, "%02d: %s", level, latin1text);
  pr=_findProgressWidget(id);
  if (pr) {
    return pr->log(level, text);
  }
  else {
    return AB_ERROR_NOT_FOUND;
  }
}



int KBanking::progressEnd(GWEN_TYPE_UINT32 id){
  KBProgress *pr;
  int res;

  if (_progressWidgets.size()==0) {
    DBG_INFO(0, "No active progress widget");
    return AB_ERROR_NOT_FOUND;
  }

  res=AB_ERROR_NOT_FOUND;
  if (id==0) {
    pr=_progressWidgets.front();
    res=pr->end();
    _progressWidgets.pop_front();
  }
  else {
    std::list<KBProgress*>::iterator it;

    for (it=_progressWidgets.begin(); it!=_progressWidgets.end(); it++) {
      if ((*it)->getId()==id) {
	res=(*it)->end();
	_progressWidgets.erase(it);
	break;
      }
    }
  }

  return res;
}



void KBanking::setParentWidget(QWidget *w) {
  _parentWidget=w;
}



int KBanking::enqueueJob(AB_JOB *j){
  int rv;

  rv=AB::Banking::enqueueJob(j);
  emit(signalQueueUpdated());
  return rv;
}



int KBanking::dequeueJob(AB_JOB *j){
  int rv;

  rv=AB::Banking::dequeueJob(j);
  emit(signalQueueUpdated());
  return rv;
}



int KBanking::executeQueue(){
  int rv;

  rv=AB::Banking::executeQueue();
  emit(signalQueueUpdated());
  return rv;
}



void KBanking::accountsUpdated(){
  emit(signalAccountsUpdated());
}



std::string KBanking::QStringToUtf8String(const QString &qs) {
  std::string result;
  QCString utfData=qs.utf8();
  int len;
  int i;

  len=utfData.length();
  for (i=0; i<len; i++)
    result+=(unsigned char)utfData[i];

  return result;
}











