#include "drawable-node.h"
#include <string.h>
#include "wsint.h"
#include <math.h>

#include <X11/Xlib.h>
#include <X11/extensions/Xfixes.h>

#if G_BYTE_ORDER == G_LITTLE_ENDIAN
#   define PIXEL_TYPE GL_UNSIGNED_INT_8_8_8_8_REV
#else
#   define PIXEL_TYPE GL_UNSIGNED_INT_8_8_8_8
#endif

typedef struct Tile
{
    gint x;
    gint y;
    gint width;
    gint height;
    
    GLuint texture;
} Tile;

G_DEFINE_TYPE (CmDrawableNode, cm_drawable_node, CM_TYPE_NODE);

static void drawable_node_render (CmNode *node);
static void drawable_node_compute_extents (CmNode  *node,
					   Extents *extents);

static void
cm_drawable_node_finalize (GObject *object)
{
    CmDrawableNode *node = CM_DRAWABLE_NODE (object);
    WsDisplay *display = WS_RESOURCE (node->drawable)->display;
    
#if 0
    g_print ("finalizing drawable node %p containing %lx\n",
	     object, WS_RESOURCE_XID (node->drawable));
#endif
    
    ws_display_begin_error_trap (display);
    
    if (node->pixmap)
	g_object_unref (node->pixmap);
    
    ws_display_end_error_trap (display);
    
    /* FIXME: remove this when Ws* has been gobjectified */
    ws_window_set_configure_callback (
	(WsWindow *)node->drawable, NULL, NULL);
    
    G_OBJECT_CLASS (cm_drawable_node_parent_class)->finalize (object);
}

static void
cm_drawable_node_class_init (CmDrawableNodeClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);
    CmNodeClass *node_class = CM_NODE_CLASS (class);
    
    object_class->finalize = cm_drawable_node_finalize;
    
    node_class->render = drawable_node_render;
    node_class->compute_extents = drawable_node_compute_extents;
}

static void
cm_drawable_node_init (CmDrawableNode *drawable_node)
{
    
}

void
cm_drawable_node_set_damage_func (CmDrawableNode *node,
				  DrawableDamageFunc damage_func,
				  gpointer data)
{
    node->damage_func = damage_func;
    node->damage_data = data;
}

static void
queue_paint (CmDrawableNode *node)
{
#if 0
    g_print ("calling into metacity\n");
#endif
    
    if (node->damage_func)
	node->damage_func (node, node->damage_data);
}

#define SHADOW_WIDTH 64

static void
bind_shadow_texture(void)
{
    static unsigned char texture[4 * SHADOW_WIDTH * SHADOW_WIDTH];
    static gboolean initialized;
    static GLuint texture_name;
    int i, j, dx, dy;
    double d, alpha;
    const int xofs = 4, yofs = 4;
    const int fuzz = 2 * SHADOW_WIDTH / 3;
    
    if (!initialized)
    {
	initialized = TRUE;
	glGenTextures (1, &texture_name);
	for (i = 0; i < SHADOW_WIDTH * 2; i++)
	{
	    for (j = 0; j < SHADOW_WIDTH * 2; j++)
	    {
		if (i < (xofs + fuzz))
		    dx = (xofs + fuzz) - i;
		else if (i < (xofs + 2 * SHADOW_WIDTH - fuzz))
		    dx = 1;
		else
		    dx = i - (xofs + 2 * SHADOW_WIDTH - fuzz + 1);
		
		if (j < (yofs + fuzz))
		    dy = (yofs + fuzz) - j;
		else if (j < (yofs + 2 * SHADOW_WIDTH - fuzz))
		    dy = 1;
		else
		    dy = j - (yofs + 2 * SHADOW_WIDTH - fuzz + 1);
		
		d = sqrt(dx * dx + dy * dy);
		alpha = pow(0.8, d * 0.7) * 90;
		
		texture[i + j * SHADOW_WIDTH * 2] = alpha;
	    }	  
	}
	
	glBindTexture (GL_TEXTURE_RECTANGLE_ARB, texture_name);
	glTexImage2D(GL_TEXTURE_RECTANGLE_ARB, 0, GL_ALPHA,
		     SHADOW_WIDTH * 2, SHADOW_WIDTH * 2, 0,
		     GL_ALPHA, GL_BYTE, texture);
    }
    
    glBindTexture (GL_TEXTURE_RECTANGLE_ARB, texture_name);
}

static void
emit_quad(int u0, int v0, int u1, int v1,
	  int x0, int y0, int x1, int y1,
	  CmDrawableNode *node)
{
    int dx, dy;
    
    glTexCoord2i (u0, v0);
    node->deform_func(x0, y0, node->real_x, node->real_y,
		      node->real_width, node->real_height,
		      &dx, &dy, node->deform_data);
    glVertex3i (dx, dy, 0);
    glTexCoord2i (u1, v0);
    node->deform_func(x1, y0, node->real_x, node->real_y,
		      node->real_width, node->real_height,
		      &dx, &dy, node->deform_data);
    glVertex3i (dx, dy, 0);
    glTexCoord2i (u1, v1);
    node->deform_func(x1, y1, node->real_x, node->real_y,
		      node->real_width, node->real_height,
		      &dx, &dy, node->deform_data);
    glVertex3i (dx, dy, 0);
    glTexCoord2i (u0, v1);
    node->deform_func(x0, y1, node->real_x, node->real_y,
		      node->real_width, node->real_height,
		      &dx, &dy, node->deform_data);
    glVertex3f (dx, dy, 0);
}

static void
draw_window (CmDrawableNode *node)
{
    const int tile_size = 32;
    int u0, v0, u1, v1;
    int w, h;
    
    w = node->real_width;
    h = node->real_height;
    
    glBegin (GL_QUADS);
    
    for (u0 = 0; u0 < w; u0 += tile_size)
    {
	for (v0 = 0; v0 < h; v0 += tile_size)
	{
	    u1 = MIN(u0 + tile_size, w);
	    v1 = MIN(v0 + tile_size, h);
	    
	    emit_quad(u0, v0, u1, v1, u0, v0, u1, v1, node);
	}
    }
    
    glEnd ();
    
    glColor4f (0.0, 0.0, 0.0, node->alpha);
    glEnable (GL_TEXTURE_RECTANGLE_ARB);
    bind_shadow_texture();
    glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glEnable (GL_BLEND);
    
    glBegin (GL_QUADS);
    
    emit_quad(0, 0, tile_size, tile_size,
	      -tile_size, -tile_size, 0, 0, node);
    emit_quad(tile_size, 0, 2 * tile_size, tile_size,
	      0, -tile_size, tile_size, 0, node);
    
    emit_quad(2 * tile_size, 0, 3 * tile_size, tile_size,
	      w - tile_size, -tile_size, w, 0, node);
    emit_quad(3 * tile_size, 0, 4 * tile_size, tile_size,
	      w, -tile_size, w + tile_size, 0, node);
    
    emit_quad(0, tile_size, tile_size, 2 * tile_size,
	      -tile_size, 0, 0, tile_size, node);
    emit_quad(3 * tile_size, tile_size, 4 * tile_size, 2 * tile_size,
	      w, 0, w + tile_size, tile_size, node);
    
    for (u0 = tile_size; u0 < w - tile_size; u0 += tile_size)
    {
	v0 = -tile_size;
	v1 = 0;
	u1 = MIN(u0 + tile_size, w - tile_size);
	
	emit_quad(2 * tile_size, 0, 2 * tile_size, tile_size,
		  u0, v0, u1, v1, node);
	
	v0 = h;
	v1 = h + tile_size;
	emit_quad(2 * tile_size, 3 * tile_size, 2 * tile_size, 4 * tile_size,
		  u0, v0, u1, v1, node);
    }
    
    
    for (v0 = tile_size; v0 < h - tile_size; v0 += tile_size)
    {
	u0 = -tile_size;
	u1 = 0;
	v1 = MIN(v0 + tile_size, h - tile_size);
	
	emit_quad(0, 2 * tile_size, tile_size, 2 * tile_size,
		  u0, v0, u1, v1, node);
	
	u0 = w;
	u1 = w + tile_size;
	
	emit_quad(3 * tile_size, 2 * tile_size, 4 * tile_size, 2 * tile_size,
		  u0, v0, u1, v1, node);
    }
    
    emit_quad(0, 2 * tile_size, tile_size, 3 * tile_size,
	      -tile_size, h - tile_size, 0, h, node);
    emit_quad(3 * tile_size, 2 * tile_size, 4 * tile_size, 3 * tile_size,
	      w, h - tile_size, w + tile_size, h, node);
    
    emit_quad(tile_size, 3 * tile_size, 2 * tile_size, 4 * tile_size,
	      0, h, tile_size, h + tile_size, node);
    emit_quad(2 * tile_size, 3 * tile_size, 3 * tile_size, 4 * tile_size,
	      w - tile_size, h, w, h + tile_size, node);
    
    emit_quad(0, 3 * tile_size, tile_size, 4 * tile_size,
	      -tile_size, h, 0, h + tile_size, node);
    emit_quad(3 * tile_size, 3 * tile_size, 4 * tile_size, 4 * tile_size,
	      w, h, w + tile_size, h + tile_size, node);
    
    glEnd ();
}

static void
print_matrix (const char *header, GLdouble m[16])
{
    int i, j;
    
    printf ("%s:\n", header);
    
    for (i = 0; i < 16; ++i)
    {
	printf ("%f ", m[i]);
	
	if (++j == 4)
	{
	    printf ("\n");
	    j = 0;
	}
    }
    
}

static void
print_coord (int x, int y, int z)
{
    GLdouble model[16];
    GLdouble proj[16];
    GLint    view[4];
    double   wx, wy, wz;
    
    glGetDoublev (GL_MODELVIEW_MATRIX, model);
    glGetDoublev (GL_PROJECTION_MATRIX, proj);
    glGetIntegerv (GL_VIEWPORT, view);
    
    gluProject (x, y, z,
		model, proj, view,
		&wx, &wy, &wz);
    
#if 0
    print_matrix ("model", model);
    print_matrix ("proj",  proj);
    g_print ("viewport: %d %d %d %d\n", view[0], view[1], view[2], view[3]);
    
    printf ("projected: %d %d %d  =>   %f %f %f\n", x, y, z, wx, wy, wz);
#endif
}

static void
update_geometry (CmDrawableNode *node)
{
    WsRectangle geometry;
    
    if (!ws_drawable_query_geometry (WS_DRAWABLE (node->drawable), &geometry))
    {
	return;
    }
    
    node->real_x = geometry.x;
    node->real_y = geometry.y;
    node->real_width = geometry.width;
    node->real_height = geometry.height;
}

static void
drawable_node_render (CmNode *node)
{
    CmDrawableNode *drawable_node = CM_DRAWABLE_NODE (node);
    GLuint texture;
    WsScreen *screen;
    int scr_w, scr_h;
#if 0
    double vw, vh;
    double vx, vy;
#endif
    
    if (!drawable_node->viewable)
	return;
    
    if (!drawable_node->pixmap)
	return;
    
    screen = ws_drawable_query_screen (WS_DRAWABLE (drawable_node->pixmap));
    texture = ws_pixmap_get_texture (drawable_node->pixmap);
    
    if (drawable_node->real_width == -1)
	update_geometry (drawable_node);
    
    scr_w = ws_screen_get_width (screen);
    scr_h = ws_screen_get_height (screen);
    
#if 0
    vw = drawable_node->width / scr_w;
    vh = drawable_node->height / scr_h;
    vx = drawable_node->x / scr_w;
    vy = drawable_node->y / scr_h;
#endif
    
#if 0
    g_print ("binding texture: %d\n", texture);
#endif
    
    glBindTexture (GL_TEXTURE_RECTANGLE_ARB, texture);
    
#if 0
    glEnable (GL_TEXTURE_RECTANGLE_ARB);
    glEnable (GL_BLEND);
    glBlendFunc (GL_ONE, GL_ONE_MINUS_SRC_ALPHA);
#endif
    
#if 0
    glBegin (GL_QUADS);
    glTexCoord2f (0, 0);
    glVertex3f (vx, 1 - vy, 0.0);
    glTexCoord2f (drawable_node->width, 0);
    glVertex3f (vx + vw, 1 - vy, 0.0);
    glTexCoord2f (drawable_node->width, drawable_node->height);
    glVertex3f (vx + vw, 1 - (vy + vh), 0.0);
    glTexCoord2f (0, drawable_node->height);
    glVertex3f (vx, 1 - (vy + vh), 0.0);
    glEnd ();
#endif
    
    glMatrixMode (GL_PROJECTION);
    glLoadIdentity ();
    
    gluOrtho2D (0, scr_w, scr_h, 0);
    
    glDisable (GL_TEXTURE_RECTANGLE_ARB);
    glColor4f (0.0, 0.0, 0.1, 0.3);
    
    glMatrixMode (GL_MODELVIEW);
    glLoadIdentity ();
    
    /* Emperically determined constant that gets rid of some fuzzyness
     */
    glTranslatef (-0.04, -0.04, 0);
    
    glShadeModel (GL_SMOOTH);
#if 0
    glEnable (GL_LIGHTING);
#endif
    glEnable (GL_BLEND);
    
    glColor4f (1.0, 1.0, 1.0, drawable_node->alpha);
    glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
    glEnable (GL_TEXTURE_RECTANGLE_ARB);
#if 0
    glColor4f (1.0, 0.9 + 0.1 * g_random_double(), 0.0, 1.0);
#endif
    
    glBlendFunc (drawable_node->alpha < 0.99 ? GL_SRC_ALPHA : GL_ONE, GL_ONE_MINUS_SRC_ALPHA);
    glDisable (GL_LIGHTING);
    
    draw_window(drawable_node);
    
#if 0
    glPopMatrix ();
#endif
    
#if 0
    g_print ("done painting\n");
#endif
}

static void
drawable_node_compute_extents (CmNode  *node,
			       Extents *extents)
{
    extents->x = 0.0;
    extents->y = 0.0;
    extents->z = 0.0;
    extents->width = 1.0;
    extents->height = 1.0;
    extents->depth = 1.0;
}

static void
on_damage (WsPixmap *pixmap, gpointer data)
{
    CmDrawableNode *node = data;
    
#if 0
    g_print ("queueing paint\n");
#endif
    
    queue_paint (node);
}

static void
refresh_pixmap (CmDrawableNode *node)
{
    WsWindow *window;
    WsDisplay *display = WS_RESOURCE (node->drawable)->display;
    WsRectangle geometry;
    
#if 0
    g_print ("refreshing pixmap\n");
#endif
    
    if (node->pixmap && !ws_pixmap_get_updates (node->pixmap))
	return;
    
    window = WS_WINDOW (node->drawable);
    
    if (node->pixmap)
    {
#if 0
	g_print ("unreffing %p\n", node->pixmap);
#endif
	g_object_unref (node->pixmap);
    }
    
    ws_display_begin_error_trap (display);
    
    node->pixmap = NULL;
    
    if (ws_window_query_mapped (window)		&&
	!ws_window_query_input_only (window))
    {
	node->pixmap = ws_window_name_pixmap (window);
    }
    
    if (ws_display_end_error_trap_with_return (display))
    {
	ws_display_begin_error_trap (display);
	
	if (node->pixmap)
	{
	    g_object_unref (node->pixmap);
	    
	    node->pixmap = NULL;
	}
	
	ws_display_end_error_trap (display);
    }
    else
    {
	if (node->pixmap)
	{
	    ws_drawable_query_geometry (WS_DRAWABLE (window), &geometry);
	    
#if 0
	    g_print ("naming pixmap %lx on window %lx (%d x %d)\n", WS_RESOURCE_XID (node->pixmap), WS_RESOURCE_XID (window), geometry.width, geometry.height);
#endif
	    
	    ws_drawable_query_geometry (WS_DRAWABLE (node->pixmap), &geometry);
	    
#if 0
	    g_print ("pixmap size (%d x %d)\n", geometry.width, geometry.height);
#endif
	    ws_pixmap_set_damage_callback (node->pixmap, on_damage, node);
	}
    }
}

static void
on_configure (WsWindow *window,
	      int x,
	      int y,
	      int width,
	      int height,
	      int bw,
	      WsWindow *above,
	      gboolean override_redirect,
	      gpointer data)
{
    CmDrawableNode *node = data;
    
    if (node->pixmap && !ws_pixmap_get_updates (node->pixmap))
	return;
    
    // g_print ("configure\n");
    if (width != node->real_width ||
	height != node->real_height)
    {
	refresh_pixmap (node);
    }
    
    node->real_x = x;
    node->real_y = y;
    node->real_width = width;
    node->real_height = height;
    
    queue_paint (node);
}

CmDrawableNode *
cm_drawable_node_new (WsDrawable *drawable)
{
    CmDrawableNode *node;
    WsDisplay *display;
    
    g_return_val_if_fail (drawable != NULL, NULL);
    
    display = WS_RESOURCE (drawable)->display;
    
    node = g_object_new (CM_TYPE_DRAWABLE_NODE, NULL);
    
    g_print ("creating drawable node %p containing %lx\n",
	     node, WS_RESOURCE_XID (drawable));
    
    node->drawable = drawable;
    
    ws_window_set_configure_callback (
	WS_WINDOW (drawable), on_configure, node);
    node->viewable = TRUE;
    
    node->deform_func = cm_identity_deform;
    node->deform_data = NULL;
    
    node->damage_func = NULL;
    node->damage_data = NULL;
    
    node->timer = g_timer_new ();
    
    node->real_x = -1;
    node->real_y = -1;
    node->real_width = -1;
    node->real_height = -1;
    
    ws_display_init_composite (ws_drawable_get_display (drawable));
    ws_display_init_damage (ws_drawable_get_display (drawable));
    
    node->alpha = 1.0;
    
    if (WS_IS_WINDOW (drawable))
    {
	refresh_pixmap (node);
    }
    else
    {
	g_print ("this should not happen \n");
	node->pixmap = WS_PIXMAP (drawable);
    }
    
    return node;
}

void
cm_drawable_node_set_viewable (CmDrawableNode *node,
			       gboolean	  viewable)
{
    node->viewable = viewable;
    
    cm_node_queue_repaint ((CmNode *)node);
}

void
cm_drawable_node_update_pixmap (CmDrawableNode *node)
{
    g_print ("refreshing pixmap because app told us to\n");
    refresh_pixmap (node);
}

void
cm_drawable_node_set_geometry (CmDrawableNode *node,
			       double x,
			       double y,
			       double width,
			       double height)
{
    node->deform_func = cm_user_geometry_deform;
    node->deform_data = node;
    
    node->user_x = x;
    node->user_y = y;
    node->user_width = width;
    node->user_height = height;
    
    queue_paint (node);
}

void
cm_drawable_node_set_patch (CmDrawableNode *node,
			    CmPoint points[][4])
{
    node->deform_func = cm_patch_deform;
    node->deform_data = node;
    
    memcpy (node->patch_points, points, sizeof node->patch_points);
    
    queue_paint (node);
}

void
cm_drawable_node_unset_geometry (CmDrawableNode *node)
{
    node->deform_func = cm_identity_deform;
    node->deform_data = NULL;
    
    queue_paint (node);
}

void
cm_drawable_node_set_updates (CmDrawableNode *node,
			      gboolean        updates)
{
    ws_display_begin_error_trap (ws_drawable_get_display (node->drawable));
    
    ws_pixmap_set_updates (node->pixmap, updates);
    
    if (node->pixmap && ws_pixmap_get_updates (node->pixmap))
	update_geometry (node);
    
    refresh_pixmap (node);
    
    ws_display_end_error_trap (ws_drawable_get_display (node->drawable));
}

void
cm_drawable_node_set_alpha (CmDrawableNode *node,
			    double alpha)
{
    node->alpha = alpha;
    
    queue_paint (node);
}
