#include "wsint.h"

G_DEFINE_TYPE (WsPixmap, ws_pixmap, WS_TYPE_DRAWABLE);

static WsFormat ws_pixmap_get_format (WsDrawable	     *drawable);
static GObject *ws_pixmap_constructor (GType                  type,
				       guint                  n_construct_properties,
				       GObjectConstructParam *construct_params);

static void
ws_pixmap_finalize (GObject *object)
{
    WsPixmap *pixmap = WS_PIXMAP (object);
    
    /* FIXME: Deal with foreign objects correctly */
#if 0
    g_print ("finalizing %p\n", object);
#endif

    if (pixmap->texture)
    {
	glXReleaseTexImageEXT (WS_RESOURCE_XDISPLAY (pixmap),
			       pixmap->glx_pixmap,
			       GLX_FRONT_LEFT_EXT);
	
	glXDestroyGLXPixmap (WS_RESOURCE_XDISPLAY (pixmap),
			     pixmap->glx_pixmap);

#if 0
	g_print ("deleting texture %d\n", pixmap->texture);
#endif
	
	glDeleteTextures (1, &pixmap->texture);
    }

    g_assert (pixmap->damage);
    
    XDamageDestroy (WS_RESOURCE_XDISPLAY (pixmap), pixmap->damage);
    _ws_display_unregister_damage (WS_RESOURCE (pixmap)->display,
				   pixmap->damage);
    
    XFreePixmap (WS_RESOURCE_XDISPLAY (object),
		 WS_RESOURCE_XID (object));
    
    G_OBJECT_CLASS (ws_pixmap_parent_class)->finalize (object);
}

static void
ws_pixmap_class_init (WsPixmapClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);
    WsDrawableClass *drawable_class = WS_DRAWABLE_CLASS (class);
    
    object_class->finalize = ws_pixmap_finalize;
    object_class->constructor = ws_pixmap_constructor;
    drawable_class->get_format = ws_pixmap_get_format;
}

static void
repair_pixmap (WsPixmap *pixmap)
{
    /* Note that it is important to clear the damage _before_
     * updating the texture. Otherwise, we will miss any damage
     * that arrives after updating, but before clearing.
     */

  XDamageSubtract (WS_RESOURCE_XDISPLAY (pixmap),
		     pixmap->damage,
		     None, None);
    
    if (pixmap->texture)
    {
	WsRectangle geometry;
	glBindTexture (GL_TEXTURE_RECTANGLE_ARB, pixmap->texture);

 	ws_drawable_query_geometry (WS_DRAWABLE (pixmap), &geometry);
 	
#if 0
	g_print ("bind size; %d %d (%d bytes)\n", geometry.width, geometry.height, geometry.width * geometry.height * 4);
#endif

	glXBindTexImageEXT (WS_RESOURCE_XDISPLAY (pixmap),
			    pixmap->glx_pixmap,
			    GLX_FRONT_LEFT_EXT, NULL);
    }
    
    if (pixmap->damage_callback)
	pixmap->damage_callback (pixmap, pixmap->damage_data);
}	       

static void
on_damage (WsDisplay *display,
	   Damage     damage,
	   gpointer   data)
{
    WsPixmap *pixmap = data;

#if 0
    g_print ("processing damage on %lx\n", WS_RESOURCE_XID (pixmap));
#endif

    if (pixmap->do_updates)
	repair_pixmap (pixmap);
}

static WsFormat
ws_pixmap_get_format (WsDrawable *drawable)
{
    WsPixmap *pixmap = WS_PIXMAP (drawable);

    return pixmap->format;
}

static GObject *
ws_pixmap_constructor (GType		        type,
		       guint		        n_construct_properties,
		       GObjectConstructParam *construct_params)
{
    GObject *object;
    WsPixmap *pixmap;
    
    object = G_OBJECT_CLASS (ws_pixmap_parent_class)->constructor (
	type, n_construct_properties, construct_params);
    
    pixmap = WS_PIXMAP (object);
    
    pixmap->damage = XDamageCreate (WS_RESOURCE_XDISPLAY (pixmap),
				    WS_RESOURCE_XID (pixmap),
				    XDamageReportNonEmpty);

    XDamageSubtract (WS_RESOURCE_XDISPLAY (pixmap),
		     pixmap->damage,
		     None, None);

    g_assert (pixmap->damage);
    
    _ws_display_register_damage (WS_RESOURCE (pixmap)->display,
				 pixmap->damage,
				 on_damage, pixmap);
    
    return object;
}

static void
ws_pixmap_init (WsPixmap *pixmap)
{
    pixmap->damage_callback = NULL;
    pixmap->damage_data = NULL;
    pixmap->do_updates = TRUE;
}

static WsPixmap *
pixmap_new (WsDisplay *display,
	    XID        xid,
	    WsFormat   format)
{
    WsPixmap *pixmap = g_object_new (WS_TYPE_PIXMAP,
				     "display", display,
				     "xid", xid,
				     NULL);

#if 0
    g_print ("allocating pixmap %lx\n", WS_RESOURCE_XID (pixmap));
#endif
    
    pixmap->format = format;
    
    g_assert (xid == WS_RESOURCE_XID (pixmap));
    
    return pixmap;
}

WsPixmap *
ws_pixmap_new (WsDrawable *drawable,
	       int	   width,
	       int	   height)
{
    Pixmap pixmap;
    Display *xdisplay;
    WsFormat format;

    g_return_val_if_fail (WS_IS_DRAWABLE (drawable), NULL);
    g_return_val_if_fail (
	ws_format_is_viewable (ws_drawable_get_format (drawable)), NULL);
    
    format = ws_drawable_get_format (drawable);
    
    xdisplay = WS_RESOURCE_XDISPLAY (drawable);

    pixmap = XCreatePixmap (xdisplay,
			    WS_RESOURCE_XID (drawable),
			    width, height, ws_format_get_depth (format));
    
    return pixmap_new (WS_RESOURCE (drawable)->display,
		       pixmap, format);
}

WsPixmap *
_ws_pixmap_ensure (WsDisplay *display,
		   XID        xid,
		   WsFormat   format)
{
    WsPixmap *pixmap =
	g_hash_table_lookup (display->xresources, (gpointer)xid);
    
    if (!pixmap)
	pixmap = pixmap_new (display, xid, format);
    
    return pixmap;
}

static GLXPixmap
create_glx_pixmap (WsPixmap *pixmap)
{
    Display *xdisplay = WS_RESOURCE_XDISPLAY (pixmap);
    int screen_no = WS_DRAWABLE_XSCREEN_NO (pixmap);
    WsFormat format = ws_drawable_get_format (WS_DRAWABLE (pixmap));
    XVisualInfo *vis_infos;
    int n_infos;
    XVisualInfo template;
    
    template.screen = screen_no;
    template.depth = ws_format_get_depth (format);
    template.class = TrueColor;
    
    ws_format_get_masks (format,
			 &template.red_mask,
			 &template.green_mask,
			 &template.blue_mask);
    
    vis_infos = XGetVisualInfo (xdisplay,
				VisualScreenMask	|
				VisualDepthMask		|
				VisualClassMask		|
				VisualRedMaskMask	|
				VisualGreenMaskMask	|
				VisualBlueMaskMask,
				&template, &n_infos);
    
    if (!vis_infos)
    {
	g_warning ("No XVisualInfos found - will likely crash");
	return None;
    }
    
    /* All of the XVisualInfos returned should be ok, so just pick
     * the first one.
     */
    
    return glXCreateGLXPixmap (
	xdisplay, vis_infos, WS_RESOURCE_XID (pixmap));
}

GLuint
ws_pixmap_get_texture (WsPixmap  *pixmap)
{
    g_return_val_if_fail (WS_IS_PIXMAP (pixmap), 0);
    
    if (!pixmap->texture)
    {
	WsRectangle geometry;
	pixmap->glx_pixmap = create_glx_pixmap (pixmap);
	
	if (!pixmap->glx_pixmap)
	{
	    g_warning ("Could not create a GLXpixmap for %lx\n",
		       WS_RESOURCE_XID (pixmap));
	    
	    return 0;
	}

	/* FIXME: at this point we are just assuming that texture from
	 * drawable is available. At some point we should reinstate the
	 * XGetImage() code. At that point configure.ac needs to be
	 * changed to not report an error.
	 */
	glGenTextures (1, &pixmap->texture);

 	ws_drawable_query_geometry (WS_DRAWABLE (pixmap), &geometry);
	
#if 0
	g_print ("creating texture %d\n", pixmap->texture);
	g_print ("size; %d %d (%d bytes)\n", geometry.width, geometry.height, geometry.width * geometry.height * 4);
#endif

	glBindTexture (GL_TEXTURE_RECTANGLE_ARB, pixmap->texture);
	
	glTexParameteri (GL_TEXTURE_RECTANGLE_ARB, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri (GL_TEXTURE_RECTANGLE_ARB, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri (GL_TEXTURE_RECTANGLE_ARB, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri (GL_TEXTURE_RECTANGLE_ARB, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexParameteri (GL_TEXTURE_RECTANGLE_ARB, GL_TEXTURE_WRAP_R, GL_CLAMP);

#if 0
        /* XXX JX experimental stuffs */
        glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE);
        glTexEnvi (GL_TEXTURE_ENV, GL_COMBINE_ALPHA, GL_REPLACE);
        glTexEnvi (GL_TEXTURE_ENV, GL_SOURCE0_ALPHA, GL_CONSTANT_ARB);
        /* and then need to set a glColor somewhere... */
#endif
        
	glXBindTexImageEXT (WS_RESOURCE_XDISPLAY (pixmap),
			    pixmap->glx_pixmap,
			    GLX_FRONT_LEFT_EXT, NULL);
	
#if 0
	g_print ("creating damage: %lx on %lx\n", pixmap->damage,
		 WS_RESOURCE_XID (pixmap));
#endif
    }
    
    return pixmap->texture;
}

void
ws_pixmap_set_damage_callback (WsPixmap *pixmap,
			       WsDamageCallback cb,
			       gpointer  data)
{
    pixmap->damage_callback = cb;
    pixmap->damage_data = data;
}

void
ws_pixmap_set_updates (WsPixmap *pixmap, gboolean updates)
{
  pixmap->do_updates = !!updates;

  if (pixmap->do_updates)
    repair_pixmap (pixmap);
}

gboolean
ws_pixmap_get_updates (WsPixmap *pixmap)
{
  return pixmap->do_updates;
}
