#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glx.h>

#include <glib-object.h>

#include "ws.h"
#include "wsint.h"

#include <X11/extensions/shape.h>

G_DEFINE_TYPE (WsWindow, ws_window, WS_TYPE_DRAWABLE);

static WsFormat ws_window_get_format (WsDrawable *drawable);

static void
ws_window_finalize (GObject *object)
{
    G_OBJECT_CLASS (ws_window_parent_class)->finalize (object);
}

static void
ws_window_class_init (WsWindowClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);
    WsDrawableClass *drawable_class = WS_DRAWABLE_CLASS (class);
    
    object_class->finalize = ws_window_finalize;
    drawable_class->get_format = ws_window_get_format;
}

static void
ws_window_init (WsWindow *window)
{
    
}

static WsFormat
ws_window_get_format (WsDrawable *drawable)
{
    XWindowAttributes attrs;

    Display *xdisplay = WS_RESOURCE_XDISPLAY (drawable);
    Window xwindow = WS_RESOURCE_XID (drawable);
    
    if (!XGetWindowAttributes (xdisplay, xwindow, &attrs))
	return WS_FORMAT_UNKNOWN;

    if (attrs.class == InputOnly)
    {
	return WS_FORMAT_INPUT_ONLY;
    }
    else if (attrs.depth == 16 &&
	     attrs.visual->red_mask == 0xf800 &&
	     attrs.visual->green_mask == 0x7e0 &&
	     attrs.visual->blue_mask == 0x1f)
    {
	return WS_FORMAT_RGB_16;
    }
    else if (attrs.depth == 24 &&
	     attrs.visual->red_mask == 0xff0000 &&
	     attrs.visual->green_mask == 0xff00 &&
	     attrs.visual->blue_mask == 0xff)
    {
	return WS_FORMAT_RGB_24;
    }
    else if (attrs.depth == 32 &&
	     attrs.visual->red_mask == 0xff0000 &&
	     attrs.visual->green_mask == 0xff00 &&
	     attrs.visual->blue_mask == 0xff)
    {
	return WS_FORMAT_ARGB_32;
    }
    else
    {
	g_warning ("Unknown visual format depth=%d, r=%#lx/g=%#lx/b=%#lx",
		   attrs.depth, attrs.visual->red_mask,
		   attrs.visual->green_mask, attrs.visual->blue_mask);

	return WS_FORMAT_UNKNOWN;
    }
}

static WsWindow *
window_new (WsDisplay *display,
	    Window xwindow)
{
    g_assert (!g_hash_table_lookup (display->xresources, (gpointer)xwindow));
    
    WsWindow *window = g_object_new (WS_TYPE_WINDOW,
				     "display", display,
				     "xid", xwindow,
				     NULL);

    g_assert (xwindow == WS_RESOURCE_XID (window));
    
    ws_display_begin_error_trap (display);
    
    XSelectInput (display->xdisplay, xwindow,
		  StructureNotifyMask);

    ws_display_end_error_trap (display);
    
    return window;
}

WsWindow *
_ws_window_ensure (WsDisplay *display,
		   Window  xwindow)
{
    WsWindow *window;

    if (!xwindow)
	return NULL;

    window = g_hash_table_lookup (display->xresources, (gpointer)xwindow);
    
    if (!window)
	window = window_new (display, xwindow);

    return window;
}

void
ws_window_set_input_shape	  (WsWindow    *window,
				   WsRegion    *shape)
{
    Display *xdisplay = WS_RESOURCE_XDISPLAY (window);
    
    XFixesSetWindowShapeRegion (xdisplay,
				WS_RESOURCE_XID (window),
				ShapeInput,
				0, 0, WS_RESOURCE_XID (shape));
}

/* Create a window and make it the active GL window
 * The right way forward here is most likely to have a new
 * subclass "GlWindow" with methods
 *
 *	   begin()				// make it active if it isn't
 *	   end()				// 
 *         swap_buffers();			// 
 *
 */
void
ws_window_redirect_subwindows (WsWindow *window)
{
    g_return_if_fail (WS_RESOURCE (window)->display->composite.available);
    
    XCompositeRedirectSubwindows (
	WS_RESOURCE_XDISPLAY (window), WS_RESOURCE_XID (window),
	CompositeRedirectManual);
}

void
ws_window_unredirect_subwindows (WsWindow *window)
{
    g_return_if_fail (WS_RESOURCE (window)->display->composite.available);

    XCompositeUnredirectSubwindows (
	WS_RESOURCE_XDISPLAY (window), WS_RESOURCE_XID (window),
	CompositeRedirectManual);
}

void
ws_window_map (WsWindow *window)
{
    Display *xdisplay;
    
    g_return_if_fail (window != NULL);
    
    xdisplay = WS_RESOURCE_XDISPLAY (window);
    
    XMapWindow (xdisplay, WS_RESOURCE_XID (window));
}

void
ws_window_unmap (WsWindow *window)
{
    Display *xdisplay;

    g_return_if_fail (window != NULL);

    XUnmapWindow (WS_RESOURCE_XDISPLAY (window), WS_RESOURCE_XID (window));
}

void
ws_window_raise (WsWindow *window)
{
    g_return_if_fail (window != NULL);
    
    XRaiseWindow (WS_RESOURCE_XDISPLAY (window), WS_RESOURCE_XID (window));
}

void
ws_window_lower (WsWindow *window)
{
    g_return_if_fail (window != NULL);

    XLowerWindow (WS_RESOURCE_XDISPLAY (window), WS_RESOURCE_XID (window));
}

void
ws_window_gl_swap_buffers (WsWindow *window)
{
    glXSwapBuffers (WS_RESOURCE_XDISPLAY (window), WS_RESOURCE_XID (window));
}

void
ws_window_unredirect (WsWindow *window)
{
    XCompositeUnredirectWindow (
	WS_RESOURCE_XDISPLAY (window),
	WS_RESOURCE_XID (window), CompositeRedirectManual);
}

void
ws_window_set_override_redirect (WsWindow *window, gboolean override_redirect)
{
    XSetWindowAttributes attrs;
    
    attrs.override_redirect = !!override_redirect;
    
    XChangeWindowAttributes (WS_RESOURCE_XDISPLAY (window), WS_RESOURCE_XID (window),
			     CWOverrideRedirect, &attrs);
}

gboolean
ws_window_query_input_only (WsWindow *window)
{
    XWindowAttributes attrs;

    if (!XGetWindowAttributes (WS_RESOURCE_XDISPLAY (window), WS_RESOURCE_XID (window), &attrs))
	return FALSE;

    return attrs.class == InputOnly;
}

gboolean
ws_window_query_mapped (WsWindow *window)
{
    XWindowAttributes attrs;

    if (!XGetWindowAttributes (WS_RESOURCE_XDISPLAY (window), WS_RESOURCE_XID (window), &attrs))
	return FALSE;

    return attrs.map_state == IsUnviewable || attrs.map_state == IsViewable;
}

gboolean
ws_window_query_viewable (WsWindow *window)
{
    XWindowAttributes attrs;

    if (!XGetWindowAttributes (WS_RESOURCE_XDISPLAY (window), WS_RESOURCE_XID (window), &attrs))
	return FALSE;

    return attrs.map_state == IsViewable;
}

GList *
ws_window_query_subwindows (WsWindow *window)
{
    Window root, parent;		/* dummies */
    Window *children;
    guint n_children;
    int i;
    GList *result = NULL;
    
    if (XQueryTree (WS_RESOURCE_XDISPLAY (window),
		    WS_RESOURCE_XID (window), &root, &parent,
		    &children, &n_children))
    {
	for (i = 0; i < n_children; ++i)
	{
	    WsWindow *child = _ws_window_ensure (WS_RESOURCE (window)->display,
						 children[i]);
	    
	    result = g_list_prepend (result, child);
	}

	XFree (children);
    }
    
    return result;
}

/* FIXME: This function just uses XFetchName - it should do the
 * _NET_WM stuff. If this function should exist at all, which maybe it
 * shouldn't
 */

gchar *
ws_window_query_title (WsWindow *window)
{
    char *result = NULL;
    char *name;
    
    if (XFetchName (WS_RESOURCE_XDISPLAY (window),
		    WS_RESOURCE_XID (window), &name))
    {
	result = g_strdup (name);
	XFree (name);
    }
    else
    {
	result = g_strdup ("No title");
    }

    return result;
}

WsWindow *
ws_window_lookup (WsDisplay *display,
		  Window     xwindow)
{
    return _ws_window_ensure (display, xwindow);
}


WsPixmap *
ws_window_name_pixmap (WsWindow *window)
{
    Pixmap xpixmap = XCompositeNameWindowPixmap (WS_RESOURCE_XDISPLAY (window),
						WS_RESOURCE_XID (window));

    return _ws_pixmap_ensure (WS_RESOURCE (window)->display,
			      xpixmap,
			      ws_drawable_get_format (WS_DRAWABLE (window)));
}

void
ws_window_set_configure_callback (WsWindow   *window,
				  WsConfigureCallback cb,
				  gpointer	data)
{
    window->configure_callback = cb;
    window->configure_data = data;
}

void
_ws_window_process_event (WsWindow   *window,
			  XEvent     *xevent)
{
    if (xevent->type == ConfigureNotify)
    {
	XConfigureEvent *configure = &xevent->xconfigure;
	WsWindow *above;

	above = _ws_window_ensure (WS_RESOURCE (window)->display,
				   configure->above);
	
	if (window->configure_callback)
	{
	    window->configure_callback (window,
					configure->x, configure->y,
					configure->width, configure->height,
					configure->border_width,
					above,
					configure->override_redirect,
					window->configure_data);
	}
    }
}
