/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.commons.validator;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Iterator;

/**
 * This contains the results of a set of validation rules processed 
 * on a JavaBean.
 *
 * @version $Revision: 478334 $ $Date: 2006-11-22 21:31:54 +0000 (Wed, 22 Nov 2006) $
 */
public class ValidatorResult implements Serializable {

    /**
     * Map of results.  The key is the name of the <code>ValidatorAction</code>
     * and the value is whether or not this field passed or not.
     */
    protected Map hAction = new HashMap();

    /**
     * <code>Field</code> being validated.
     * TODO This variable is not used.  Need to investigate removing it.
     */
    protected Field field = null;

    /**
     * Constructs a <code>ValidatorResult</code> with the associated field being
     * validated.
     * @param field Field that was validated.
     */
    public ValidatorResult(Field field) {
        this.field = field;
    }

    /**
     * Add the result of a validator action.
     * @param validatorName Name of the validator.
     * @param result Whether the validation passed or failed.
     */
    public void add(String validatorName, boolean result) {
        this.add(validatorName, result, null);
    }

    /**
     * Add the result of a validator action.
     * @param validatorName Name of the validator.
     * @param result Whether the validation passed or failed.
     * @param value Value returned by the validator.
     */
    public void add(String validatorName, boolean result, Object value) {
        hAction.put(validatorName, new ResultStatus(result, value));
    }

    /**
     * Indicate whether a specified validator is in the Result.
     * @param validatorName Name of the validator.
     * @return true if the validator is in the result.
     */
    public boolean containsAction(String validatorName) {
        return hAction.containsKey(validatorName);
    }

    /**
     * Indicate whether a specified validation passed.
     * @param validatorName Name of the validator.
     * @return true if the validation passed.
     */
    public boolean isValid(String validatorName) {
        ResultStatus status = (ResultStatus) hAction.get(validatorName);
        return (status == null) ? false : status.isValid();
    }

    /**
     * Return the result of a validation.
     * @param validatorName Name of the validator.
     * @return The validation result.
     */
    public Object getResult(String validatorName) {
        ResultStatus status = (ResultStatus) hAction.get(validatorName);
        return (status == null) ? null : status.getResult();
    }

    /**
     * Return an Iterator of the action names contained in this Result.
     * @return The set of action names.
     */
    public Iterator getActions() {
        return Collections.unmodifiableMap(hAction).keySet().iterator();
    }

    /**
     * Return a Map of the validator actions in this Result.
     * @return Map of validator actions.
     * @deprecated Use getActions() to return the set of actions
     *             the isValid(name) and getResult(name) methods
     *             to determine the contents of ResultStatus.
     *
     */
    public Map getActionMap() {
        return Collections.unmodifiableMap(hAction);
    }

    /**
     * Returns the Field that was validated.
     * @return The Field associated with this result.
     */
    public Field getField() {
        return this.field;
    }

    /**
     * Contains the status of the validation.
     */
    protected class ResultStatus implements Serializable {
        private boolean valid = false;
        private Object result = null;

       /**
        * Construct a Result status.
         * @param valid Whether the validator passed or failed.
         * @param result Value returned by the validator.
        */
        public ResultStatus(boolean valid, Object result) {
            this.valid = valid;
            this.result = result;
        }

        /**
         * Tests whether or not the validation passed.
         * @return true if the result was good.
         */
        public boolean isValid() {
            return valid;
        }

        /**
         * Sets whether or not the validation passed.
         * @param valid Whether the validation passed.
         */
        public void setValid(boolean valid) {
            this.valid = valid;
        }

        /**
         * Gets the result returned by a validation method.
         * This can be used to retrieve to the correctly
         * typed value of a date validation for example.
         * @return The value returned by the validation.
         */
        public Object getResult() {
            return result;
        }

        /**
         * Sets the result returned by a validation method.
         * This can be used to retrieve to the correctly
         * typed value of a date validation for example.
         * @param result The value returned by the validation.
         */
        public void setResult(Object result) {
            this.result = result;
        }

    }

}