#if   !defined(__SEMAPHORE_HPP)
#define  __SEMAPHORE_HPP

/*
  CoreLinux++ 
  Copyright (C) 1999,2000 CoreLinux Consortium
  
   The CoreLinux++ Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The CoreLinux++ Library Library is distributed in the hope that it will 
   be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  
*/

#if !defined(__COMMON_HPP)
#include <Common.hpp>
#endif

#if   !defined(__ABSTRACTSEMAPHORE_HPP)
#include <AbstractSemaphore.hpp>
#endif

#if   !defined(__SEMAPHOREEXCEPTION_HPP)
#include <SemaphoreException.hpp>
#endif

namespace   corelinux
{

   DECLARE_CLASS( Semaphore );

   /**
   A Semaphore supports the protocol that processes and/or threads agree to
   follow for the purpose of controlled access to a resource. The
   resource can be anything that the developer considers to need access
   controls on such as memory, hardware, methods, computer instructions,
   and so on.
   
   Callers can elect to avoid being put into a blocked state and
   return immediately without control to the resource. Callers may also
   request that they are put into a blocked state for a specified amount
   of time. If, at the end of the specified time, the request has not
   been satisfied, it is returned with a Timeout indicator.
   
   The owner or creator of the semaphore can elect to enforce balking
   behavior on a Semaphore. When so designated, the Semaphore can turn
   back any request until some condition in their solution space is met
   regardless of the callers blocking options. If a caller access
   attempt is balked, is it returned with a Balked indicator.
   */

   class Semaphore : public AbstractSemaphore
   {

   public:


      //
      // Constructors and destructors
      //

                           /**
                           Default constructor requires the identifier
                           of the semaphore in the semaphore group
                           @param SemaphoreGroupPtr The 
                           owning SemaphoreGroup
                           @param SemaphoreIdentifier The identifier
                           from the Semaphore Group
                           @param bool true if recursion enabled
                           @param bool true if balking enabled
                           */

                           Semaphore
                              ( 
                                 SemaphoreGroupPtr,
                                 SemaphoreIdentifierRef,
                                 bool Recursive=false,
                                 bool Balking=false
                              ) throw ( NullPointerException );

                           /// Virtual Destructor

      virtual              ~Semaphore( void );

      //
      // Operator overloads
      //

               /// Equality operator returns true if identifiers match

               bool     operator==( SemaphoreCref aRef ) const;

      //
      // Accessors
      //

               /// Returns true if balking enabled

      virtual  bool  isBalkingEnabled( void ) const;

               /// Returns true if recursion allowed

      virtual  bool  isRecursionEnabled( void ) const;

               /// Returns the identifier of who currently owns the semaphore

      virtual  ThreadIdentifierCref getOwningThreadIdentifier( void ) const;

               /// Return the depth of the recursion for the owner

      virtual  CounterCref    getRecursionQueueLength( void ) const;


   protected:

      //
      // Constructors
      //

                           /// Default constructor throws assertion

                           Semaphore( void ) 
                              throw(Assertion);

                           /// Copy constructor throws assertion

                           Semaphore( SemaphoreCref ) 
                              throw(Assertion);

      //
      // Operator overloads
      //
               /// Assignment operator throws assertion

               SemaphoreRef operator=( SemaphoreCref ) 
                              throw(Assertion);

               /// Operator for increasing theRecursionQueueLength

               CounterCref operator++( void );

               /// Operator for decreasing theRecursionQueueLength

               CounterCref operator--( void );

      //
      // Accessors
      //

               /// Returns a reference to the owning thread

      virtual  ThreadIdentifierRef  getOwnerId( void );

      //
      // Mutators
      //

               /// Sets the owner id to the current thread

      virtual  void  setOwnerId( void );

               /// Sets the recursion length

      virtual  void  setRecursionQueueLength( Counter );

               /// Sets the owner thread id to not owned

      virtual  void  resetOwnerId( void );


   private:

               /// ThreadIdentifier for recursive check

               ThreadIdentifier           theOwningThread;

               /// The recursion queue length

               Counter  theRecursionQueueLength;
         
               /// Recursive mode flag

               bool     theRecursiveMode;

               /// Balking mode flag

               bool     theBalkingMode;

   };
}

#endif // if !defined(__SEMAPHORE_HPP)

/*
   Common rcs information do not modify
   $Author: frankc $
   $Revision: 1.2 $
   $Date: 2000/06/02 11:51:52 $
   $Locker:  $
*/

