/* 
 * $Id: ctkdraw.c,v 1.89 2001/06/13 23:50:48 neuro Exp $
 *
 * CTK - Console Toolkit
 *
 * Copyright (C) 1998-2000 Stormix Technologies Inc.
 *
 * License: LGPL
 *
 * Authors: Kevin Lindsay, Wesley Terpstra
 *  
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *    
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *    
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

/* CTK drawing and widget sizing algorithm
 * 
 * 1.  An GList Nth tree contains the entire object list for each
 *     window object.
 * 2.  Two passes through this tree must be made.
 *     i)  One to go recursively backward from the bottom of each leaf
 *         to set the minimum size of each widget.
 *     ii) Second, to go through the tree recursively from the root
 *         down in order to set the object's size according to the
 *         window size.  Objects may resize their children depending on
 *         container and widget expand, fill, and homogeneous settings.
 */

#include <stdio.h>
#include <glib.h>
#include <stdlib.h>

#include "ctk.h"
#include "ctkstack.h"
#include "ctksize.h"
#include "ctkutil.h"
#include "ctkcolor.h"
#include "ctkmain.h"

void ctk_noise(const gchar* sound)
{
	char* buf;
	
	buf = g_strdup_printf("bplay "
		"/usr/share/enlightenment/themes/ShinyMetal/sound/samples/%s.wav"
		" >/dev/null 2>&1 &", sound);
	
	system(buf);
	g_free(buf);
}

void ctk_beep()
{
	arr_beep();
}

void ctk_inverse_char(gint x, gint y)
{
	arr_set_col_xy(x, y, arr_get_col_xy(x, y) ^ (FG_MASK | BG_MASK));
}

/* Check View Area validity */
gboolean ctk_check_clipping_xy(CtkWidget* widget, gint x, gint y)
{
	if (y >= widget->clip_row &&
	    y < widget->clip_row + widget->clip_height &&
	    x >= widget->clip_col &&
	    x < widget->clip_col + widget->clip_width)
	    return TRUE;
	
	return FALSE;
}

gboolean ctk_clipping_trivial_rejection(CtkWidget* widget)
{
	return (widget->clip_width <= 0 || widget->clip_height <= 0);
}

gint ctk_helper_min(gint a, gint b)
{
	if (a > b) return b;
	return a;
}

gint ctk_helper_max(gint a, gint b)
{
	if (a > b) return a;
	return b;
}

void ctk_widget_intersect(CtkWidget* parent, CtkWidget* widget)
{
	widget->clip_row = ctk_helper_max(parent->clip_row, widget->row);
	widget->clip_col = ctk_helper_max(parent->clip_col, widget->col);
	
	widget->clip_width = ctk_helper_min(
			parent->clip_col + parent->clip_width - widget->clip_col,
			widget->col      + widget->width      - widget->clip_col);

	widget->clip_height = ctk_helper_min(
			parent->clip_row + parent->clip_height - widget->clip_row,
			widget->row      + widget->height      - widget->clip_row);
	
	// FIXME: Stupid special cases; I feel unclean.
	if (((CtkObject*)parent)->type == CtkTypeScrolledWindow)
	{
		CtkScrolledWindow* sw = CTK_SCROLLED_WINDOW(parent);
		if (parent->clip_height == widget->clip_height)
			widget->clip_height -= sw->hscrollbar_shown;
		if (parent->clip_width == widget->clip_width)
			widget->clip_width  -= sw->vscrollbar_shown;
	}
	if (((CtkObject*)parent)->type == CtkTypeOptionMenu)
	{
	      CtkOptionMenu* om = CTK_OPTION_MENU(parent);
	      CtkMenu*       menu;

	      if (om->menu)
	      {
		    menu = CTK_MENU(om->menu);
		    
		    widget->clip_height = ctk_helper_min(widget->clip_height,
			  ctk_table_get_row_row(CTK_TABLE(menu), menu->selected_item+2) -
			  ctk_table_get_row_row(CTK_TABLE(menu), menu->selected_item+1));
	      }
	}
}						

void ctk_shadow_xy(CtkWidget* widget, gint x, gint y)
{
	if (!ctk_check_clipping_xy(widget, x, y)) 
		return;
			
	arr_set_col_xy(x, y, FG_WHITE | BG_BLACK);
}

void ctk_draw_char(CtkWidget* widget, gint x, gint y, gchar ch)
{
	gint row = y;
	gint col = x;
	
	if (ctk_check_clipping_xy(widget, col, row))
		arr_print_xy(ch, col, row);
}

void ctk_draw_char_acs(CtkWidget* widget, gint x, gint y, gint ch)
{
	gint row = y;
	gint col = x;

	if (ctk_check_clipping_xy(widget, col, row)) {
		arr_print_acs_xy(ch, col, row);
	}
}

void ctk_draw_vline(CtkWidget* widget, gint x, gint y1, gint y2)
{
	gint i;
	
	for (i = y1; i <= y2; i++)
		ctk_draw_char_acs(widget, x, i, TC_VERTLINE);
}

void ctk_draw_hline(CtkWidget* widget, gint y, gint x1, gint x2)
{
	gint i;
	
	for (i = x1; i <= x2; i++)
		ctk_draw_char_acs(widget, i, y, TC_HORZLINE);
}

void ctk_draw_frame(CtkWidget* widget, gint x1, gint y1, gint x2, gint y2)
{
	ctk_draw_char_acs(widget, x1, y1, TC_ULCORNER);
	ctk_draw_char_acs(widget, x1, y2, TC_LLCORNER);
	ctk_draw_char_acs(widget, x2, y2, TC_LRCORNER);
	ctk_draw_char_acs(widget, x2, y1, TC_URCORNER);
	
	ctk_draw_hline(widget, y1, x1+1, x2-1);
	ctk_draw_hline(widget, y2, x1+1, x2-1);
	ctk_draw_vline(widget, x1, y1+1, y2-1);
	ctk_draw_vline(widget, x2, y1+1, y2-1);
}

/* Draw Window or Frame */
void ctk_draw_window(CtkWidget *widget)
{
	CtkWindow *window = NULL;
	CtkFrame *frame = NULL;
	CtkWidget *parent_widget = NULL;
	gint title_len = 0;	
	/* Temp Iteration variables */
	gint i,j;
	gchar *ptr;
	gint col;
	
	if (!widget)
	{
		ctk_close();
		g_error("ctk_draw_window: CtkWidget *widget == NULL!");
	}

	if (((CtkObject *)widget)->type == CtkTypeWindow)
	    window = (CtkWindow *)widget;
	else
	    frame = (CtkFrame *)widget;

	/* If just a frame, then get the parent color! */

	parent_widget = ctk_color_get_parent_widget(widget);

	if (parent_widget)
	{
		widget->main_col = parent_widget->main_col;
		widget->title_col = parent_widget->main_col;
	}

	/* Print the top of the window */
	if (widget->inversed)
		arr_set_col(widget->inverse_col);
	else
		arr_set_col(widget->main_col);	

	for (i = 0; i < widget->width-2; i ++)
	{
		if (i == 0)
			ctk_draw_char_acs(widget, widget->col+i, widget->row,
					TC_ULCORNER);
		else
			ctk_draw_char_acs(widget, widget->col+i, widget->row,
					TC_HORZLINE);
	}

	/* Check if the Closable char should be printed */
	if (window && (window->OPTIONS_MASK & WINMASK_CLOSABLE) != 0)
		ctk_draw_char(widget, widget->col+i, widget->row, 'x');
	else 
	{
		ctk_draw_char_acs(widget, widget->col+i, widget->row, TC_HORZLINE);
	}
	
	ctk_draw_char_acs(widget, widget->col+i+1, widget->row, TC_URCORNER);
	
	for (i = 1; i < widget->height-1; i += 1)
	{				
		for (j = 0; j < widget->width; j += 1)
		{
			if (widget->inversed)
				arr_set_col(widget->inverse_col);
			else
				arr_set_col(widget->main_col);
				
			if (j == 0 || j == widget->width-1)
			{
				ctk_draw_char_acs(widget, widget->col+j, widget->row+i, TC_VERTLINE);
			}
			else
			{
				ctk_draw_char(widget, widget->col+j, widget->row+i, ' ');
			}
		}

		/* Print right hand shadow if turned on */
		if (window && window->shadow)
		{
			ctk_shadow_xy(widget, widget->col+j, widget->row+i);
			ctk_shadow_xy(widget, widget->col+j+1, widget->row+i);
		}
	}

	/* Print Bottom of Window */

	if (widget->inversed)
		arr_set_col(widget->inverse_col);
	else
		arr_set_col(widget->main_col);
	
	for (i = 0; i < widget->width-2; i += 1)
	{
		if (i == 0)
			ctk_draw_char_acs(widget, widget->col+i,
				widget->row+widget->height-1,
				TC_LLCORNER);
		else
			ctk_draw_char_acs(widget, widget->col+i,
				widget->row+widget->height-1,
				TC_HORZLINE);
	}

	/* Check if the resize char should be printed */
	if (window && (window->OPTIONS_MASK & WINMASK_RESIZABLE) != 0)
	{
		ctk_draw_char(widget, widget->col+i, widget->row+widget->height-1, ' ');
		ctk_draw_char_acs(widget, widget->col+i+1, widget->row+widget->height-1, TC_DIAMOND);
	}
	else
	{
		ctk_draw_char_acs(widget, widget->col+i, widget->row+widget->height-1, 
			TC_HORZLINE);
		ctk_draw_char_acs(widget, widget->col+i+1, widget->row+widget->height-1, 
			TC_LRCORNER);
	}
	
	/* Print Bottom Shadow */
	if (window && window->shadow)
	{
		ctk_shadow_xy(widget, widget->col+i+2, widget->row+widget->height-1);
		ctk_shadow_xy(widget, widget->col+i+3, widget->row+widget->height-1);
		
		for (i = 0; i < widget->width; i += 1)
			ctk_shadow_xy(widget, widget->col+2+i, widget->row+widget->height);
	}

	/* Print Window Title */
	
	if ((window && window->title) || (frame && frame->label))
	{
		if (window)
		    title_len = strlen(window->title)+2;
		else
		    title_len = strlen(frame->label)+2;
		
		if (widget->width - 5 < title_len)
		    title_len = widget->width-5;
		
		if (title_len < 0)
		    title_len = 0;

		arr_set_col(widget->title_col);

		if (title_len-2 >= 0)
		{
			if (window)
			{
				// FIXME: clipping... Hell, rewrite window drawing completely
				arr_goto_xy(widget->col + 2 + 
					(widget->width-4-title_len)/2, 
					widget->row);
				arr_print(' ');
				arr_strprint(window->title);
				arr_print(' ');
			}
			else
			{
				ptr = frame->label;
				col = (widget->col+2)+(widget->width-title_len+1)/2;
				for (i = 0; i < title_len; i += 1)
				{
					if (ctk_check_clipping_xy(widget, col+i, widget->row))
					{
						arr_goto_xy(col+i, widget->row);
						if (*ptr)
						    arr_print(*ptr);
					}
					if (*ptr)
					    ptr += 1;
				}
			}
		}
	}
}

/* Label.
 * This widget is fun, because you have to force the color
 * to be what the next relevant parent's color is.
 */
void ctk_draw_label(CtkWidget* widget)
{
	CtkLabel *label;
	CtkMisc *misc;
	CtkWidget *parent_widget = NULL;
	gint i;
	gint j;
	gint text_len;
	gint row;
	gint col;
	gint num_lines;
	gint max_width;
	gchar** text_array;
	
	if (!widget) {
		ctk_close();
		g_error("ctk_draw_label: CtkWidget *widget == NULL!");
	}

	label = (CtkLabel *)widget;
	misc = (CtkMisc *)widget;
	
	if (!label->text)
	    return;
	
	parent_widget = ctk_color_get_parent_widget(widget);
	
	if (parent_widget)
	{
		if (widget->inversed)
		{
			arr_set_col(parent_widget->inverse_col);
		}
		else if (widget->selected)
		{
			arr_set_col(parent_widget->selected_col);
		}
		else if (parent_widget->inversed &&
			CTK_OBJECT(parent_widget)->type != CtkTypeWindow &&
			CTK_OBJECT(parent_widget)->type != CtkTypeNotebook)
		{
			if (CTK_OBJECT(parent_widget)->type == CtkTypeToggleButton &&
				CTK_TOGGLE_BUTTON(parent_widget)->active)
			{
				arr_set_col(parent_widget->inv_selected_col);
			}
			else
			{
				arr_set_col(parent_widget->inverse_col);
			}
		}
		else if (CTK_OBJECT(parent_widget)->type == CtkTypeToggleButton &&
				CTK_TOGGLE_BUTTON(parent_widget)->active)
		{
			arr_set_col(parent_widget->selected_col);
		}
		else if (!parent_widget->sensitive)
			arr_set_col(parent_widget->insensitive_col);
		else
			arr_set_col(parent_widget->main_col);
	}
	else
	{
		arr_set_col(widget->main_col);
	}

	text_array = label->text_split;
	if (!text_array) 
	      return;

	num_lines = 0;
	max_width = 0;
	for (num_lines = 0; text_array[num_lines]; num_lines++) 
	{
		text_len = strlen(text_array[num_lines]);
		if (text_len > max_width)
		    max_width = text_len;
	}

	row = widget->row + (widget->height - (misc->ypad*2 + num_lines)) * misc->yalign;
	row += misc->ypad;
	col = widget->col + (widget->width  - (misc->xpad*2 + max_width)) * misc->xalign;
	col += misc->xpad;

	if (col < widget->col)
	    col = widget->col;
	if (row < widget->row)
	    row = widget->row;
	
	for (i = 0; i < widget->height; i++)
	{
		if (i >= row - widget->row && i < row + num_lines - widget->row)
			text_len = strlen(text_array[i - row + widget->row]);
		else
			text_len = 0;
			
		for (j = 0; j < widget->width; j ++)
		{
			if (!ctk_check_clipping_xy(widget, widget->col+j, widget->row+i))
			    continue;
			arr_goto_xy(widget->col+j, widget->row+i);
			
			if (text_len && j >= col - widget->col && j < col - widget->col + text_len)
			{
				arr_print(text_array[i - row + widget->row][j - col + widget->col]);
			}
			else
			{
				arr_print(' ');
			}
		}
	}
}

/* Button and Toggle Button */
void ctk_draw_button(CtkWidget* widget)
{
	CtkButton *button;
	CtkToggleButton *togglebutton = NULL;
	/* Temp Iteration variables */
	gint i,j;
	gint color;
	
	if (!widget) {
		ctk_close();
		g_error("ctk_draw_button: CtkWidget *widget == NULL!");
	}

	button = CTK_BUTTON(widget);
	
	if (((CtkObject *)widget)->type == CtkTypeToggleButton) {
		togglebutton = CTK_TOGGLE_BUTTON(widget);
/*		ctk_close();
		printf("MW: %d\n",widget->min_width);
		printf("MH: %d\n",widget->min_height);
		exit(0); */
	}

	if (togglebutton && togglebutton->active && widget->sensitive) {
		if (widget->inversed) {
			color = widget->inv_selected_col;
		} else {
			color = widget->selected_col;
		}
	} else if (widget->inversed) {
		color = widget->inverse_col;
	} else if (!widget->sensitive) {
		color = widget->insensitive_col;
	} else {
		color = widget->main_col;
	}

	arr_set_col(color);

	for (i = 0; i < widget->height; i += 1) {

		if (ctk_check_clipping_xy(widget, widget->col, widget->row+i)) {
		
			arr_goto_xy(widget->col, widget->row+i);
			
			if (widget->height > 1 && i == 0)
			    arr_print_acs(TC_ULCORNER);
			else if (widget->height > 1 && i == widget->height-1)
			    arr_print_acs(TC_LLCORNER);
			else if (widget->height > 1 && i > 0)
			    arr_print_acs(TC_VERTLINE);
			else if (widget->height == 1) {
				arr_print('[');
			}
		}
		    
		for (j = 0; j < widget->width-2; j += 1) {
			
			if (!ctk_check_clipping_xy(widget, widget->col+j+1, widget->row+i))
			    continue;
						
			arr_goto_xy(widget->col+j+1, widget->row+i);
			
			if ((widget->height > 2 && i == 0) ||
			    (widget->height > 2 && i == widget->height-1))
			    arr_print_acs(TC_HORZLINE);
			else {
				arr_set_col(color);
				arr_print(' ');
/*				if (togglebutton && togglebutton->active)
				    arr_print(' ');
				else if (!widget->sensitive) {
					arr_set_col(widget->insensitive_col);
					arr_print(' ');
				} else if (widget->inversed) {
					arr_set_col(widget->inverse_col);
					arr_print(' ');
				} else {
					arr_set_col(widget->main_col);
					arr_print(' ');
				} */
			}
		}

		if (ctk_check_clipping_xy(widget, widget->col+j+1,
					widget->row+i)) {
			arr_goto_xy(widget->col+j+1, widget->row+i);
			
			if (widget->height > 1 && i == 0)
			    arr_print_acs(TC_URCORNER);
			else if (widget->height > 1 && i == widget->height-1)
			    arr_print_acs(TC_LRCORNER);
			else if (widget->height > 1 && i > 0)
			    arr_print_acs(TC_VERTLINE);
			else if (widget->height == 1) {
				arr_print(']');
			}
		}
	}
}

/* Draw Check Button */
void ctk_draw_check_button(CtkWidget* widget)
{
	CtkToggleButton *togglebutton;
	gint row;
	CtkWidget *parent_widget;
	gint p_color;
	
	if (!widget) {
		ctk_close();
		g_error("ctk_draw_check_button: CtkWidget *widget == NULL!");
	}

	parent_widget = ctk_color_get_parent_widget(widget);

	if (parent_widget) {
		if (parent_widget->inversed)
		    p_color = parent_widget->inverse_col;
		else if (!parent_widget->sensitive)
		    p_color = parent_widget->insensitive_col;
		else
		    p_color =parent_widget->main_col;
	} else
	    p_color = widget->main_col;
	
	
	togglebutton = ((CtkToggleButton *)widget);
	
	row = widget->row+((widget->height-1)/2);
	
	arr_goto_xy(widget->col, row);
	
	if (widget->inversed)
	    arr_set_col(widget->inverse_col);
	else if (!widget->sensitive)
	    arr_set_col(widget->insensitive_col);
	else
	    arr_set_col(widget->main_col);

	ctk_draw_char(widget, widget->col+0, row, '[');
       	ctk_draw_char(widget, widget->col+1, row, togglebutton->active?'X':' ');
	ctk_draw_char(widget, widget->col+2, row, ']');
}

/* Draw Radio Button */
void ctk_draw_radio_button(CtkWidget* widget)
{
	CtkToggleButton *togglebutton;
	gint row;
	gint p_color;
	
	if (!widget) {
		ctk_close();
		g_error("ctk_draw_radio_button: CtkWidget *widget == NULL!");
	}

	if (widget->inversed)
	    p_color = widget->inverse_col;
	else if (!widget->sensitive)
	    p_color = widget->insensitive_col;
	else
	    p_color = widget->main_col;
	
	togglebutton = ((CtkToggleButton *)widget);

	arr_set_col(p_color);
	
	row = widget->row+((widget->height-1)/2);

	ctk_draw_char(widget, widget->col+0, row, '<');
       	ctk_draw_char(widget, widget->col+1, row, togglebutton->active?'*':' ');
	ctk_draw_char(widget, widget->col+2, row, '>');
}

/* Display the hbox corners */
void
    ctk_draw_hbox(CtkWidget *widget)
{
	arr_print_xy('A', widget->col, widget->row);
	arr_print_xy('B', widget->col+(widget->width-1), widget->row);
	arr_print_xy('C', widget->col, widget->row+(widget->height-1));
	arr_print_xy('D', widget->col+(widget->width-1),widget->row +
			(widget->height-1));
}

/* Display an Editable Area */
void ctk_draw_editable(CtkWidget *widget, gint width, gint height)
{
	CtkEditable *editable;
	gchar *ptr;
	gint i;

	if (!widget) {
		ctk_close();
		g_error("ctk_draw_editable: CtkWidget *widget == NULL!");
	}

	editable = CTK_EDITABLE(widget);

	ptr = editable->text_start;
	if (widget->sensitive)
	    arr_set_col(widget->main_col);
	else
	    arr_set_col(widget->insensitive_col);
	
	for (i = 0; i < width; i += 1) {
		arr_goto_xy(widget->col+i, widget->row);

		if (!ptr || *ptr == 0) {
			ctk_draw_char(widget, widget->col+i, widget->row, ' ');
		} else {
			ctk_draw_char(widget, widget->col+i, widget->row, 
				editable->visible ? *ptr : '*');
			ptr += 1;
		}
	}
}

/* Display a Entry widget */
void ctk_draw_entry(CtkWidget* widget)
{
	ctk_draw_editable(widget, widget->width, widget->height);
}

/* Display a Entry widget */
void ctk_draw_spin_button(CtkWidget* widget)
{	
	ctk_draw_editable(widget, widget->width-2, widget->height);
	
	arr_set_col(widget->title_col);

	if (ctk_check_clipping_xy(widget, widget->col+widget->width-2,
			widget->row))
		arr_print_acs_xy(TC_UARROW, widget->col+widget->width-2,
				widget->row);
	if (ctk_check_clipping_xy(widget, widget->col+widget->width-1,
			widget->row)) {
		arr_print_acs_xy(TC_DARROW, widget->col+widget->width-1,
				widget->row);
	}
}

/* Draw Scrolled Window */
void ctk_draw_scrolled_window(CtkWidget* widget)
{	
	gint i;
	CtkScrolledWindow *scrolledwindow;
	CtkWidget *child = NULL;
	gint hslider;
	gint vslider;
	gint hslider_col;
	gint vslider_row;
	gint vupper=0;
	gint hupper=0;

	/* Figure out the slider part of the horiz and vert scrollbar */
	
	scrolledwindow = ((CtkScrolledWindow *)widget);
	
	if (!scrolledwindow->hadjustment || !scrolledwindow->vadjustment)
	    return;
	
	if (widget->node->children)
	    child = ((CtkWidget *)widget->node->children->data);

	if (!scrolledwindow->vscrollbar_shown && !scrolledwindow->hscrollbar_shown)
	    return;
	
	if (child) {
		child = ((CtkWidget *)widget->node->children->data);
		if (!scrolledwindow->vscrollbar_shown)
		    hupper = child->width-widget->width;
		else
		    hupper = child->width-widget->width+1;
		
		if (!scrolledwindow->hscrollbar_shown)
		    vupper = child->height-widget->height;
		else
		    vupper = child->height-widget->height+1;

		if (child->width-1 > hupper && hupper > 0)
		    hslider = (widget->width-3) * ((float)(widget->width-3) /
						   (float)(child->width));
		else
		    hslider = widget->width - 3;

		if (child->height-1 > vupper && vupper > 0)
		    vslider = widget->height-3 * ((float)child->height /
						  (float)(widget->height-3));
		else
		    vslider = widget->height - 3;
	} else {
		hslider = widget->width-3;
		vslider = widget->height-3;
	}

	if (hslider > widget->width-3)
	    hslider = widget->width-3;
	
	if (vslider > widget->height-3)
	    vslider = widget->height-3;

	if (hslider < 0)
	    hslider = 0;
	
	if (vslider < 0)
	    vslider = 0;

	
	hslider_col = scrolledwindow->hadjustment->value * 
	    ((float)((widget->width-2-scrolledwindow->vscrollbar_shown)
	    	-(hslider+1)) / (float)hupper);	
	
	hslider_col += widget->col+1;
	
	if (hslider_col < widget->col+1)
	    hslider_col = widget->col+1;
	
	vslider_row = scrolledwindow->vadjustment->value * 
	    ((float)((widget->height-2-scrolledwindow->hscrollbar_shown)
	    	-(vslider+1)) / (float)vupper);
	
	vslider_row += widget->row+1;
	
	if (vslider_row < widget->row+1)
	    vslider_row = widget->row+1;
	
	scrolledwindow->hadjustment->slider_len = hslider;
	scrolledwindow->hadjustment->slider_col = hslider_col;
	scrolledwindow->vadjustment->slider_len = vslider;
	scrolledwindow->vadjustment->slider_row = vslider_row;

	if (widget->inversed)
		arr_set_col(widget->inverse_col);
	else
		arr_set_col(widget->main_col);

	/* Verticle scroll bar */
	for (i = 0; scrolledwindow->vscrollbar_shown && i < widget->height-scrolledwindow->hscrollbar_shown; i += 1) {		
		arr_goto_xy(widget->col+widget->width-1, widget->row+i);
		if (ctk_check_clipping_xy(widget, widget->col+widget->width-1,
				widget->row + i)) {
			if (i == 0)
			    arr_print_acs(TC_UARROW);
			else if (i == widget->height-1-scrolledwindow->hscrollbar_shown)
			    arr_print_acs(TC_DARROW);
			else {
				if (widget->row+i >= vslider_row &&
				    widget->row+i <= vslider_row+vslider)
				    arr_print_acs(TC_BLOCK);
				else
				    arr_print_acs(TC_CHKBOARD);
			}
		}
	}

	/* Horizontal Scroll Bar */
	for (i = 0; scrolledwindow->hscrollbar_shown && i < widget->width-scrolledwindow->vscrollbar_shown; i += 1) {
		arr_goto_xy(widget->col+i,widget->row+widget->height-1);

		if (ctk_check_clipping_xy(widget, widget->col+i, widget->row+widget->height-1)) {
			if (i == 0)
			    arr_print_acs(TC_LARROW);
			else if (i == widget->width-1-scrolledwindow->vscrollbar_shown)
			    arr_print_acs(TC_RARROW);
			else {
				if (widget->col+i >= hslider_col &&
				    widget->col+i <= hslider_col+hslider)
				    arr_print_acs(TC_BLOCK);
				else
				    arr_print_acs(TC_CHKBOARD);
			}
		}
	}	
}

/* Draw notebook */
void ctk_draw_notebook(CtkWidget* widget)
{
	CtkNotebook* notebook;
	GSList*      tab_list;
	gint         accum_x;
	gint         accum_y;
	
	notebook = CTK_NOTEBOOK(widget);
	
	arr_set_col(widget->main_col);

	/* Draw the border if it has one */
	if (CTK_CONTAINER(widget)->border_width)
	{
		ctk_draw_frame(widget, 
			notebook->page_col - 1,
			notebook->page_row - 1,
			notebook->page_col + notebook->page_width,
			notebook->page_row + notebook->page_height);
	}
	
	/* Draw the bars inbetween the labels */
	accum_x = 0;
	accum_y = 0;
	for (tab_list = notebook->tab_labels; tab_list; tab_list = tab_list->next)
	{
		CtkNotebookTab* tab = (CtkNotebookTab*)tab_list->data;
		
		if (notebook->show_tabs && tab_list->next && 
			tab->label && tab->label->show)
		{
			switch (notebook->tab_pos)
			{
				case CTK_POS_TOP:
				case CTK_POS_BOTTOM:
					accum_x += tab->label->width + 1;
					ctk_draw_vline(widget, 
						notebook->tab_col + accum_x, 
						notebook->tab_row,
						notebook->tab_row + notebook->tab_height - 1);
					accum_x += 2;
					break;
				case CTK_POS_LEFT:
				case CTK_POS_RIGHT:
					accum_y += tab->label->height;
					ctk_draw_hline(widget, 
						notebook->tab_row + accum_y, 
						notebook->tab_col,
						notebook->tab_col + notebook->tab_width - 1);
					accum_y++;
					break;
			}
		}
	}
}

/* Draw Combo Widget */
void ctk_draw_combo(CtkWidget* widget)
{
	if (widget->inversed)
		arr_set_col(widget->inverse_col);
	else
		arr_set_col(widget->main_col);
		
	ctk_draw_char_acs(widget, widget->col+widget->width-1, widget->row,
			TC_DARROW);
}

/* Draw HSeparator */
void ctk_draw_hseparator(CtkWidget* widget)
{
	CtkWidget *parent_widget;
	gint i;
	gint row;

	parent_widget = ctk_color_get_parent_widget(widget);

	if (parent_widget)
	    arr_set_col(parent_widget->main_col);
	else
	    arr_set_col(widget->main_col);
	
	if (widget->height > 1) {
		row = widget->row+((widget->height-1)/2);
		if (row > widget->row+widget->height-1)
		    row = widget->row;
	} else
	    row = widget->row;

	for (i = 0; i < widget->width; i += 1) {
		if (!ctk_check_clipping_xy(widget, widget->col+i, row))
		    continue;
		arr_print_acs_xy(TC_HORZLINE, widget->col+i, row);
	}
} 

/* Draw VSeparator */
void ctk_draw_vseparator(CtkWidget* widget)
{
	CtkWidget *parent_widget;
	gint i;
	gint col;

	parent_widget = ctk_color_get_parent_widget(widget);
	
	if (parent_widget)
	    arr_set_col(parent_widget->main_col);
	else
	    arr_set_col(widget->main_col);

	if (widget->width > 1) {
		col = widget->col+((widget->width-1)/2);
		if (col > widget->col+widget->width-1)
		    col = widget->col;
	} else
	    col = widget->col;
	
	for (i = 0; i < widget->height; i += 1) {
		if (!ctk_check_clipping_xy(widget, col, widget->row+i))
		    continue;
		arr_print_acs_xy(TC_VERTLINE, col, widget->row+i);
	}
} 

/* Draw Progress bar */
void ctk_draw_progress_bar(CtkWidget* widget)
{
	gint i,j;
	CtkProgress *progress;
	gchar percent[20];
	gint color;
	gint len;
	gint cutoff;
	gint start;
	
	progress = CTK_PROGRESS(widget);

	if (progress->show_text)
		sprintf(percent,"%2.0f%%",(progress->value/progress->max)*100);
	else
		strcpy(percent, "");
	
	/* no text if no max */
	if (!progress->max)
		percent[0] = 0;
	
	len = strlen(percent);
	if (progress->max)
		cutoff = progress->value * widget->width / progress->max;
	else
		cutoff = 0;
	start = (widget->width - len) / 2;
	
	/* Draw Background */
	for (i = 0; i < widget->height; i++) 
	{
		for (j = 0; j < widget->width; j++) 
		{
			if (j < cutoff)
			    color = widget->inverse_col;
			else
			    color = widget->main_col;
			arr_set_col(color);
			
			if (i == widget->height / 2)
			{
				if (j >= start && j < start + len)
				{
					ctk_draw_char(widget, widget->col+j, widget->row+i, 
						percent[j - start]);
				}
				else
				{
					ctk_draw_char(widget, widget->col+j, widget->row+i, ' ');
				}
			}
			else
			{
				ctk_draw_char(widget, widget->col+j, widget->row+i, ' ');
			}
		}
	}
}

/* Draw HScale */
void ctk_draw_hscale(CtkWidget* widget)
{
	gint i,j;
	gint col,row;
	CtkAdjustment *adjustment;
	CtkWidget *parent_widget;
	gint color=-1;
	gchar *value;
	
	parent_widget = ctk_color_get_parent_widget(widget);

	if (parent_widget) {
		color = parent_widget->main_col;
	} else {
		color = widget->main_col;
	}

	/* color */
	arr_set_col(widget->main_col);
	
	if (widget->height > 2) {
		row = widget->row+(widget->height/2);
		if (row < widget->row)
		    row = widget->row;
		else if (row+1 > widget->row+widget->height-1)
		    row = widget->row+widget->height-2;
	} else
	    row = widget->row;

	/* Draw Background */
	
	for (i = 0; i < 2; i += 1) {
		for (j = 0; j < widget->width; j += 1) {
			if (!ctk_check_clipping_xy(widget,widget->col+j,row+i))
			    continue;
			arr_goto_xy(widget->col+j, row+i);
			if (i == 1 && j == 0)
			    arr_print_acs(TC_LARROW);
			else if (i == 1 && j == widget->width-1)
			    arr_print_acs(TC_RARROW);
			else if (i == 1)
			    arr_print_acs(TC_HORZLINE);
			else {
				arr_set_col(widget->main_col);
				arr_print(' ');
			}
		}
	}

	row += 1;
	
	adjustment = ((CtkRange *)widget)->adjustment;
	
	/* Color */
	if (widget->inversed)
	    arr_set_col(widget->inverse_col);
	else
	    arr_set_col(widget->selected_col);

	/* col Position for scale marker */
	col = (gfloat)(widget->width-2) *
	    (gfloat)((adjustment->value-adjustment->lower)/
		     (adjustment->upper-adjustment->lower));
	col += widget->col+1;
	if (col <= widget->col)
	    col = widget->col + 1;
	else if (col >= widget->col+widget->width-2)
	    col = widget->col+widget->width-2;
	
	arr_goto_xy(col, row);
	if (ctk_check_clipping_xy(widget, col, row))
	    arr_print_acs(TC_DIAMOND);

	if (!((CtkScale *)widget)->draw_value)
	    return;
	
	arr_set_col(color);	

	value = ctk_util_ftos(adjustment->value,((CtkRange *)widget)->digits);

	col -= strlen(value)/2;

	if (col+strlen(value) > widget->col+widget->width-1)
	    col = widget->col+widget->width-strlen(value);
	else if (col < widget->col)
	    col = widget->col;

	row -= 1;
	for (i = 0; i < strlen(value); i += 1) {
		if (ctk_check_clipping_xy(widget, col+i, row)) {
			arr_goto_xy(col+i, row);
			if (*(value+i))
			    arr_print(*(value+i));
		}
	}

}

/* Draw VScale */
void ctk_draw_vscale(CtkWidget* widget)
{
	gint i,j;
	gint col,row;
	CtkAdjustment *adjustment;
	gint color=-1;
	gchar *value;
	CtkWidget *parent_widget;

	parent_widget = ctk_color_get_parent_widget(widget);

	/* color */
	if (parent_widget) {
		color = parent_widget->main_col;
	} else {
		color = widget->main_col;
	}
	
	/* Get the col */
	col = widget->col+(widget->width/2);
	if (col < widget->col)
	    col = widget->col;
	else if (col > widget->col+(widget->width-1))
	    col = widget->col+(widget->width-1);

	/* Draw Background */

	for (i = 0; i < widget->height; i += 1) {
		for (j = 0; j < widget->width; j += 1) {
			if (!ctk_check_clipping_xy(widget, widget->col+j,
						widget->row + i))
			    continue;			
			arr_goto_xy(widget->col+j, widget->row + i);
			/* Top Arrow */
			if (i == 1 && widget->col+j == col)
			    arr_print_acs(TC_UARROW);
			/* Bottom Arrow */
			else if (i == widget->height-1 && widget->col+j == col)
			    arr_print_acs(TC_DARROW);
			else if (i > 0 && widget->col+j == col)
			    arr_print_acs(TC_VERTLINE);
			else {
				arr_set_col(widget->main_col);
				arr_print(' ');
			}
		}
	}

	adjustment = ((CtkRange *)widget)->adjustment;
	
	/* Color */
	if (widget->inversed)
	    arr_set_col(widget->inverse_col);
	else
	    arr_set_col(widget->selected_col);

	/* row Position for scale marker */
	row = (gfloat)(widget->height-3) *
	    (gfloat)((adjustment->value-adjustment->lower)/
		     (adjustment->upper-adjustment->lower));
	row += widget->row+2;
	if (row <= widget->row+2)
	    row = widget->row + 2;
	else if (row >= widget->row+widget->height-2)
	    row = widget->row+widget->height-2;
	
	arr_goto_xy(col, row);

	if (ctk_check_clipping_xy(widget, col, row))
	    arr_print_acs(TC_DIAMOND);

	arr_set_col(color);
	
	value = ctk_util_ftos(adjustment->value,((CtkRange *)widget)->digits);

	col -= (gfloat)strlen(value)/(gfloat)2;
	
	for (i = 0; i < strlen(value); i += 1) {
		if (ctk_check_clipping_xy(widget, col+i, widget->row)) {
			arr_goto_xy(col+i, widget->row);
			if (*(value+i))
			    arr_print(*(value+i));
		}
	}
}

/* Traverse Draw Widget Function */
gint node_draw_widget(GNode* node, gpointer data)
{
	CtkWidget* widget;
	
	widget = CTK_WIDGET(node->data);
	
	if (node->parent)
		ctk_widget_intersect(CTK_WIDGET(node->parent->data), widget);

	if (ctk_clipping_trivial_rejection(widget))
		return FALSE;
	
	switch(((CtkObject *)widget)->type) {
	 case CtkTypeWindow:
	 case CtkTypeFrame:
		ctk_draw_window(widget);
		break;
	 case CtkTypeLabel:
		ctk_draw_label(widget);
		break;
	 case CtkTypeToggleButton:
	 case CtkTypeButton:
		ctk_draw_button(widget);
		break;
	 case CtkTypeCheckButton:
		ctk_draw_check_button(widget);
		break;
	 case CtkTypeRadioButton:
		ctk_draw_radio_button(widget);
		break;
	 case CtkTypeEntry:
		ctk_draw_entry(widget);
		break;
	 case CtkTypeSpinButton:
		ctk_draw_spin_button(widget);
		break;
	 case CtkTypeScrolledWindow:
		ctk_draw_scrolled_window(widget);
		break;
	 case CtkTypeNotebook:
		ctk_draw_notebook(widget);
		break;
	 case CtkTypeOptionMenu:
	        // ctk_draw_option_menu(widget);
		ctk_draw_combo(widget);
		break;
	 case CtkTypeCombo:
		ctk_draw_combo(widget);
		break;
	 case CtkTypeHSeparator:
		ctk_draw_hseparator(widget);
		break;
	 case CtkTypeVSeparator:
		ctk_draw_vseparator(widget);
		break;
	 case CtkTypeProgressBar:
		ctk_draw_progress_bar(widget);
		break;
	 case CtkTypeHScale:
		ctk_draw_hscale(widget);
		break;
	 case CtkTypeVScale:
		ctk_draw_vscale(widget);
		break;		
	 default:
		break;
	}

	return FALSE;
}

void ctk_size_window(CtkWidget* widget, gboolean pass)
{
	/* Recursively find minimal size of widget */
 	ctk_size_set_minimum(widget);
	
	/* If the toplevel needs more space, grant it */
	if (widget->min_height > widget->height)
		widget->height = widget->min_height;
	
	if (widget->min_width > widget->width)
		widget->width = widget->min_width;
	
	if (CTK_CHECK_TYPE(widget, CtkTypeWindow) && 
		(CTK_WINDOW(widget)->OPTIONS_MASK & WINMASK_AUTOSHRINK) != 0)
	{
		widget->width  = widget->min_width;
		widget->height = widget->min_height;
	}
	
	if (pass)
	{
	      /* Center Window Here */
	      if (CTK_CHECK_TYPE(widget, CtkTypeWindow) &&
		  (CTK_WINDOW(widget)->OPTIONS_MASK & WINMASK_AUTOCENTER) != 0)
	      {
		    widget->col = (ctk_scr_c-widget->width +1)/2;
		    widget->row = (ctk_scr_r-widget->height+1)/2;
	      }
	      
	      /* Make sure window is not pushed off the screen */
	      if (widget->col + widget->width > ctk_scr_c)
		    widget->col = ctk_scr_c - widget->width;
	      
	      if (widget->row + widget->height > ctk_scr_r)
		    widget->row = ctk_scr_r - widget->height;
	
	      if (widget->row < 0) widget->row = 0;
	      if (widget->col < 0) widget->col = 0;
	}

	/* Recursively set coords of widget */
	ctk_size_set_real(widget);
}

/* Display Window */
void display_window(CtkWidget* widget, CtkRedraw rdtype)
{
      ctk_label_wrap_hack_report_vertible = FALSE;
      ctk_size_window(widget, FALSE);
      ctk_label_wrap_hack_report_vertible = TRUE;
      ctk_size_window(widget, TRUE);

      /* Set clipping on the window to the window */
      widget->clip_col    = ctk_helper_max(0, widget->col);
      widget->clip_row    = ctk_helper_max(0, widget->row);
      widget->clip_height = ctk_helper_min(ctk_scr_r-widget->clip_row, widget->height+2);
      widget->clip_width  = ctk_helper_min(ctk_scr_c-widget->clip_col, widget->width +2);
      
      /* Start drawing */
      g_node_traverse(widget->node,
		      G_LEVEL_ORDER,
		      G_TRAVERSE_ALL,
		      -1,
		      node_draw_widget,
		      NULL);
}

/* Redraw the Screen */
void ctk_redraw_screen(CtkRedraw rdtype)
{
      gint i;
      
      if (ctk_clean_redraw)
             rdtype = CTK_REDRAW_ALL;

      /* Paint blue background */
      arr_nprint_xy_col(' ', 0, 0, (FG_WHITE | BG_BLUE), 
            arr_scr_x() * arr_scr_y());
    
      if (focus_stack == NULL || !ctk_main_loops || !g_main_is_running(ctk_main_loops->data))
	    return;
      
      for (i = stack_num_items(focus_stack)-1; i >= 0; i -= 1)
      {
	    CtkWidget* widget = CTK_WIDGET(focus_stack[i]->data);

	    if (widget->show)
	    {
		  display_window(widget, rdtype);
	    }
	    else
	    {
		  widget->clip_height = 0;
		  widget->clip_width  = 0;
	    }
      }
      
      switch (rdtype)
      {
      case CTK_REDRAW_ALL:
	    arr_redraw();
	    break;
      case CTK_REDRAW_CHANGED:
	    arr_dump_screen();
	    break;
      default:
	    ctk_assert_not_reached("Invalid input to ctk_redraw_screen");
      }
      
      if (ctk_mouse_draw)
      {
            ctk_inverse_char(ctk_mouse_x, ctk_mouse_y);
            arr_dump_screen();
      }
}
 
void ctk_draw_mark_changed(CtkWidget* widget)
{
      /* See notes (on paper) for a description of future use */
      ctk_screen_dirty = TRUE;
}
