/* 
 * $Id: ctkio.c,v 1.11 2000/07/27 03:23:46 terpstra Exp $
 *
 * CTK - Console Toolkit
 *
 * Copyright (C) 1998-2000 Stormix Technologies Inc.
 *
 * License: LGPL
 *
 * Authors: Kevin Lindsay, Wesley Terpstra
 *  
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *    
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *    
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <stdio.h>
#include <glib.h>

#include "ctk.h"

typedef struct CtkIOClosure_S {
	CtkInputFunction function;
	CtkInputCondition condition;
	gpointer data;
} CtkIOClosure;

static gboolean
    ctk_io_invoke (GIOChannel *source,
		   GIOCondition condition,
		   gpointer data)
{
	CtkIOClosure *closure = data;
	CtkInputCondition ctk_cond = 0;
	
	if (condition && G_IO_IN)
	    ctk_cond |= CTK_INPUT_READ;
	else if (condition && G_IO_OUT)
	    ctk_cond |= CTK_INPUT_WRITE;
	else if (condition && G_IO_PRI)
	    ctk_cond |= CTK_INPUT_EXCEPTION;

	if (closure->condition && ctk_cond)
	    closure->function(closure->data, 
			      g_io_channel_unix_get_fd(source),
			      ctk_cond);

	return TRUE;
}

typedef struct CtkInputWrap_S
{
	gpointer func;
	gpointer data;
} CtkInputWrap;

gboolean ctk_input_wrap(gpointer data, gint source, CtkInputCondition condition)
{
	CtkInputWrap* wrap = data;
	gboolean      out;
	gboolean    (*func)(gpointer, gint, CtkInputCondition);
	
	func = wrap->func;
	out = func(wrap->data, source, condition);
	
	/* If the screen is dirty, repaint it */
	if (ctk_screen_dirty)
	{
		ctk_redraw_screen(CTK_REDRAW_CHANGED);
		ctk_screen_dirty = FALSE;
	}
	
	/* Do actual release of all widgets after we reach the main loop
	 * where we are fairly certain that nothing will harm them.
	 */
	if (g_slist_length(ctk_main_loops) <= 1)
	{
		ctk_main_garbage_cleanup();
	}
	
	return out;
}

guint ctk_input_add (gint fd, CtkInputCondition condition, gpointer function, gpointer data)
{
	GIOChannel *new_channel;
	guint id;
	GIOCondition g_cond = 0;
	/* This has to be freeed somewhere still! */
	CtkIOClosure *closure = g_new(CtkIOClosure,1);
	CtkInputWrap* wrapper = g_malloc(sizeof(CtkInputWrap));

	if (fd < 0) {
		ctk_close();
		g_error("ctk_input_add: gint fd < 0 !");
	}

	if (condition != CTK_INPUT_READ && condition != CTK_INPUT_WRITE) {
		ctk_close();
		g_error("ctk_input_add: Invalid CtkInputCondition condition value!");
	}

	if (!function) {
		ctk_close();
		g_error("ctk_input_add: gpointer function == NULL!");
	}
	
	wrapper->func = function;
	wrapper->data = data;
	
	closure->function = (gpointer)&ctk_input_wrap;
	closure->condition = condition;
	closure->data = wrapper;

	if (condition & CTK_INPUT_READ)
	    g_cond |= G_IO_IN;
	else if (condition & CTK_INPUT_WRITE)
	    g_cond |= G_IO_OUT;
	else if (condition & CTK_INPUT_EXCEPTION)
	    g_cond |= G_IO_PRI;

	new_channel = g_io_channel_unix_new(fd);

	id = g_io_add_watch(new_channel,g_cond,ctk_io_invoke,closure);
	g_io_channel_unref(new_channel);

	return id;
}

void ctk_input_remove(guint tag)
{
	g_source_remove(tag);
}

gboolean ctk_timeout_wrap(gpointer data)
{
	CtkInputWrap* wrap = data;
	gboolean      out;
	gboolean    (*func)(gpointer);
	
	func = (gboolean(*)(gpointer))wrap->func;
	out = func(wrap->data);
	
	/* If the screen is dirty, repaint it */
	if (ctk_screen_dirty)
	{
		ctk_redraw_screen(CTK_REDRAW_CHANGED);
		ctk_screen_dirty = FALSE;
	}
	
	/* Do actual release of all widgets after we reach the main loop
	 * where we are fairly certain that nothing will harm them.
	 */
	if (g_slist_length(ctk_main_loops) <= 1)
	{
		ctk_main_garbage_cleanup();
	}
	
	return out;
}

guint ctk_timeout_add(guint interval, gpointer function, gpointer data)
{
	/* This has to be freeed somewhere still! */
	CtkInputWrap* wrapper = g_malloc(sizeof(CtkInputWrap));

	if (!function) {
		ctk_close();
		g_error("ctk_input_add: gpointer function == NULL!");
	}
	
	wrapper->func = function;
	wrapper->data = data;
	
	return g_timeout_add(interval, &ctk_timeout_wrap, wrapper);
}

void ctk_timeout_remove(guint tag)
{
	g_source_remove(tag);
}
