/**
 * @file libgalago/galago-account.c Galago Account API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-account.h>
#include <libgalago/galago-assert.h>
#include <libgalago/galago-context.h>
#include <libgalago/galago-core.h>
#include <libgalago/galago-hashtable.h>
#include <libgalago/galago-marshal.h>
#include <libgalago/galago-object-utils.h>
#include <libgalago/galago-utils.h>
#include <libgalago/galago-utils-priv.h>
#include <stdio.h>
#include <string.h>

struct _GalagoAccountPrivate
{
	GalagoService *service;
	GalagoPerson *person;
	GalagoPresence *presence;
	GalagoAvatar *avatar;
	char *username;
	char *display_name;

	galago_bool connected;

	GalagoHashTable *contacts_table;
	GalagoList *contacts;
};

static void _galago_dbus_account_set_presence(GalagoAccount *account,
                                              GalagoPresence *presence);
static void _galago_dbus_account_set_avatar(GalagoAccount *account,
                                            GalagoAvatar *avatar);
static void _galago_dbus_account_add_contact(GalagoAccount *account,
                                             GalagoAccount *contact);


/**************************************************************************
 * Object/Class support
 **************************************************************************/
GALAGO_REGISTER_CLASS(galago_account, GalagoAccount, NULL,
                      GALAGO_DBUS_ACCOUNT_INTERFACE);
DEPRECATE_OLD_GET_CLASS(galago_accounts, galago_account);

static void
galago_account_object_init(GalagoAccount *account)
{
	account->priv = galago_new0(GalagoAccountPrivate, 1);

	account->priv->contacts_table =
		galago_hash_table_new_full(galago_str_hash, galago_str_equal,
		                           free, NULL);
}

static void
galago_account_object_finalize(GalagoObject *object)
{
	GalagoAccount *account = (GalagoAccount *)object;
	GalagoService *service;
	GalagoPerson *person;

	galago_return_if_fail(account != NULL);

	service = galago_account_get_service(account);
	person  = galago_account_get_person(account);

	galago_service_remove_account(service, account);

	if (person != NULL)
		galago_person_remove_account(person, account);

	if (account->priv->presence != NULL)
	{
		GalagoPresence *presence = account->priv->presence;

		account->priv->presence = NULL;

		galago_object_unref(presence);
	}

	if (account->priv->avatar != NULL)
	{
		GalagoAvatar *avatar = account->priv->avatar;

		account->priv->avatar = NULL;

		galago_object_unref(avatar);
	}

	galago_hash_table_destroy(account->priv->contacts_table);
	galago_list_destroy(account->priv->contacts);

	if (account->priv->username != NULL)
		free(account->priv->username);

	free(account->priv);
}

static void
galago_account_dbus_message_append(DBusMessageIter *iter,
                                   const GalagoObject *object)
{
	GalagoAccount *account = (GalagoAccount *)object;
	const char *username;
	galago_bool connected;

	/* Service */
	galago_dbus_message_iter_append_object(iter,
		galago_account_get_service(account));

	/* Person */
	galago_dbus_message_iter_append_object(iter,
		galago_account_get_person(account));

	/* Username */
	username = galago_account_get_username(account);
	galago_dbus_message_iter_append_string(iter, username);

	/* Display name */
	galago_dbus_message_iter_append_string_or_nil(iter,
		galago_account_get_display_name(account));

	/* Connected? */
	connected = galago_account_is_connected(account);
	galago_dbus_message_iter_append_boolean(iter, connected);
}

static void *
galago_account_dbus_message_get(DBusMessageIter *iter)
{
	GalagoService *service = NULL;
	GalagoPerson *person = NULL;
	GalagoAccount *account;
	galago_bool connected;
	char *display_name;
	char *username = NULL;

	service = galago_dbus_message_iter_get_object(iter, GALAGO_CLASS_SERVICE);
	dbus_message_iter_next(iter);

	person = galago_dbus_message_iter_get_object(iter, GALAGO_CLASS_PERSON);
	dbus_message_iter_next(iter);

	galago_dbus_message_iter_get_string(iter, username);
	dbus_message_iter_next(iter);

	display_name = galago_dbus_message_iter_get_string_or_nil(iter);
	dbus_message_iter_next(iter);

	galago_dbus_message_iter_get_boolean(iter, connected);

	account = galago_account_new(service, person, username);
	galago_account_set_display_name(account, display_name);
	galago_account_set_connected(account, connected);

#if !GALAGO_CHECK_DBUS_VERSION(0, 30)
	dbus_free(username);

	if (display_name != NULL)
		dbus_free(display_name);
#endif

	return account;
}

static void
galago_account_dbus_push_full(GalagoObject *object)
{
	GalagoAccount *account, *contact;
	const GalagoList *l;

	account = (GalagoAccount *)object;

	_galago_dbus_account_set_presence(account,
		galago_account_get_presence(account, FALSE));
	_galago_dbus_account_set_avatar(account,
		galago_account_get_avatar(account, FALSE));

	for (l = galago_account_get_contacts(account, FALSE);
		 l != NULL;
		 l = l->next)
	{
		contact = (GalagoAccount *)l->data;

		_galago_dbus_account_add_contact(account, contact);
	}
}

static void
galago_account_class_init(GalagoObjectClass *klass)
{
	klass->finalize            = galago_account_object_finalize;
	klass->dbus_message_append = galago_account_dbus_message_append;
	klass->dbus_message_get    = galago_account_dbus_message_get;
	klass->dbus_push_full      = galago_account_dbus_push_full;

	galago_signal_register(klass->signal_context, "connected",
						   galago_marshal_VOID, 0);
	galago_signal_register(klass->signal_context, "disconnected",
						   galago_marshal_VOID, 0);
	galago_signal_register(klass->signal_context, "display-name-changed",
						   galago_marshal_VOID, 0);
	galago_signal_register(klass->signal_context, "presence-set",
						   galago_marshal_VOID__POINTER, 1,
						   GALAGO_TYPE_OBJECT);
	galago_signal_register(klass->signal_context, "avatar-set",
						   galago_marshal_VOID__POINTER, 1,
						   GALAGO_TYPE_OBJECT);
	galago_signal_register(klass->signal_context, "contact-added",
						   galago_marshal_VOID__POINTER, 1,
						   GALAGO_TYPE_OBJECT);
	galago_signal_register(klass->signal_context, "contact-removed",
						   galago_marshal_VOID__POINTER, 1,
						   GALAGO_TYPE_OBJECT);
	galago_signal_register(klass->signal_context, "updated",
						   galago_marshal_VOID, 0);
}

/**************************************************************************
 * GalagoAccount API
 **************************************************************************/
GalagoAccount *
galago_account_new(GalagoService *service, GalagoPerson *person,
                   const char *username)
{
	GalagoAccount *account;

	galago_return_val_if_fail(service != NULL,                       NULL);
	galago_return_val_if_fail(person  != NULL,                       NULL);
	galago_return_val_if_fail(GALAGO_IS_SERVICE(service),            NULL);
	galago_return_val_if_fail(GALAGO_IS_PERSON(person),              NULL);
	galago_return_val_if_fail(username != NULL && *username != '\0', NULL);
	galago_return_val_if_fail(galago_service_is_native(service) ==
							  galago_person_is_native(person),       NULL);

	account = galago_service_get_account(service, username, FALSE);

	if (account == NULL)
	{
		const char *obj_prefix;

		galago_context_push(galago_object_get_context(service));
		account = galago_object_new(GALAGO_CLASS_ACCOUNT);
		galago_context_pop();

		account->priv->service  = service;
		account->priv->person   = person;
		account->priv->username = strdup(username);

		if ((obj_prefix = galago_object_get_dbus_path(service)) != NULL)
		{
			size_t len;
			char *obj_path;
			const char *escaped_username;

			escaped_username = galago_dbus_normalize_name(username);

			len = strlen(obj_prefix) + strlen(escaped_username) + 2;

			obj_path = galago_new(char, len);
			snprintf(obj_path, len, "%s/%s", obj_prefix, escaped_username);

			galago_object_set_dbus_path(account, obj_path);

			free(obj_path);
		}

		galago_person_add_account(person, account);
		galago_service_add_account(service, account);
	}

	return account;
}

void
galago_account_set_connected(GalagoAccount *account, galago_bool connected)
{
	GalagoPresence *presence;
	GalagoObjectClass *klass;
	GalagoSignalContext *signal_context;

	galago_return_if_fail(account != NULL);
	galago_return_if_fail(GALAGO_IS_ACCOUNT(account));

	if (account->priv->connected == connected &&
		account->priv->presence != NULL)
	{
		return;
	}

	klass = GALAGO_OBJECT_CLASS(account);
	signal_context = galago_class_get_signal_context(klass);

	account->priv->connected = connected;

	galago_signal_context_freeze(signal_context);

	presence = galago_account_get_presence(account, FALSE);

	if (presence == NULL)
	{
		if (galago_account_is_native(account))
			presence = galago_presence_new(account);
	}

	if (presence != NULL)
	{
		galago_context_push(galago_object_get_context(presence));

		if (connected)
		{
			if (galago_presence_has_status(presence, "offline") ||
				galago_presence_get_active_status(presence) == NULL)
			{
				galago_presence_add_status(presence,
					galago_status_new(GALAGO_STATUS_AVAILABLE, "available",
									  "Available", TRUE));
			}
		}
		else
		{
			galago_presence_set_idle(presence, FALSE, 0);

			if (!galago_presence_has_status(presence, "offline"))
			{
				galago_presence_add_status(presence,
					galago_status_new(GALAGO_STATUS_OFFLINE, "offline",
									  "Offline", TRUE));
			}
		}

		galago_context_pop();
	}

	galago_signal_context_thaw(signal_context);

	if (galago_account_is_native(account) && galago_is_connected() &&
		galago_core_is_feed())
	{
		galago_dbus_send_message(account, "SetConnected",
								 galago_value_new(GALAGO_TYPE_BOOLEAN,
												  &connected, NULL),
								 NULL);
	}

	galago_signal_emit(account, (connected ? "connected" : "disconnected"));
	galago_signal_emit(account, "updated");
}

GalagoService *
galago_account_get_service(const GalagoAccount *account)
{
	galago_return_val_if_fail(account != NULL,            NULL);
	galago_return_val_if_fail(GALAGO_IS_ACCOUNT(account), NULL);

	return account->priv->service;
}

void
galago_account_set_person(GalagoAccount *account, GalagoPerson *person)
{
	galago_return_if_fail(account != NULL);
	galago_return_if_fail(person  != NULL);
	galago_return_if_fail(GALAGO_IS_ACCOUNT(account));
	galago_return_if_fail(GALAGO_IS_PERSON(person));
	galago_return_if_fail(galago_person_is_native(person) ==
						  galago_account_is_native(account));

	if (account->priv->person == person)
		return;

	galago_person_remove_account(account->priv->person, account);
	account->priv->person = person;
	galago_person_add_account(person, account);
}


GalagoPerson *
galago_account_get_person(const GalagoAccount *account)
{
	galago_return_val_if_fail(account != NULL,            NULL);
	galago_return_val_if_fail(GALAGO_IS_ACCOUNT(account), NULL);

	return account->priv->person;
}

const char *
galago_account_get_username(const GalagoAccount *account)
{
	galago_return_val_if_fail(account != NULL,            NULL);
	galago_return_val_if_fail(GALAGO_IS_ACCOUNT(account), NULL);

	return account->priv->username;
}

galago_bool
galago_account_is_connected(const GalagoAccount *account)
{
	galago_return_val_if_fail(account != NULL,            FALSE);
	galago_return_val_if_fail(GALAGO_IS_ACCOUNT(account), FALSE);

	return account->priv->connected;
}

galago_bool
galago_account_is_native(const GalagoAccount *account)
{
	galago_return_val_if_fail(account != NULL,            FALSE);
	galago_return_val_if_fail(GALAGO_IS_ACCOUNT(account), FALSE);

	return galago_service_is_native(galago_account_get_service(account));
}

void
galago_account_set_display_name(GalagoAccount *account,
                                const char *display_name)
{
	galago_return_if_fail(account != NULL);
	galago_return_if_fail(GALAGO_IS_ACCOUNT(account));

	if (display_name != NULL &&
		(*display_name == '\0' ||
		 !strcmp(display_name, galago_account_get_username(account))))
	{
		display_name = NULL;
	}

	if (display_name == account->priv->display_name ||
		(display_name != NULL && account->priv->display_name != NULL &&
		 !strcmp(account->priv->display_name, display_name)))
	{
		return;
	}

	if (account->priv->display_name != NULL)
		free(account->priv->display_name);

	account->priv->display_name = (display_name == NULL
	                               ? NULL : strdup(display_name));

	if (galago_account_is_native(account) && galago_is_connected() &&
		galago_core_is_feed() && !galago_is_daemon())
	{
		galago_dbus_send_message(account, "SetDisplayName",
		                         galago_value_new(GALAGO_TYPE_STRING,
		                                          &display_name, NULL),
					 NULL);
	}

	galago_signal_emit(account, "display-name-changed");
	galago_signal_emit(account, "updated");
}

const char *
galago_account_get_display_name(const GalagoAccount *account)
{
	galago_return_val_if_fail(account != NULL,            NULL);
	galago_return_val_if_fail(GALAGO_IS_ACCOUNT(account), NULL);

	if (account->priv->display_name == NULL)
		return galago_account_get_username(account);

	return account->priv->display_name;
}

galago_bool
galago_account_is_display_name_set(const GalagoAccount *account)
{
	galago_return_val_if_fail(account != NULL,            FALSE);
	galago_return_val_if_fail(GALAGO_IS_ACCOUNT(account), FALSE);

	return (account->priv->display_name != NULL);
}

void
galago_account_add_contact(GalagoAccount *account, GalagoAccount *contact)
{
	const char *username;
	GalagoService *service;

	galago_return_if_fail(account != NULL);
	galago_return_if_fail(contact != NULL);
	galago_return_if_fail(GALAGO_IS_ACCOUNT(account));
	galago_return_if_fail(GALAGO_IS_ACCOUNT(contact));
	galago_return_if_fail(account != contact);

	username = galago_account_get_username(contact);

	if (galago_account_get_contact(account, username, FALSE) != NULL)
	{
#if 0
		galago_log_warning("A contact with username %s has already been "
		                   "added to account %s\n",
		                   username, galago_account_get_username(account));
#endif
		return;
	}

	service = galago_account_get_service(account);

	galago_hash_table_insert(account->priv->contacts_table,
		strdup(galago_service_normalize(service, username)),
		contact);

	account->priv->contacts = galago_list_append(account->priv->contacts,
												 contact);

	if (galago_account_is_native(account))
		_galago_dbus_account_add_contact(account, contact);

	galago_signal_emit(account, "contact-added", contact);
	galago_signal_emit(account, "updated");
}

void
galago_account_remove_contact(GalagoAccount *account, GalagoAccount *contact)
{
	galago_return_if_fail(account != NULL);
	galago_return_if_fail(contact != NULL);
	galago_return_if_fail(GALAGO_IS_ACCOUNT(account));
	galago_return_if_fail(GALAGO_IS_ACCOUNT(contact));
	galago_return_if_fail(account != contact);

	galago_hash_table_remove(account->priv->contacts_table,
							 galago_account_get_username(contact));

	account->priv->contacts = galago_list_remove(account->priv->contacts,
												 contact);

	if (galago_account_is_native(account) && galago_is_connected() &&
		galago_core_is_feed())
	{
		galago_dbus_send_message(account, "RemoveContact",
								 galago_value_new(GALAGO_TYPE_OBJECT,
												  &contact,
												  GALAGO_CLASS_ACCOUNT),
								 NULL);
	}

	galago_signal_emit(account, "contact-removed", contact);
	galago_signal_emit(account, "updated");
}

GalagoAccount *
galago_account_get_contact(const GalagoAccount *account, const char *username,
                           galago_bool query)
{
	GalagoAccount *contact;
	GalagoService *service;

	galago_return_val_if_fail(account != NULL,                       NULL);
	galago_return_val_if_fail(GALAGO_IS_ACCOUNT(account),            NULL);
	galago_return_val_if_fail(username != NULL && *username != '\0', NULL);

	service = galago_account_get_service(account);

	contact =
		galago_hash_table_lookup(account->priv->contacts_table,
								 galago_service_normalize(service, username));

	if (contact == NULL && query &&
		!galago_account_is_native(account) && !galago_is_daemon() &&
		galago_is_connected())
	{
		GalagoSignalContext *signal_context = GALAGO_SIGNAL_CONTEXT(account);

		galago_context_push(galago_object_get_context(account));
		galago_signal_context_freeze(signal_context);

		contact = galago_dbus_send_message_with_reply(account, "GetContact",
			galago_value_new(GALAGO_TYPE_OBJECT, NULL, GALAGO_CLASS_ACCOUNT),
			galago_value_new(GALAGO_TYPE_STRING, &username, NULL),
			NULL);

		galago_signal_context_thaw(signal_context);
		galago_context_pop();
	}

	return contact;
}

const GalagoList *
galago_account_get_contacts(const GalagoAccount *account, galago_bool query)
{
	galago_return_val_if_fail(account != NULL,            NULL);
	galago_return_val_if_fail(GALAGO_IS_ACCOUNT(account), NULL);

	if (query && !galago_account_is_native(account) && !galago_is_daemon() &&
		galago_is_connected())
	{
		GalagoSignalContext *signal_context = GALAGO_SIGNAL_CONTEXT(account);
		GalagoList *temp;

		galago_context_push(galago_object_get_context(account));
		galago_signal_context_freeze(signal_context);

		temp = galago_dbus_send_message_with_reply(account, "GetContacts",
			galago_value_new_list(GALAGO_TYPE_OBJECT, NULL,
								  GALAGO_CLASS_ACCOUNT),
			NULL);
		galago_list_destroy(temp);

		galago_signal_context_thaw(signal_context);
		galago_context_pop();
	}

	return account->priv->contacts;
}

void
galago_account_set_presence(GalagoAccount *account, GalagoPresence *presence)
{
	galago_return_if_fail(account != NULL);
	galago_return_if_fail(GALAGO_IS_ACCOUNT(account));

	if (account->priv->presence == presence)
		return;

	if (account->priv->presence != NULL)
	{
		GalagoPresence *old_presence = account->priv->presence;

		account->priv->presence = NULL;

		galago_object_unref(old_presence);
	}

	account->priv->presence = presence;

	if (galago_account_is_native(account))
		_galago_dbus_account_set_presence(account, presence);

	galago_signal_emit(account, "presence-set", presence);
	galago_signal_emit(account, "updated");
}

GalagoPresence *
galago_account_get_presence(const GalagoAccount *account, galago_bool query)
{
	galago_return_val_if_fail(account != NULL,            NULL);
	galago_return_val_if_fail(GALAGO_IS_ACCOUNT(account), NULL);

	if (account->priv->presence == NULL && query &&
		!galago_account_is_native(account) && !galago_is_daemon() &&
		galago_is_connected())
	{
		GalagoSignalContext *signal_context = GALAGO_SIGNAL_CONTEXT(account);

		galago_signal_context_freeze(signal_context);
		account->priv->presence =
			galago_dbus_send_message_with_reply(account, "GetPresence",
				galago_value_new(GALAGO_TYPE_OBJECT, NULL,
								 GALAGO_CLASS_PRESENCE),
				NULL);
		galago_signal_context_thaw(signal_context);
	}

	return account->priv->presence;
}

void
galago_account_set_avatar(GalagoAccount *account, GalagoAvatar *avatar)
{
	galago_return_if_fail(account != NULL);
	galago_return_if_fail(GALAGO_IS_ACCOUNT(account));

	if (account->priv->avatar == avatar)
		return;

	if (account->priv->avatar != NULL)
	{
		GalagoAvatar *old_avatar = account->priv->avatar;

		account->priv->avatar = NULL;

		galago_object_unref(old_avatar);
	}

	account->priv->avatar = avatar;

	if (galago_account_is_native(account))
		_galago_dbus_account_set_avatar(account, avatar);

	galago_signal_emit(account, "avatar-set", avatar);
	galago_signal_emit(account, "updated");
}

GalagoAvatar *
galago_account_get_avatar(const GalagoAccount *account, galago_bool query)
{
	galago_return_val_if_fail(account != NULL,            NULL);
	galago_return_val_if_fail(GALAGO_IS_ACCOUNT(account), NULL);

	if (account->priv->avatar == NULL && query &&
		!galago_account_is_native(account) && !galago_is_daemon() &&
		galago_is_connected())
	{
		GalagoSignalContext *signal_context = GALAGO_SIGNAL_CONTEXT(account);

		galago_signal_context_freeze(signal_context);
		account->priv->avatar =
			galago_dbus_send_message_with_reply(account, "GetAvatar",
				galago_value_new(GALAGO_TYPE_OBJECT, NULL,
								 GALAGO_CLASS_AVATAR),
				NULL);
		galago_signal_context_thaw(signal_context);
	}

	return account->priv->avatar;
}


/**************************************************************************
 * D-BUS Functions
 **************************************************************************/
static void
_galago_dbus_account_set_presence(GalagoAccount *account,
                                  GalagoPresence *presence)
{
	if (!galago_is_connected() || !galago_core_is_feed())
		return;

	if (presence == NULL)
	{
		galago_dbus_send_message(account, "UnsetPresence", NULL);
	}
	else
	{
		galago_dbus_send_message(account, "SetPresence",
								 galago_value_new(GALAGO_TYPE_OBJECT,
												  &presence,
												  GALAGO_CLASS_PRESENCE),
								 NULL);
	}
}

static void
_galago_dbus_account_set_avatar(GalagoAccount *account, GalagoAvatar *avatar)
{
	if (!galago_is_connected() || !galago_core_is_feed())
		return;

	if (avatar == NULL)
	{
		galago_dbus_send_message(account, "UnsetAvatar", NULL);
	}
	else
	{
		galago_dbus_send_message(account, "SetAvatar",
								 galago_value_new(GALAGO_TYPE_OBJECT,
												  &avatar,
												  GALAGO_CLASS_AVATAR),
								 NULL);
	}
}

static void
_galago_dbus_account_add_contact(GalagoAccount *account,
                                 GalagoAccount *contact)
{
	if (!galago_is_connected() || !galago_core_is_feed())
		return;

	galago_dbus_send_message(account, "AddContact",
							 galago_value_new(GALAGO_TYPE_OBJECT, &contact,
											  GALAGO_CLASS_ACCOUNT),
							 NULL);
}
