/**
 * @file libgalago/galago-account.h Galago Account API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_ACCOUNT_H_
#define _GALAGO_ACCOUNT_H_

typedef struct _GalagoAccount        GalagoAccount;
typedef struct _GalagoAccountClass   GalagoAccountClass;
typedef struct _GalagoAccountPrivate GalagoAccountPrivate;

#include <libgalago/galago-avatar.h>
#include <libgalago/galago-dbus.h>
#include <libgalago/galago-list.h>
#include <libgalago/galago-object.h>
#include <libgalago/galago-person.h>
#include <libgalago/galago-presence.h>
#include <libgalago/galago-service.h>
#include <libgalago/galago-signals.h>

/**
 * An account on a service.
 */
struct _GalagoAccount
{
	GalagoObject parent_object;

	GalagoAccountPrivate *priv;

	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
};

/**
 * An account's class.
 */
struct _GalagoAccountClass
{
	GalagoObjectClass parent_class;

	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
};

#define GALAGO_CLASS_ACCOUNT (galago_account_get_class())
#define GALAGO_DBUS_ACCOUNT_INTERFACE "org.freedesktop.Galago.Account"
#define GALAGO_IS_ACCOUNT(obj) (GALAGO_IS_OBJECT(obj) && galago_object_check_cast((obj), GALAGO_CLASS_ACCOUNT))

#ifdef __cplusplus
extern "C" {
#endif

/**************************************************************************/
/** @name Account API                                                     */
/**************************************************************************/
/*@{*/

/**
 * Returns the class for a GalagoAccount.
 *
 * @return The GalagoAccount class.
 */
GalagoObjectClass *galago_account_get_class(void);

/**
 * Creates an account.
 *
 * @param service  The service.
 * @param person   The person the account belongs to.
 * @param username The account name.
 *
 * @return The account.
 */
GalagoAccount *galago_account_new(GalagoService *service,
								  GalagoPerson *person,
								  const char *username);

/**
 * Sets whether or not the account is connected.
 *
 * @param account   The account.
 * @param connected The connected state.
 */
void galago_account_set_connected(GalagoAccount *account,
								  galago_bool connected);

/**
 * Returns an account's service.
 *
 * @param account The account.
 *
 * @return The account's service.
 */
GalagoService *galago_account_get_service(const GalagoAccount *account);

/**
 * Sets the account's person.
 *
 * This is used internally. You don't need to call this.
 *
 * @param account The account.
 * @param person  The account's person.
 */
void galago_account_set_person(GalagoAccount *account, GalagoPerson *person);

/**
 * Returns the person that owns an account.
 *
 * @param account The account.
 *
 * @return The person that owns the account.
 */
GalagoPerson *galago_account_get_person(const GalagoAccount *account);

/**
 * Returns an account's username.
 *
 * @param account The account.
 *
 * @return The account's username.
 */
const char *galago_account_get_username(const GalagoAccount *account);

/**
 * Returns whether or not an account is connected.
 *
 * @param account The account.
 *
 * @return The account's connected state.
 */
galago_bool galago_account_is_connected(const GalagoAccount *account);

/**
 * Returns whether or not this is a native account.
 *
 * Accounts are native if they originated from this local application.
 * If they originated from another application through a feed or query,
 * they are not native.
 *
 * @param account The account.
 *
 * @return TRUE if the account is native, or FALSE if foreign.
 */
galago_bool galago_account_is_native(const GalagoAccount *account);

/**
 * Sets the account's displayed name.
 *
 * This is the alias that the account may be shown as on another client.
 * It's purely optional.
 *
 * @param account      The account.
 * @param display_name The display name.
 */
void galago_account_set_display_name(GalagoAccount *account,
									 const char *display_name);

/**
 * Returns the account's displayed name.
 *
 * If the displayed name is not explicitly set, this will return the
 * screen name.
 *
 * @param account The account.
 *
 * @return The displayed name.
 */
const char *galago_account_get_display_name(const GalagoAccount *account);

/**
 * Returns whether or not a custom displayed name is set.
 *
 * @param account The account.
 *
 * @return TRUE if a custom displayed name is set, or FALSE.
 */
galago_bool galago_account_is_display_name_set(const GalagoAccount *account);

/**
 * Adds a contact to an account.
 *
 * @param account The account.
 * @param contact The contact's account to add.
 */
void galago_account_add_contact(GalagoAccount *account,
								GalagoAccount *contact);

/**
 * Removes a contact from an account.
 *
 * @param account The account.
 * @param contact The contact's account to remove.
 */
void galago_account_remove_contact(GalagoAccount *account,
								   GalagoAccount *contact);

/**
 * Returns the contact with the specified username in an account.
 *
 * @param account  The account.
 * @param username The username.
 * @param query    TRUE if a remote query should be done if there is no
 *                 local contact found, or FALSE.
 *
 * @return The contact's account, if found, or NULL.
 */
GalagoAccount *galago_account_get_contact(const GalagoAccount *account,
										  const char *username,
										  galago_bool query);

/**
 * Returns a list of accounts of users seen from this account.
 *
 * @param account The account.
 * @param query   TRUE if a remote query should be done if there is no
 *                local contact found, or FALSE.
 *
 * @return A list of accounts of other users, or NULL.
 */
const GalagoList *galago_account_get_contacts(const GalagoAccount *account,
											  galago_bool query);

/**
 * Sets the account's presence.
 *
 * This is used internally. You don't need to call this.
 *
 * @param account  The account.
 * @param presence The presence to set.
 */
void galago_account_set_presence(GalagoAccount *account,
								 GalagoPresence *presence);

/**
 * Returns the account's presence.
 *
 * @param account The account.
 * @param query   TRUE if a remote query should be done if there is no
 *                local presence found, or FALSE.
 *
 * @return The presence, if found, or NULL.
 */
GalagoPresence *galago_account_get_presence(const GalagoAccount *account,
											galago_bool query);

/**
 * Sets the account's avatar.
 *
 * This is used internally. You don't need to call this.
 *
 * @param account The account.
 * @param avatar  The avatar to set.
 */
void galago_account_set_avatar(GalagoAccount *account, GalagoAvatar *avatar);

/**
 * Returns the account's avatar.
 *
 * @param account The account.
 * @param query   TRUE if a remote query should be done if there is no
 *                local avatar found, or FALSE.
 *
 * @return The avatar, if found, or NULL.
 */
GalagoAvatar *galago_account_get_avatar(const GalagoAccount *account,
										galago_bool query);

/*@}*/

#ifdef __cplusplus
}
#endif

#endif /* _GALAGO_ACCOUNT_H_ */
