/**
 * @file libgalago/galago-context-priv.h Galago Context Private API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_CONTEXT_PRIV_H_
#define _GALAGO_CONTEXT_PRIV_H_

typedef struct _GalagoContextOps GalagoContextOps;

#include <libgalago/galago-person.h>
#include <libgalago/galago-service.h>

struct _GalagoContextOps
{
	void (*service_added)(GalagoService *service);
	void (*service_removed)(GalagoService *service);
	void (*person_added)(GalagoPerson *person);
	void (*person_removed)(GalagoPerson *person);
};


/**************************************************************************/
/** @name Galago Context API                                              */
/**************************************************************************/
/*@{*/

/**
 * Sets the context ops to use for a context.
 *
 * @param context The context.
 * @param ops     The ops structure.
 */
void galago_context_set_ops(GalagoContext *context, GalagoContextOps *ops);

/**
 * Adds a service to the context's services cache.
 *
 * @param service The service to add.
 */
void galago_context_add_service(GalagoService *service);

/**
 * Removes a service from the context's services cache.
 *
 * @param service The service to remove.
 */
void galago_context_remove_service(GalagoService *service);

/**
 * Adds a person to the context's person cache.
 *
 * @param person The person to add.
 */
void galago_context_add_person(GalagoPerson *person);

/**
 * Removes a person from the context's person cache.
 *
 * @param person The person to remove.
 */
void galago_context_remove_person(GalagoPerson *person);

/**
 * Adds an object to the internal object tree.
 *
 * This should only be used internally.
 *
 * @param obj The object to add.
 */
void galago_context_add_object(void *obj);

/**
 * Removes an object from the internal object tree.
 *
 * This should only be used internally.
 *
 * @param obj The object to remove.
 */
void galago_context_remove_object(void *obj);

/**
 * Clears all native or foreign objects from the context.
 *
 * This should only be used internally.
 *
 * @param native TRUE if native objects should be cleared.
 *               FALSE if foreign objects should be cleared.
 */
void galago_context_clear_objects(galago_bool native);

/*@}*/

#ifdef __cplusplus
}
#endif

#endif /* _GALAGO_CONTEXT_PRIV_H_ */
