/**
 * @file libgalago/galago-context.c Galago Context API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-context.h>
#include <libgalago/galago-context-priv.h>
#include <libgalago/galago-assert.h>
#include <libgalago/galago-core.h>
#include <libgalago/galago-hashtable.h>
#include <libgalago/galago-list.h>
#include <libgalago/galago-utils.h>
#include <string.h>

struct _GalagoContextPrivate
{
	GalagoContextOps *ops;

	char *obj_path_prefix;

	GalagoHashTable *services_table;
	GalagoHashTable *people_table;

	struct
	{
		GalagoList *services;
		GalagoList *people;

	} native;

	struct
	{
		GalagoList *services;
		GalagoList *people;

	} foreign;

	GalagoHashTable *obj_tree;
};

typedef struct
{
	char *id;

	galago_bool native;

} ServicePersonCacheKey;

static GalagoList *contexts = NULL;

static unsigned int
cache_key_hash(const void *key)
{
	return galago_str_hash(((ServicePersonCacheKey *)key)->id);
}

static galago_bool
cache_key_equal(const void *a, const void *b)
{
	ServicePersonCacheKey *key1 = (ServicePersonCacheKey *)a;
	ServicePersonCacheKey *key2 = (ServicePersonCacheKey *)b;

	return (galago_str_equal(key1->id, key2->id) &&
			key1->native == key2->native);
}

static void
cache_key_destroy(void *ptr)
{
	ServicePersonCacheKey *key = (ServicePersonCacheKey *)ptr;

	if (key->id != NULL)
		free(key->id);

	free(key);
}

GalagoContext *
galago_context_new(void)
{
	GalagoContext *context;

	context       = galago_new0(GalagoContext,        1);
	context->priv = galago_new0(GalagoContextPrivate, 1);

	context->priv->services_table =
		galago_hash_table_new_full(cache_key_hash, cache_key_equal,
								   cache_key_destroy, NULL);

	context->priv->people_table =
		galago_hash_table_new_full(cache_key_hash, cache_key_equal,
								   cache_key_destroy, NULL);

	context->priv->obj_tree =
		galago_hash_table_new_full(galago_str_hash, galago_str_equal,
								   free, NULL);

	return context;
}

void
galago_context_destroy(GalagoContext *context)
{
	galago_return_if_fail(context != NULL);

	galago_list_foreach(context->priv->native.services,
						(GalagoForEachFunc)galago_object_unref, NULL);
	galago_list_foreach(context->priv->foreign.services,
						(GalagoForEachFunc)galago_object_unref, NULL);
	galago_list_foreach(context->priv->native.people,
						(GalagoForEachFunc)galago_object_unref, NULL);
	galago_list_foreach(context->priv->foreign.people,
						(GalagoForEachFunc)galago_object_unref, NULL);

	galago_hash_table_destroy(context->priv->services_table);
	galago_hash_table_destroy(context->priv->people_table);
	galago_hash_table_destroy(context->priv->obj_tree);

	if (context->priv->obj_path_prefix != NULL)
		free(context->priv->obj_path_prefix);

	free(context->priv);
	free(context);
}

void
galago_context_push(GalagoContext *context)
{
	galago_return_if_fail(context != NULL);

	contexts = galago_list_prepend(contexts, context);
}

void
galago_context_pop(void)
{
	GalagoContext *context = galago_context_get();

	if (context != NULL)
		contexts = galago_list_remove(contexts, context);
}

GalagoContext *
galago_context_get(void)
{
	if (contexts == NULL)
		return NULL;

	return (GalagoContext *)contexts->data;
}

void
galago_context_set_obj_path_prefix(const char *prefix)
{
	GalagoContext *context;

	galago_return_if_fail(prefix != NULL);

	context = galago_context_get();

	galago_return_if_fail(context != NULL);

	if (context->priv->obj_path_prefix != NULL)
		free(context->priv->obj_path_prefix);

	context->priv->obj_path_prefix = (prefix == NULL ? NULL : strdup(prefix));
}

const char *
galago_context_get_obj_path_prefix(void)
{
	GalagoContext *context;

	context = galago_context_get();

	galago_return_val_if_fail(context != NULL, NULL);

	return context->priv->obj_path_prefix;
}

void
galago_context_set_ops(GalagoContext *context, GalagoContextOps *ops)
{
	galago_return_if_fail(context != NULL);

	context->priv->ops = ops;
}

void
galago_context_add_service(GalagoService *service)
{
	ServicePersonCacheKey *key;
	GalagoContext *context;

	galago_return_if_fail(galago_is_initted());
	galago_return_if_fail(service != NULL);

	context = galago_context_get();
	galago_return_if_fail(context != NULL);

	key         = galago_new0(ServicePersonCacheKey, 1);
	key->id     = galago_str_lower(galago_service_get_id(service));
	key->native = galago_service_is_native(service);

	if (galago_context_get_service(key->id, key->native) != NULL)
	{
		galago_log_warning(
			   (key->native
				? "A native service with ID %s has already been added.\n"
				: "A foreign service with ID %s has already been added.\n"),
				key->id);

		cache_key_destroy(key);

		return;
	}

	galago_hash_table_insert(context->priv->services_table, key, service);

	if (key->native)
	{
		context->priv->native.services =
			galago_list_append(context->priv->native.services, service);
	}
	else
	{
		context->priv->foreign.services =
			galago_list_append(context->priv->foreign.services, service);
	}

	if (context->priv->ops != NULL &&
		context->priv->ops->service_added != NULL)
	{
		context->priv->ops->service_added(service);
	}
}

void
galago_context_remove_service(GalagoService *service)
{
	GalagoContext *context;
	ServicePersonCacheKey key;

	galago_return_if_fail(galago_is_initted());
	galago_return_if_fail(service != NULL);

	context = galago_context_get();
	galago_return_if_fail(context != NULL);

	key.id     = galago_str_lower(galago_service_get_id(service));
	key.native = galago_service_is_native(service);

	if (key.native)
	{
		context->priv->native.services =
			galago_list_remove(context->priv->native.services, service);
	}
	else
	{
		context->priv->foreign.services =
			galago_list_remove(context->priv->foreign.services, service);
	}

	galago_hash_table_remove(context->priv->services_table, &key);

	free(key.id);

	if (context->priv->ops != NULL &&
		context->priv->ops->service_removed != NULL)
	{
		context->priv->ops->service_removed(service);
	}
}

GalagoService *
galago_context_get_service(const char *id, galago_bool native)
{
	GalagoContext *context;
	GalagoService *service;
	ServicePersonCacheKey key;

	galago_return_val_if_fail(galago_is_initted(), NULL);
	galago_return_val_if_fail(id != NULL, NULL);

	context = galago_context_get();
	galago_return_val_if_fail(context != NULL, NULL);

	key.id     = galago_str_lower(id);
	key.native = native;

	service = galago_hash_table_lookup(context->priv->services_table, &key);

	free(key.id);

	return service;
}

const GalagoList *
galago_context_get_services(galago_bool native)
{
	GalagoContext *context;

	galago_return_val_if_fail(galago_is_initted(), NULL);

	context = galago_context_get();
	galago_return_val_if_fail(context != NULL, NULL);

	return (native
			? context->priv->native.services
			: context->priv->foreign.services);
}

void
galago_context_add_person(GalagoPerson *person)
{
	GalagoContext *context;
	ServicePersonCacheKey *key;

	galago_return_if_fail(galago_is_initted());
	galago_return_if_fail(person != NULL);

	context = galago_context_get();
	galago_return_if_fail(context != NULL);

	key = galago_new0(ServicePersonCacheKey, 1);
	key->id     = galago_str_lower(galago_person_get_id(person));
	key->native = galago_person_is_native(person);

	if (galago_context_get_person(key->id, key->native) != NULL)
	{
		galago_log_warning(
			   (key->native
				? "A native person with ID %s has already been added.\n"
				: "A foreign person with ID %s has already been added.\n"),
				key->id);
		return;
	}

	if (key->native)
	{
		context->priv->native.people =
			galago_list_append(context->priv->native.people, person);
	}
	else
	{
		context->priv->foreign.people =
			galago_list_append(context->priv->foreign.people, person);
	}

	galago_hash_table_insert(context->priv->people_table, key, person);

	if (context->priv->ops != NULL &&
		context->priv->ops->person_added != NULL)
	{
		context->priv->ops->person_added(person);
	}
}

void
galago_context_remove_person(GalagoPerson *person)
{
	GalagoContext *context;
	ServicePersonCacheKey key;

	galago_return_if_fail(galago_is_initted());
	galago_return_if_fail(person != NULL);

	context = galago_context_get();
	galago_return_if_fail(context != NULL);

	key.id     = galago_str_lower(galago_person_get_id(person));
	key.native = galago_person_is_native(person);

	if (key.native)
	{
		context->priv->native.people =
			galago_list_remove(context->priv->native.people, person);
	}
	else
	{
		context->priv->foreign.people =
			galago_list_remove(context->priv->foreign.people, person);
	}

	galago_hash_table_remove(context->priv->people_table, &key);

	free(key.id);

	if (context->priv->ops != NULL &&
		context->priv->ops->person_removed != NULL)
	{
		context->priv->ops->person_removed(person);
	}
}

GalagoPerson *
galago_context_get_person(const char *id, galago_bool native)
{
	GalagoContext *context;
	GalagoPerson *person;
	ServicePersonCacheKey key;

	galago_return_val_if_fail(galago_is_initted(), FALSE);
	galago_return_val_if_fail(id != NULL, NULL);

	context = galago_context_get();
	galago_return_val_if_fail(context != NULL, NULL);

	key.id     = galago_str_lower(id);
	key.native = native;

	person = galago_hash_table_lookup(context->priv->people_table, &key);

	free(key.id);

	return person;
}

const GalagoList *
galago_context_get_people(galago_bool native)
{
	GalagoContext *context;

	galago_return_val_if_fail(galago_is_initted(), NULL);

	context = galago_context_get();
	galago_return_val_if_fail(context != NULL, NULL);

	return (native
			? context->priv->native.people
			: context->priv->foreign.people);
}

void
galago_context_add_object(void *obj)
{
	GalagoContext *context;

	galago_return_if_fail(obj != NULL);
	galago_return_if_fail(GALAGO_IS_OBJECT(obj));
	galago_return_if_fail(galago_object_get_dbus_path(obj) != NULL);
	galago_return_if_fail(galago_is_initted());

	context = galago_context_get();

	galago_hash_table_insert(context->priv->obj_tree,
							 strdup(galago_object_get_dbus_path(obj)), obj);
}

void
galago_context_remove_object(void *obj)
{
	GalagoContext *context;

	galago_return_if_fail(obj != NULL);
	galago_return_if_fail(GALAGO_IS_OBJECT(obj));
	galago_return_if_fail(galago_object_get_dbus_path(obj) != NULL);
	galago_return_if_fail(galago_is_initted());

	context = galago_context_get();

	galago_hash_table_remove(context->priv->obj_tree,
							 galago_object_get_dbus_path(obj));
}

void
galago_context_clear_objects(galago_bool native)
{
	GalagoContext *context;

	galago_return_if_fail(galago_is_initted());

	context = galago_context_get();

	if (native)
	{
		galago_list_foreach(context->priv->native.services,
							(GalagoForEachFunc)galago_object_unref, NULL);
		galago_list_foreach(context->priv->native.people,
							(GalagoForEachFunc)galago_object_unref, NULL);

		context->priv->native.services = NULL;
		context->priv->native.people   = NULL;
	}
	else
	{
		galago_list_foreach(context->priv->foreign.services,
							(GalagoForEachFunc)galago_object_unref, NULL);
		galago_list_foreach(context->priv->foreign.people,
							(GalagoForEachFunc)galago_object_unref, NULL);

		context->priv->foreign.services = NULL;
		context->priv->foreign.people   = NULL;
	}
}

void *
galago_context_get_object(const char *path)
{
	GalagoContext *context;

	galago_return_val_if_fail(path != NULL,        NULL);
	galago_return_val_if_fail(galago_is_initted(), NULL);

	context = galago_context_get();

	return galago_hash_table_lookup(context->priv->obj_tree, path);
}
