/**
 * @file libgalago/galago-context.h Galago Context API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_CONTEXT_H_
#define _GALAGO_CONTEXT_H_

/*
 * NOTE: The base GalagoContext struct has been split out into
 *       galago-context-base.h in order to break a fairly nasty header
 *       file dependency hell problem that was caused by the fact that
 *       GalagoObject requires GalagoContext, and GalagoContext makes use of
 *       things that inherit GalagoObject. Eventually, it all just broke. This
 *       only matters inside of here, since nothing outside libgalago requests
 *       specific header files.
 */

#include <libgalago/galago-context-base.h>
#include <libgalago/galago-person.h>
#include <libgalago/galago-service.h>

#ifdef __cplusplus
extern "C" {
#endif

/**************************************************************************/
/** @name Galago Context API                                              */
/**************************************************************************/
/*@{*/

/**
 * Creates a new context.
 *
 * @return The new context.
 */
GalagoContext *galago_context_new(void);

/**
 * Destroys a context.
 *
 * @param context The context to destroy.
 */
void galago_context_destroy(GalagoContext *context);

/**
 * Pushes a context onto the stack.
 *
 * @param context The context to push.
 */
void galago_context_push(GalagoContext *context);

/**
 * Pops a context off the stack.
 */
void galago_context_pop(void);

/**
 * Returns the current context.
 *
 * @return The current context.
 */
GalagoContext *galago_context_get(void);

/**
 * Sets the base object path prefix for this context.
 *
 * @param prefix The object path prefix.
 */
void galago_context_set_obj_path_prefix(const char *prefix);

/**
 * Returns the base object path prefix for this context.
 *
 * @return The object path prefix.
 */
const char *galago_context_get_obj_path_prefix(void);

/**
 * Returns the service with the specified ID.
 *
 * @param id     The service ID.
 * @param native TRUE if the desired service is native.
 *
 * @return The service, if found, or NULL.
 */
GalagoService *galago_context_get_service(const char *id, galago_bool native);

/**
 * Returns a list of all services.
 *
 * @param native TRUE if the desired services are native.
 *
 * @return The list of services, if found, or NULL.
 */
const GalagoList *galago_context_get_services(galago_bool native);

/**
 * Returns the person with the specified ID.
 *
 * @param id     The person ID.
 * @param native TRUE if the desired person is native.
 *
 * @return The person, if found, or NULL.
 */
GalagoPerson *galago_context_get_person(const char *id, galago_bool native);

/**
 * Returns a list of all people.
 *
 * @param native TRUE if the desired people are native.
 *
 * @return The list of people, if found, or NULL.
 */
const GalagoList *galago_context_get_people(galago_bool native);

/**
 * Finds an object with the specified path.
 *
 * @param path The D-BUS object path.
 *
 * @return The object if found, or NULL.
 */
void *galago_context_get_object(const char *path);

/*@}*/

#ifdef __cplusplus
}
#endif

#endif /* _GALAGO_CONTEXT_H_ */
