/**
 * @file libgalago/galago-list.h List API
 *
 * @Copyright (C) 2004-2005 Christian Hammond.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_LIST_H_
#define _GALAGO_LIST_H_

typedef struct _GalagoList GalagoList;

#include <libgalago/galago-types.h>

/**
 * An node in a linked list.
 */
struct _GalagoList
{
	GalagoList *prev;
	GalagoList *next;

	void  *data;
};

#ifdef __cplusplus
extern "C" {
#endif

/**
 * Creates a new linked list node.
 *
 * @return The linked list node.
 */
GalagoList *galago_list_new(void);

/**
 * Destroys a linked list.
 *
 * The data inside the list will not be destroyed.
 *
 * @param list The first node in the linked list.
 */
void galago_list_destroy(GalagoList *list);

/**
 * Destroys a single node in a linked list.
 *
 * The node will not be unlinked. Use galago_list_delete_link() for that.
 *
 * @param list The linked list node.
 */
void galago_list_destroy_1(GalagoList *list);

/**
 * Duplicates a list.
 *
 * @param list The source list.
 *
 * @return The new list.
 */
GalagoList *galago_list_dup(const GalagoList *list);

/**
 * Appends data to a linked list.
 *
 * @param list The start of the linked list.
 * @param data The data to append.
 *
 * @return The new beginning of the linked list.
 */
GalagoList *galago_list_append(GalagoList *list, void *data);

/**
 * Prepends data to a linked list.
 *
 * @param list The start of the linked list.
 * @param data The data to prepend.
 *
 * @return The new beginning of the linked list.
 */
GalagoList *galago_list_prepend(GalagoList *list, void *data);

/**
 * Removes data and its node from a linked list.
 *
 * @param list The start of the linked list.
 * @param data The data to remove.
 *
 * @return The new beginning of the linked list.
 */
GalagoList *galago_list_remove(GalagoList *list, void *data);

/**
 * Unlinks a node from a linked list.
 *
 * This will not destroy the node.
 *
 * @param list The linked list.
 * @param link The node to unlink.
 *
 * @return The new start of the linked list.
 */
GalagoList *galago_list_remove_link(GalagoList *list, GalagoList *link);

/**
 * Unlinks and deletes a node from a linked list.
 *
 * @param list The linked list.
 * @param link The link to unlink and destroy.
 *
 * @return The new start of the linked list.
 */
GalagoList *galago_list_delete_link(GalagoList *list, GalagoList *link);

/**
 * Finds the node in a linked list with the specified data.
 *
 * @param list The linked list.
 * @param data The data contained in the node.
 *
 * @return The linked list node, if found, or NULL.
 */
GalagoList *galago_list_find(GalagoList *list, void *data);

/**
 * Calls a function on each node in a linked list.
 *
 * @param list      The linked list.
 * @param func      The function to call.
 * @param user_data User-specified data to pass to the function.
 */
void galago_list_foreach(const GalagoList *list, GalagoForEachFunc func,
						 void *user_data);

/**
 * Returns the first node in a linked list.
 *
 * @param list The linked list.
 *
 * @return The first node in the linked list.
 */
GalagoList *galago_list_get_first(GalagoList *list);

/**
 * Returns the last node in a linked list.
 *
 * @param list The linked list.
 *
 * @return The last node in the linked list.
 */
GalagoList *galago_list_get_last(GalagoList *list);

/**
 * Returns the previous node in a linked list.
 *
 * @param list The current node in the linked list.
 *
 * @return The previous item in the list.
 */
GalagoList *galago_list_get_prev(GalagoList *list);

/**
 * Returns the next node in a linked list.
 *
 * @param list The current node in the linked list.
 *
 * @return The next item in the list.
 */
GalagoList *galago_list_get_next(GalagoList *list);

/**
 * Returns the data in a node in a linked list.
 *
 * @param list The current node in the linked list.
 *
 * @return The data in the node.
 */
void *galago_list_get_data(GalagoList *list);

/**
 * Returns the number of nodes in a linked list.
 *
 * @param list The linked list.
 *
 * @return The number of nodes in the linked list.
 */
size_t galago_list_get_count(const GalagoList *list);

#ifdef __cplusplus
}
#endif

#endif /* _GALAGO_LIST_H_ */
