/**
 * @file libgalago/galago-log.c Log API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-log.h>
#include <libgalago/galago-assert.h>
#include <stdlib.h>
#include <stdio.h>

static galago_bool log_enabled[GALAGO_LOG_NUM_LEVELS] =
{
	0, TRUE, TRUE, TRUE, TRUE, TRUE, TRUE
	/* 0, FALSE, FALSE, FALSE, TRUE, FALSE */
};

static const char *level_names[] =
{
	NULL, "MISC", "INFO", "WARNING", "ERROR", "FATAL", "ASSERT"
};

static struct
{
	GalagoLogHandlerFunc func;
	void *data;

} log_handlers[GALAGO_LOG_NUM_LEVELS] =
{
	{ NULL, NULL }, /* ALL     */
	{ NULL, NULL }, /* MISC    */
	{ NULL, NULL }, /* INFO    */
	{ NULL, NULL }, /* WARNING */
	{ NULL, NULL }, /* ERROR   */
	{ NULL, NULL }, /* FATAL   */
	{ NULL, NULL }  /* ASSERT  */
};

void
galago_log_set_enabled(GalagoLogLevel level, galago_bool enabled)
{
	galago_return_if_fail(level != GALAGO_LOG_NUM_LEVELS);

	if (level == GALAGO_LOG_ALL)
	{
		log_enabled[GALAGO_LOG_MISC]    = enabled;
		log_enabled[GALAGO_LOG_INFO]    = enabled;
		log_enabled[GALAGO_LOG_WARNING] = enabled;
		log_enabled[GALAGO_LOG_ERROR]   = enabled;
		log_enabled[GALAGO_LOG_FATAL]   = enabled;
		log_enabled[GALAGO_LOG_ASSERT]  = enabled;
	}
	else
		log_enabled[level] = enabled;
}

void
galago_log_set_handler(GalagoLogLevel level, GalagoLogHandlerFunc func,
					   void *user_data)
{
	galago_return_if_fail(level >= GALAGO_LOG_ALL &&
						  level <  GALAGO_LOG_NUM_LEVELS);

	if (level == GALAGO_LOG_ALL)
	{
		galago_log_set_handler(GALAGO_LOG_MISC,    func, user_data);
		galago_log_set_handler(GALAGO_LOG_INFO,    func, user_data);
		galago_log_set_handler(GALAGO_LOG_WARNING, func, user_data);
		galago_log_set_handler(GALAGO_LOG_ERROR,   func, user_data);
		galago_log_set_handler(GALAGO_LOG_FATAL,   func, user_data);
		galago_log_set_handler(GALAGO_LOG_ASSERT,  func, user_data);
	}
	else
	{
		log_handlers[level].func = func;
		log_handlers[level].data = (func == NULL ? NULL : user_data);
	}
}

void
galago_log_unset_handler(GalagoLogLevel level)
{
	galago_return_if_fail(level >= GALAGO_LOG_ALL &&
						  level <  GALAGO_LOG_NUM_LEVELS);

	galago_log_set_handler(level, NULL, NULL);
}

void
galago_log(GalagoLogLevel level, const char *format, ...)
{
	va_list args;

	galago_return_if_fail(level  != GALAGO_LOG_ALL);
	galago_return_if_fail(level  != GALAGO_LOG_NUM_LEVELS);
	galago_return_if_fail(format != NULL);

	va_start(args, format);
	galago_log_args(level, format, args);
	va_end(args);
}

void
galago_log_args(GalagoLogLevel level, const char *format, va_list args)
{
	galago_return_if_fail(level  != GALAGO_LOG_ALL);
	galago_return_if_fail(level  != GALAGO_LOG_NUM_LEVELS);
	galago_return_if_fail(format != NULL);

	if (log_enabled[level])
	{
		if (log_handlers[level].func == NULL)
		{
			fprintf(stderr, "D:%s: ", level_names[level]);
			vfprintf(stderr, format, args);
		}
		else
		{
			log_handlers[level].func(level, format, args,
									 log_handlers[level].data);
		}

		if (level == GALAGO_LOG_FATAL)
			exit(EXIT_FAILURE);
	}
}

void
galago_log_misc(const char *format, ...)
{
	va_list args;

	galago_return_if_fail(format != NULL);

	va_start(args, format);
	galago_log_args(GALAGO_LOG_MISC, format, args);
	va_end(args);
}

void
galago_log_info(const char *format, ...)
{
	va_list args;

	galago_return_if_fail(format != NULL);

	va_start(args, format);
	galago_log_args(GALAGO_LOG_INFO, format, args);
	va_end(args);
}

void
galago_log_warning(const char *format, ...)
{
	va_list args;

	galago_return_if_fail(format != NULL);

	va_start(args, format);
	galago_log_args(GALAGO_LOG_WARNING, format, args);
	va_end(args);
}

void
galago_log_error(const char *format, ...)
{
	va_list args;

	galago_return_if_fail(format != NULL);

	va_start(args, format);
	galago_log_args(GALAGO_LOG_ERROR, format, args);
	va_end(args);
}

void
galago_log_fatal(const char *format, ...)
{
	va_list args;

	galago_return_if_fail(format != NULL);

	va_start(args, format);
	galago_log_args(GALAGO_LOG_FATAL, format, args);
	va_end(args);
}
