/**
 * @file libgalago/galago-photo.c Galago Photo API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-photo.h>
#include <libgalago/galago-assert.h>
#include <libgalago/galago-context.h>
#include <libgalago/galago-core.h>
#include <libgalago/galago-image.h>
#include <libgalago/galago-object-utils.h>
#include <libgalago/galago-utils.h>
#include <libgalago/galago-utils-priv.h>
#include <stdio.h>
#include <string.h>

struct _GalagoPhotoPrivate
{
	GalagoPerson *person;
};


/**************************************************************************
 * Object/Class support
 **************************************************************************/
GALAGO_REGISTER_CLASS(galago_photo, GalagoPhoto, GALAGO_CLASS_IMAGE,
					  GALAGO_DBUS_PHOTO_INTERFACE);
DEPRECATE_OLD_GET_CLASS(galago_photos, galago_photo);

static void
galago_photo_object_init(GalagoPhoto *photo)
{
	photo->priv = galago_new0(GalagoPhotoPrivate, 1);
}

static void
galago_photo_object_finalize(GalagoObject *object)
{
	GalagoPhoto *photo = (GalagoPhoto *)object;
	GalagoPerson *person;

	person = galago_photo_get_person(photo);

	if (galago_person_get_photo(person, FALSE) == photo)
		galago_person_set_photo(person, NULL);

	free(photo->priv);
}

static void
galago_photo_dbus_message_append(DBusMessageIter *iter,
								 const GalagoObject *object)
{
	GalagoPhoto *photo = (GalagoPhoto *)object;
	unsigned char *image_data;
	size_t image_len;

	galago_dbus_message_iter_append_object(iter,
		galago_photo_get_person(photo));

	galago_photo_get_image_data(photo, &image_data, &image_len);
	galago_dbus_message_iter_append_byte_array(iter, image_data, image_len);
}

static void *
galago_photo_dbus_message_get(DBusMessageIter *iter)
{
	GalagoPhoto *photo;
	GalagoPerson *person;
	unsigned char *image_data;
	int image_len;

	person = galago_dbus_message_iter_get_object(iter, GALAGO_CLASS_PERSON);
	dbus_message_iter_next(iter);

	galago_dbus_message_iter_get_byte_array(iter, &image_data, &image_len);

	photo = galago_photo_new(person, image_data, image_len);

	return photo;
}

static void
galago_photo_class_init(GalagoObjectClass *klass)
{
	klass->finalize            = galago_photo_object_finalize;
	klass->dbus_message_append = galago_photo_dbus_message_append;
	klass->dbus_message_get    = galago_photo_dbus_message_get;
}


/**************************************************************************
 * Photo API
 **************************************************************************/
GalagoPhoto *
galago_photo_new(GalagoPerson *person, const unsigned char *data, size_t len)
{
	GalagoPhoto *photo;

	galago_return_val_if_fail(galago_is_initted(),      NULL);
	galago_return_val_if_fail(person != NULL,           NULL);
	galago_return_val_if_fail(GALAGO_IS_PERSON(person), NULL);

	photo = galago_person_get_photo(person, FALSE);

	if (photo == NULL)
	{
		const char *obj_prefix;

		galago_context_push(galago_object_get_context(person));
		photo = galago_object_new(GALAGO_CLASS_PHOTO);
		galago_context_pop();

		photo->priv->person = person;

		if ((obj_prefix = galago_object_get_dbus_path(person)) != NULL)
		{
			size_t len;
			char *obj_path;

			len = strlen(obj_prefix) + strlen("/photo") + 1;

			obj_path = galago_new(char, len);
			snprintf(obj_path, len, "%s/photo", obj_prefix);

			galago_object_set_dbus_path(photo, obj_path);

			free(obj_path);
		}

		galago_photo_set_image_data(photo, data, len);
		galago_person_set_photo(person, photo);
	}
	else
	{
		galago_photo_set_image_data(photo, data, len);
	}

	return photo;
}

void
galago_photo_set_image_data(GalagoPhoto *photo,
							const unsigned char *data, size_t len)
{
	galago_return_if_fail(photo != NULL);
	galago_return_if_fail(GALAGO_IS_PHOTO(photo));
	galago_return_if_fail(data == NULL || len > 0);

	galago_image_set_image_data(GALAGO_IMAGE(photo), data, len);

	if (galago_person_is_native(galago_photo_get_person(photo)) &&
		galago_is_connected() && galago_core_is_feed())
	{
		unsigned char *image_data;
		size_t image_len;

		galago_photo_get_image_data(photo, &image_data, &image_len);
		galago_dbus_send_message(photo, "SetImage",
			galago_value_new_array(GALAGO_TYPE_CHAR, image_data, image_len,
								   NULL),
			NULL);
	}
}

GalagoPerson *
galago_photo_get_person(const GalagoPhoto *photo)
{
	galago_return_val_if_fail(photo != NULL,          NULL);
	galago_return_val_if_fail(GALAGO_IS_PHOTO(photo), NULL);

	return photo->priv->person;
}

void
galago_photo_get_image_data(const GalagoPhoto *photo,
							unsigned char **ret_data, size_t *ret_len)
{
	galago_return_if_fail(photo != NULL);
	galago_return_if_fail(GALAGO_IS_PHOTO(photo));
	galago_return_if_fail(ret_data != NULL || ret_len != NULL);

	galago_image_get_image_data(GALAGO_IMAGE(photo), ret_data, ret_len);
}
