/**
 * @file libgalago/galago-service.c Galago Service API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-private.h>
#include <libgalago/galago-service.h>
#include <libgalago/galago-assert.h>
#include <libgalago/galago-context.h>
#include <libgalago/galago-context-priv.h>
#include <libgalago/galago-core.h>
#include <libgalago/galago-hashtable.h>
#include <libgalago/galago-marshal.h>
#include <libgalago/galago-object-utils.h>
#include <libgalago/galago-utils.h>
#include <libgalago/galago-utils-priv.h>
#include <stdio.h>
#include <string.h>

struct _GalagoServicePrivate
{
	galago_bool native;
	GalagoServiceFlags flags;

	char *id;
	char *name;

	GalagoHashTable *accounts_table;
	GalagoList *accounts;
};

static GalagoService *_galago_service_new_common(const char *id,
												 const char *name,
												 galago_bool native,
												 const char *obj_path,
												 GalagoServiceFlags flags);
static void _galago_dbus_service_add_account(GalagoService *service,
											 GalagoAccount *account);

static GalagoHashTable *service_id_map_table = NULL;

/**************************************************************************
 * Object/Class support
 **************************************************************************/
GALAGO_REGISTER_CLASS(galago_service, GalagoService, NULL,
					  GALAGO_DBUS_SERVICE_INTERFACE);
DEPRECATE_OLD_GET_CLASS(galago_services, galago_service);

static void
galago_service_object_init(GalagoService *service)
{
	service->priv = galago_new0(GalagoServicePrivate, 1);

	service->priv->accounts_table =
		galago_hash_table_new_full(galago_str_hash, galago_str_equal,
								   free, NULL);
}

static void
galago_service_object_finalize(GalagoObject *object)
{
	GalagoService *service = (GalagoService *)object;

	if (service->priv->accounts != NULL)
	{
		galago_list_foreach(service->priv->accounts,
							(GalagoForEachFunc)galago_object_unref, NULL);
		galago_list_destroy(service->priv->accounts);
	}

	galago_hash_table_destroy(service->priv->accounts_table);

	galago_context_push(galago_object_get_context(service));
	galago_context_remove_service(service);
	galago_context_pop();

	if (service->priv->id != NULL)
		free(service->priv->id);

	if (service->priv->name != NULL)
		free(service->priv->name);

	free(service->priv);
}

static void
galago_service_dbus_message_append(DBusMessageIter *iter,
								   const GalagoObject *object)
{
	GalagoService *service = (GalagoService *)object;
	const char *obj_path, *id, *name;
	dbus_uint32_t flags;

	obj_path = galago_object_get_dbus_path(service);
	id       = galago_service_get_id(service);
	name     = galago_service_get_name(service);
	flags    = galago_service_get_flags(service);

	galago_dbus_message_iter_append_string(iter, obj_path);
	galago_dbus_message_iter_append_string(iter, id);
	galago_dbus_message_iter_append_string(iter, name);
	galago_dbus_message_iter_append_uint32(iter, flags);
}

static void *
galago_service_dbus_message_get(DBusMessageIter *iter)
{
	GalagoService *service;
	char *obj_path, *id, *name;
	GalagoServiceFlags flags;

	galago_dbus_message_iter_get_string(iter, obj_path);
	dbus_message_iter_next(iter);

	galago_dbus_message_iter_get_string(iter, id);
	dbus_message_iter_next(iter);

	galago_dbus_message_iter_get_string(iter, name);
	dbus_message_iter_next(iter);

	galago_dbus_message_iter_get_uint32(iter, flags);

	service = _galago_service_new_common(id, name, FALSE, obj_path, flags);

#if !GALAGO_CHECK_DBUS_VERSION(0, 30)
	dbus_free(obj_path);
	dbus_free(id);
	dbus_free(name);
#endif

	return service;
}

static void
galago_service_dbus_push_full(GalagoObject *object)
{
	GalagoService *service = (GalagoService *)object;
	GalagoAccount *account;
	const GalagoList *l;

	for (l = galago_service_get_accounts(service, FALSE);
		 l != NULL;
		 l = l->next)
	{
		account = (GalagoAccount *)l->data;

		_galago_dbus_service_add_account(service, account);
	}

	/* We must go through it a second time. */
	for (l = galago_service_get_accounts(service, FALSE);
		 l != NULL;
		 l = l->next)
	{
		account = (GalagoAccount *)l->data;

		galago_dbus_object_push_full(account);
	}
}

static void
galago_service_class_init(GalagoObjectClass *klass)
{
	klass->finalize            = galago_service_object_finalize;
	klass->dbus_message_append = galago_service_dbus_message_append;
	klass->dbus_message_get    = galago_service_dbus_message_get;
	klass->dbus_push_full      = galago_service_dbus_push_full;

	galago_signal_register(klass->signal_context, "account-added",
						   galago_marshal_VOID__BOXED, 1,
						   GALAGO_TYPE_OBJECT);
	galago_signal_register(klass->signal_context, "account-removed",
						   galago_marshal_VOID__BOXED, 1,
						   GALAGO_TYPE_OBJECT);
	galago_signal_register(klass->signal_context, "updated",
						   galago_marshal_VOID, 0);
}

/**************************************************************************
 * GalagoService API
 **************************************************************************/
typedef struct
{
	char *name;
	GalagoServiceFlags flags;

} ServiceMapInfo;

#define ADD_SERVICE_MAP(id, _name, _flags) \
	map_info = galago_new0(ServiceMapInfo, 1); \
	map_info->name  = strdup((_name)); \
	map_info->flags = (_flags); \
	galago_hash_table_insert(service_id_map_table, strdup(id), map_info)

static void
destroy_map_info(ServiceMapInfo *map_info)
{
	free(map_info->name);
	free(map_info);
}

static void
_init_service_id_map_table(void)
{
	ServiceMapInfo *map_info;

	service_id_map_table =
		galago_hash_table_new_full(galago_str_hash, galago_str_equal,
								   free, (GalagoFreeFunc)destroy_map_info);

	ADD_SERVICE_MAP(GALAGO_SERVICE_ID_AIM,       _("AOL Instant Messenger"), 0);
	ADD_SERVICE_MAP(GALAGO_SERVICE_ID_GADUGADU,  _("Gadu-Gadu"),
					GALAGO_PRESERVE_SPACES);
	ADD_SERVICE_MAP(GALAGO_SERVICE_ID_GROUPWISE, _("Novell GroupWise"), 0);
	ADD_SERVICE_MAP(GALAGO_SERVICE_ID_ICQ,       _("ICQ"),
					GALAGO_PRESERVE_SPACES);
	ADD_SERVICE_MAP(GALAGO_SERVICE_ID_IRC,       _("Internet Relay Chat"),
					GALAGO_PRESERVE_SPACES);
	ADD_SERVICE_MAP(GALAGO_SERVICE_ID_JABBER,    _("Jabber"),
					GALAGO_PRESERVE_SPACES | GALAGO_STRIP_SLASH);
	ADD_SERVICE_MAP(GALAGO_SERVICE_ID_MSN,       _("MSN Messenger"),
					GALAGO_PRESERVE_SPACES);
	ADD_SERVICE_MAP(GALAGO_SERVICE_ID_NAPSTER,   _("Napster"), 0);
	ADD_SERVICE_MAP(GALAGO_SERVICE_ID_SILC,      _("SILC"), 0);
	ADD_SERVICE_MAP(GALAGO_SERVICE_ID_TREPIA,    _("Trepia"), 0);
	ADD_SERVICE_MAP(GALAGO_SERVICE_ID_YAHOO,     _("Yahoo! Messenger"),
					GALAGO_PRESERVE_SPACES);
	ADD_SERVICE_MAP(GALAGO_SERVICE_ID_ZEPHYR,    _("Zephyr"),
					GALAGO_PRESERVE_SPACES | GALAGO_PRESERVE_CASE);
}

static ServiceMapInfo *
_galago_services_map_id_to_info(const char *id)
{
	ServiceMapInfo *map_info;
	char *temp;

	galago_return_val_if_fail(id  != NULL, NULL);
	galago_return_val_if_fail(*id != '\0', NULL);

	if (service_id_map_table == NULL)
		_init_service_id_map_table();

	temp = galago_str_lower(id);

	map_info = (ServiceMapInfo *)galago_hash_table_lookup(service_id_map_table,
														  temp);
	free(temp);

	return map_info;
}

static GalagoService *
_galago_service_new_common(const char *id, const char *name,
						   galago_bool native, const char *obj_path,
						   GalagoServiceFlags flags)
{
	ServiceMapInfo *map_info;
	GalagoService *service;

	galago_return_val_if_fail(galago_is_initted(), NULL);
	galago_return_val_if_fail(id  != NULL,         NULL);
	galago_return_val_if_fail(*id != '\0',         NULL);

	if ((map_info = _galago_services_map_id_to_info(id)) != NULL)
	{
		name  = map_info->name;
		flags = map_info->flags;
	}

	galago_return_val_if_fail(name  != NULL, NULL);
	galago_return_val_if_fail(*name != '\0', NULL);

	service = galago_context_get_service(id, native);

	if (service == NULL)
	{
		const char *obj_prefix;

		service = galago_object_new(GALAGO_CLASS_SERVICE);

		service->priv->id     = strdup(id);
		service->priv->name   = strdup(name);
		service->priv->native = native;
		service->priv->flags  = flags;

		if (obj_path != NULL)
		{
			/* Assumes an escaped path already */
			galago_object_set_dbus_path(service, obj_path);
		}
		else if ((obj_prefix = galago_context_get_obj_path_prefix()) != NULL)
		{
			size_t len;
			char *new_obj_path;
			const char *escaped_id = galago_dbus_normalize_name(id);

			len = strlen(obj_prefix) + strlen("/services/") +
			      strlen(escaped_id) + 1;

			new_obj_path = galago_new(char, len);
			snprintf(new_obj_path, len, "%s/services/%s", obj_prefix,
					 escaped_id);

			galago_object_set_dbus_path(service, new_obj_path);

			free(new_obj_path);
		}

		galago_context_add_service(service);
	}

	return service;
}

GalagoService *
galago_service_new(const char *id, const char *name, galago_bool native,
				   GalagoServiceFlags flags)
{
	galago_return_val_if_fail(galago_is_initted(), NULL);
	galago_return_val_if_fail(id != NULL && *id != '\0', NULL);

	return _galago_service_new_common(id, name, native, NULL, flags);
}

const char *
galago_service_get_id(const GalagoService *service)
{
	galago_return_val_if_fail(service != NULL,            NULL);
	galago_return_val_if_fail(GALAGO_IS_SERVICE(service), NULL);

	return service->priv->id;
}

const char *
galago_service_get_name(const GalagoService *service)
{
	galago_return_val_if_fail(service != NULL,            NULL);
	galago_return_val_if_fail(GALAGO_IS_SERVICE(service), NULL);

	return service->priv->name;
}

galago_bool
galago_service_is_native(const GalagoService *service)
{
	galago_return_val_if_fail(service != NULL,            FALSE);
	galago_return_val_if_fail(GALAGO_IS_SERVICE(service), FALSE);

	return service->priv->native;
}

GalagoServiceFlags
galago_service_get_flags(const GalagoService *service)
{
	galago_return_val_if_fail(service != NULL,            0);
	galago_return_val_if_fail(GALAGO_IS_SERVICE(service), 0);

	return service->priv->flags;
}

GalagoAccount *
galago_service_get_account(const GalagoService *service, const char *username,
						   galago_bool query)
{
	GalagoAccount *account;

	galago_return_val_if_fail(service  != NULL,           NULL);
	galago_return_val_if_fail(username != NULL,           NULL);
	galago_return_val_if_fail(GALAGO_IS_SERVICE(service), NULL);

	account =
		galago_hash_table_lookup(service->priv->accounts_table,
								 galago_service_normalize(service, username));

	if (account == NULL && query &&
		!galago_service_is_native(service) && !galago_is_daemon() &&
		galago_is_connected())
	{
		GalagoSignalContext *signal_context = GALAGO_SIGNAL_CONTEXT(service);

		galago_context_push(galago_object_get_context(service));
		galago_signal_context_freeze(signal_context);

		account = galago_dbus_send_message_with_reply(service, "GetAccount",
			galago_value_new(GALAGO_TYPE_OBJECT, NULL, GALAGO_CLASS_ACCOUNT),
			galago_value_new(GALAGO_TYPE_STRING, &username, NULL),
			NULL);

		galago_signal_context_thaw(signal_context);
		galago_context_pop();
	}

	return account;
}

const GalagoList *
galago_service_get_accounts(const GalagoService *service, galago_bool query)
{
	galago_return_val_if_fail(service != NULL,            NULL);
	galago_return_val_if_fail(GALAGO_IS_SERVICE(service), NULL);

	if (query && !galago_service_is_native(service) && !galago_is_daemon() &&
		galago_is_connected())
	{
		GalagoSignalContext *signal_context = GALAGO_SIGNAL_CONTEXT(service);
		GalagoList *temp;

		galago_context_push(galago_object_get_context(service));
		galago_signal_context_freeze(signal_context);

		temp = galago_dbus_send_message_with_reply(service, "GetAccounts",
			galago_value_new_list(GALAGO_TYPE_OBJECT, NULL,
								  GALAGO_CLASS_ACCOUNT),
			NULL);
		galago_list_destroy(temp);

		galago_signal_context_thaw(signal_context);
		galago_context_pop();
	}

	return service->priv->accounts;
}

void
galago_service_add_account(GalagoService *service, GalagoAccount *account)
{
	const char *username;

	galago_return_if_fail(service != NULL);
	galago_return_if_fail(account != NULL);
	galago_return_if_fail(GALAGO_IS_SERVICE(service));
	galago_return_if_fail(GALAGO_IS_ACCOUNT(account));

	username = galago_account_get_username(account);

	if (galago_service_get_account(service, username, FALSE) != NULL)
	{
		galago_log_warning("An account with username %s has already been "
						   "added to service %s\n",
						   username, galago_service_get_id(service));
		return;
	}

	galago_hash_table_insert(service->priv->accounts_table,
		strdup(galago_service_normalize(service, username)),
		account);

	service->priv->accounts = galago_list_append(service->priv->accounts,
												 account);

	if (galago_service_is_native(service))
		_galago_dbus_service_add_account(service, account);

	galago_signal_emit(service, "account-added", account);
	galago_signal_emit(service, "updated");
}

void
galago_service_remove_account(GalagoService *service, GalagoAccount *account)
{
	const char *username;

	galago_return_if_fail(service != NULL);
	galago_return_if_fail(account != NULL);
	galago_return_if_fail(GALAGO_IS_SERVICE(service));
	galago_return_if_fail(GALAGO_IS_ACCOUNT(account));

	username = galago_account_get_username(account);

	galago_hash_table_remove(service->priv->accounts_table,
							 galago_service_normalize(service, username));

	service->priv->accounts = galago_list_remove(service->priv->accounts,
												 account);

	if (galago_service_is_native(service) && galago_is_connected() &&
		galago_core_is_feed())
	{
		galago_dbus_send_message(service, "RemoveAccount",
			galago_value_new(GALAGO_TYPE_OBJECT, &account,
							 GALAGO_CLASS_ACCOUNT),
			NULL);
	}

	galago_signal_emit(service, "account-removed", account);
	galago_signal_emit(service, "updated");
}

const char *
galago_service_normalize(const GalagoService *service, const char *username)
{
	static char buffer[BUFSIZ];
	const char *c;
	char *d;
	GalagoServiceFlags flags;

	galago_return_val_if_fail(service  != NULL,           NULL);
	galago_return_val_if_fail(username != NULL,           NULL);
	galago_return_val_if_fail(GALAGO_IS_SERVICE(service), NULL);

	flags = galago_service_get_flags(service);

	for (c = username, d = buffer; *c != '\0'; c++)
	{
		if (*c == ' ' && (flags & GALAGO_PRESERVE_SPACES) == 0)
		{
			while (*c == ' ')
				c++;
		}

		*d++ = *c;
	}

	*d = '\0';

	if ((flags & GALAGO_PRESERVE_CASE) == 0)
	{
		char *tmp = galago_str_lower(buffer);
		strncpy(buffer, tmp, sizeof(buffer));
		free(tmp);
	}

	if ((flags & GALAGO_STRIP_SLASH) == GALAGO_STRIP_SLASH &&
		(d = strchr(buffer, '/')) != NULL)
	{
		*d = '\0';
	}

	return buffer;
}


/**************************************************************************
 * D-BUS Functions
 **************************************************************************/
static void
_galago_dbus_service_add_account(GalagoService *service, GalagoAccount *account)
{
	if (!galago_is_connected() || !galago_core_is_feed())
		return;

	galago_dbus_send_message(service, "AddAccount",
		galago_value_new(GALAGO_TYPE_OBJECT, &account, GALAGO_CLASS_ACCOUNT),
		NULL);
}
