/**
 * @file libgalago/galago-value.c Value wrapper API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-value.h>
#include <libgalago/galago-assert.h>
#include <libgalago/galago-utils.h>
#include <string.h>

/**
 * A wrapper for a type of values.
 */
struct _GalagoValue
{
	GalagoType type;
	GalagoType subtype;
	void *detail;

	union
	{
		char char_data;
		unsigned char uchar_data;
		galago_bool boolean_data;
		short short_data;
		unsigned short ushort_data;
		int int_data;
		unsigned int uint_data;
		long long_data;
		unsigned long ulong_data;
		char *string_data;
		void *object_data;
		void *pointer_data;
		GalagoList *list_data;

		struct
		{
			const void *array;
			size_t size;

		} array_data;

	} data;
};

#define DEFINE_ACCESSOR(name, type, defaultval) \
void \
galago_value_set_##name(GalagoValue *value, type data) \
{ \
	galago_return_if_fail(value != NULL); \
	galago_return_if_fail(galago_value_get_type(value) != GALAGO_TYPE_LIST); \
\
	value->data.name##_data = data; \
} \
\
type \
galago_value_get_##name(const GalagoValue *value) \
{ \
	galago_return_val_if_fail(value != NULL, defaultval); \
	galago_return_val_if_fail(galago_value_get_type(value) != GALAGO_TYPE_LIST,\
							  defaultval); \
\
	return value->data.name##_data; \
}

#define DEFINE_ACCESSOR_MEM(name, type, defaultval, destroy, cpy) \
void \
galago_value_set_##name(GalagoValue *value, type data) \
{ \
	galago_return_if_fail(value != NULL); \
	galago_return_if_fail(galago_value_get_type(value) != GALAGO_TYPE_LIST); \
\
	if (value->data.name##_data != NULL) \
		(destroy)(value->data.name##_data); \
\
	value->data.name##_data = (data == NULL ? NULL : (cpy)(data)); \
} \
\
type \
galago_value_get_##name(const GalagoValue *value) \
{ \
	galago_return_val_if_fail(value != NULL, defaultval); \
	galago_return_val_if_fail(galago_value_get_type(value) != GALAGO_TYPE_LIST,\
							  defaultval); \
\
	return value->data.name##_data; \
}

GalagoValue *
galago_value_new(GalagoType type, const void *data, void *detail)
{
	GalagoValue *value;

	galago_return_val_if_fail(type != GALAGO_TYPE_UNKNOWN, NULL);

	if (type == GALAGO_TYPE_OBJECT &&
		(detail == NULL || !GALAGO_IS_CLASS(detail)))
	{
		galago_log_error("A valid class type must be passed for objects to "
						 "galago_value_new\n");
		return NULL;
	}

	value = galago_new0(GalagoValue, 1);
	value->type = type;
	value->detail = detail;

	if (data != NULL)
	{
#define CHECK_SET_TYPE(type, name, datatype) \
	case type: \
		galago_value_set_##name(value, *(datatype *)data); \
		break;

		switch (type)
		{
			CHECK_SET_TYPE(GALAGO_TYPE_CHAR,    char,    char);
			CHECK_SET_TYPE(GALAGO_TYPE_UCHAR,   uchar,   unsigned char);
			CHECK_SET_TYPE(GALAGO_TYPE_BOOLEAN, boolean, galago_bool);
			CHECK_SET_TYPE(GALAGO_TYPE_SHORT,   short,   short);
			CHECK_SET_TYPE(GALAGO_TYPE_USHORT,  ushort,  unsigned short);
			CHECK_SET_TYPE(GALAGO_TYPE_INT,     int,     int);
			CHECK_SET_TYPE(GALAGO_TYPE_UINT,    uint,    unsigned int);
			CHECK_SET_TYPE(GALAGO_TYPE_LONG,    long,    long);
			CHECK_SET_TYPE(GALAGO_TYPE_ULONG,   ulong,   unsigned long);
			CHECK_SET_TYPE(GALAGO_TYPE_STRING,  string,  const char *);
			CHECK_SET_TYPE(GALAGO_TYPE_OBJECT,  object,  void *);
			CHECK_SET_TYPE(GALAGO_TYPE_POINTER, pointer, void *);
			CHECK_SET_TYPE(GALAGO_TYPE_LIST,    list,    const GalagoList *);

			default:
				galago_value_destroy(value);
				return NULL;
		}
	}

	return value;
}

GalagoValue *
galago_value_new_list(GalagoType type, const GalagoList *list, void *detail)
{
	GalagoValue *value;

	galago_return_val_if_fail(type != GALAGO_TYPE_UNKNOWN, NULL);

	value = galago_value_new(GALAGO_TYPE_LIST, list, detail);
	value->subtype = type;

	if (type == GALAGO_TYPE_OBJECT &&
		(detail == NULL || !GALAGO_IS_CLASS(detail)))
	{
		galago_log_error("A valid class type must be passed for objects to "
						 "galago_value_new\n");
		return NULL;
	}

	return value;
}

GalagoValue *
galago_value_new_array(GalagoType type, const void *array, size_t size,
					   void *detail)
{
	GalagoValue *value;

	galago_return_val_if_fail(type != GALAGO_TYPE_UNKNOWN, NULL);

	value = galago_value_new(GALAGO_TYPE_ARRAY, NULL, detail);
	value->subtype = type;

	galago_value_set_array(value, array, size);

	return value;
}

void
galago_value_destroy(GalagoValue *value)
{
	GalagoType type;

	galago_return_if_fail(value != NULL);

	type = galago_value_get_type(value);

	if (type == GALAGO_TYPE_LIST && value->data.list_data != NULL)
	{
		galago_list_foreach(value->data.list_data,
							(GalagoForEachFunc)galago_value_destroy,
							NULL);
		galago_list_destroy(value->data.list_data);
	}

	free(value);
}

GalagoType
galago_value_get_type(const GalagoValue *value)
{
	galago_return_val_if_fail(value != NULL, GALAGO_TYPE_UNKNOWN);

	return value->type;
}

GalagoType
galago_value_get_subtype(const GalagoValue *value)
{
	galago_return_val_if_fail(value != NULL, GALAGO_TYPE_UNKNOWN);
	galago_return_val_if_fail(
		galago_value_get_type(value) == GALAGO_TYPE_LIST ||
		galago_value_get_type(value) == GALAGO_TYPE_ARRAY,
		GALAGO_TYPE_UNKNOWN);

	return value->subtype;
}

GalagoObjectClass *
galago_value_get_object_class(const GalagoValue *value)
{
	galago_return_val_if_fail(value != NULL, NULL);
	galago_return_val_if_fail(galago_value_get_type(value) ==
							  GALAGO_TYPE_OBJECT ||
							  galago_value_get_subtype(value) ==
							  GALAGO_TYPE_OBJECT,
							  NULL);

	return (GalagoObjectClass *)value->detail;
}

void
galago_value_set_list(GalagoValue *value, const GalagoList *list)
{
	const GalagoList *l;
	GalagoList *new_list = NULL;
	GalagoType subtype;

	galago_return_if_fail(value != NULL);
	galago_return_if_fail(galago_value_get_type(value) == GALAGO_TYPE_LIST);

	if (value->data.list_data != NULL)
	{
		galago_list_foreach(value->data.list_data,
							(GalagoForEachFunc)galago_value_destroy,
							NULL);
		galago_list_destroy(value->data.list_data);
	}

	subtype = galago_value_get_subtype(value);

	for (l = list; l != NULL; l = l->next)
	{
		new_list = galago_list_append(new_list,
									  galago_value_new(subtype, &l->data,
													   value->detail));
	}

	value->data.list_data = new_list;
}

const GalagoList *
galago_value_get_list(const GalagoValue *value)
{
	galago_return_val_if_fail(value != NULL, NULL);
	galago_return_val_if_fail(galago_value_get_type(value) == GALAGO_TYPE_LIST,
							  NULL);

	return value->data.list_data;
}

void
galago_value_set_array(GalagoValue *value, const void *data, size_t size)
{
	galago_return_if_fail(value != NULL);
	galago_return_if_fail(galago_value_get_type(value) == GALAGO_TYPE_ARRAY);

	value->data.array_data.size  = size;
	value->data.array_data.array = data;
}

void
galago_value_get_array(const GalagoValue *value, const void **ret_array,
					   size_t *ret_size)
{
	galago_return_if_fail(value != NULL);
	galago_return_if_fail(galago_value_get_type(value) == GALAGO_TYPE_ARRAY);

	if (ret_array != NULL)
		*ret_array = value->data.array_data.array;

	if (ret_size != NULL)
		*ret_size = value->data.array_data.size;
}

DEFINE_ACCESSOR(char,       char,           '\0')
DEFINE_ACCESSOR(uchar,      unsigned char,  '\0')
DEFINE_ACCESSOR(boolean,    galago_bool,    FALSE)
DEFINE_ACCESSOR(short,      short,          0)
DEFINE_ACCESSOR(ushort,     unsigned short, 0)
DEFINE_ACCESSOR(int,        int,            0)
DEFINE_ACCESSOR(uint,       unsigned int,   0)
DEFINE_ACCESSOR(long,       long,           0)
DEFINE_ACCESSOR(ulong,      unsigned long,  0)
DEFINE_ACCESSOR(object,     void *,         NULL)
DEFINE_ACCESSOR(pointer,    void *,         NULL)
DEFINE_ACCESSOR_MEM(string, const char *,   NULL, free, strdup)
