/**
 * @file get-presence.c Queries the presence for a user.
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA  02111-1307  USA
 */
#include <libgalago/galago.h>
#include <stdio.h>
#include <string.h>

static const char *
get_name_from_status_type(GalagoStatusType type)
{
	switch (type)
	{
		case GALAGO_STATUS_UNSET:         return "Unset";
		case GALAGO_STATUS_OFFLINE:       return "Offline";
		case GALAGO_STATUS_AVAILABLE:     return "Available";
		case GALAGO_STATUS_AWAY:          return "Away";
		case GALAGO_STATUS_EXTENDED_AWAY: return "Extended Away";
		case GALAGO_STATUS_HIDDEN:        return "Hidden";
		default:
			return NULL;
	};
}

int
main(int argc, char **argv)
{
	const GalagoList *l;
	char *username, *service_id;
	GalagoPresence *presence;
	GalagoStatus *status;
	GalagoAccount *account;
	GalagoService *service;
	GalagoPerson *person;

	if (argc != 3)
	{
		printf("Usage: get-presence service-id username\n");
		exit(1);
	}

	service_id = argv[1];
	username   = argv[2];

	galago_init("get-presence", FALSE);

	service = galago_core_get_service(service_id, FALSE, TRUE);

	if (service == NULL)
	{
		printf("Unknown service %s\n", service_id);
		exit(1);
	}

	account = galago_service_get_account(service, username, TRUE);

	if (account == NULL)
	{
		printf("Unknown account %s on service %s\n", username, service_id);
		exit(1);
	}

	presence = galago_account_get_presence(account, TRUE);

	person   = galago_account_get_person(account);

	printf("Service:      %s\n", galago_service_get_name(service));
	printf("Username:     %s\n", galago_account_get_username(account));
	printf("Display name: %s\n", galago_account_get_display_name(account));

	if (presence == NULL)
	{
		printf("No presence\n");
	}
	else
	{
		printf("Idle:         %d\n", galago_presence_is_idle(presence));
		printf("Idle Time:    %d\n",
			   (int)galago_presence_get_idle_time(presence));

		printf("Status(es):\n");

		for (l = galago_presence_get_statuses(presence);
			 l != NULL;
			 l = l->next)
		{
			status = (GalagoStatus *)l->data;

			printf("\tType:    %s\n",
				   get_name_from_status_type(galago_status_get_type(status)));
			printf("\tID:      %s\n", galago_status_get_id(status));
			printf("\tName:    %s\n", galago_status_get_name(status));
		}
	}

	return 0;
}
