#line 61 "../noweb/o_text_basic.nw"
/* gEDA - GPL Electronic Design Automation
 * libgeda - gEDA's library
 * Copyright (C) 1998-2000 Ales V. Hvezda
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA
 */


#line 11 "../noweb/o_text_basic.nw"
/* DO NOT read or edit this file ! Use ../noweb/o_text_basic.nw instead */

#line 84 "../noweb/o_text_basic.nw"
#include <config.h>

#include <stdio.h>
#include <math.h>
#include <sys/stat.h>
#include <assert.h>
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <gtk/gtk.h>
#include <guile/gh.h>

#include "defines.h"
#include "struct.h"
#include "globals.h"
#include "o_types.h"
#include "colors.h"
#include "funcs.h"

#include "../include/prototype.h"

#line 113 "../noweb/o_text_basic.nw"
#define WINONLY	1
#define BACKING 2

/* font storage and friends are staying global so that all can access */
#define NUM_CHARS 256

#line 123 "../noweb/o_text_basic.nw"
OBJECT font_set[NUM_CHARS];

/* Size of a tab in characters */
int tab_in_chars = 8;

#line 137 "../noweb/o_text_basic.nw"
void
get_text_bounds(TOPLEVEL * w_current, OBJECT * o_current, int *left,
		int *top, int *right, int *bottom)
{
  get_complex_bounds(w_current, o_current->text->prim_objs, left, top,
		     right, bottom);
}


#line 155 "../noweb/o_text_basic.nw"
void
world_get_text_bounds(TOPLEVEL * w_current, OBJECT * o_current, int *left,
		      int *top, int *right, int *bottom)
{
  world_get_complex_bounds(w_current, o_current->text->prim_objs,
			   left, top, right, bottom);
}


#line 173 "../noweb/o_text_basic.nw"
OBJECT *o_text_add_head(void)
{
  OBJECT *new_node = NULL;

  new_node = s_basic_init_object("text_head");
  new_node->type = OBJ_HEAD;

  /* don't need to do this for head nodes */
  /* ret = s_basic_link_object(new_node, NULL); */
  return (new_node);
}


#line 196 "../noweb/o_text_basic.nw"
void o_text_init(void)
{
  int i;

  for (i = 0; i < NUM_CHARS; i++) {
    font_set[i].font_prim_objs = NULL;
    font_set[i].font_text_size = 100;
  }
}


#line 217 "../noweb/o_text_basic.nw"
void o_text_print_set(void)
{
  OBJECT *o_current;
  int i;

  for (i = 'A'; i < 'Z' + 1; i++) {
    if (font_set[i].font_prim_objs != NULL) {
      printf("%c: LOADED\n", i);
      /* for (o_current=font_set[i].font_prim_objs; o_current; 
         o_current=o_current->next) */
      for (o_current = return_tail(font_set[i].font_prim_objs); o_current;
	   o_current = o_current->prev) {
	printf("  %s\n", o_current->name);
      }
    } else {
      printf("%c: unloaded\n", i);
    }
  }
}


#line 249 "../noweb/o_text_basic.nw"
OBJECT *o_text_load_font(TOPLEVEL * w_current, unsigned char needed_char)
{
  char *temp_string = NULL;
  OBJECT *temp_parent;
  int not_found = FALSE;

  switch (needed_char) {

  case (' '):
    temp_string = g_strdup_printf("%s%cspace.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('\n'):
    temp_string = g_strdup_printf("%s%cnewline.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);

  case ('!'):
    temp_string = g_strdup_printf("%s%cexcl.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (','):
    temp_string = g_strdup_printf("%s%ccomma.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('('):
    temp_string = g_strdup_printf("%s%clparen.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (')'):
    temp_string = g_strdup_printf("%s%crparen.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('-'):
    temp_string = g_strdup_printf("%s%cminus.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('+'):
    temp_string = g_strdup_printf("%s%cplus.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('#'):
    temp_string = g_strdup_printf("%s%cpound.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('?'):
    temp_string = g_strdup_printf("%s%cquest.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('"'):
    temp_string = g_strdup_printf("%s%cquote.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (':'):
    temp_string = g_strdup_printf("%s%ccolon.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('@'):
    temp_string = g_strdup_printf("%s%cat.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('='):
    temp_string = g_strdup_printf("%s%cequal.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('>'):
    temp_string = g_strdup_printf("%s%cmore.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('<'):
    temp_string = g_strdup_printf("%s%cless.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('/'):
    temp_string = g_strdup_printf("%s%cslash.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('$'):
    temp_string = g_strdup_printf("%s%cdollar.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (';'):
    temp_string = g_strdup_printf("%s%csemi.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('&'):
    temp_string = g_strdup_printf("%s%camper.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('\\'):
    temp_string = g_strdup_printf("%s%cbackslash.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('{'):
    temp_string = g_strdup_printf("%s%clbrace.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('}'):
    temp_string = g_strdup_printf("%s%crbrace.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('\''):
    temp_string = g_strdup_printf("%s%capost.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('`'):
    temp_string = g_strdup_printf("%s%cbacktick.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('^'):
    temp_string = g_strdup_printf("%s%ccaret.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('%'):
    temp_string = g_strdup_printf("%s%cpercent.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('['):
    temp_string = g_strdup_printf("%s%clbrack.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (']'):
    temp_string = g_strdup_printf("%s%crbrack.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('*'):
    temp_string = g_strdup_printf("%s%castericks.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('.'):
    temp_string = g_strdup_printf("%s%cperiod.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('_'):
    temp_string = g_strdup_printf("%s%cunder.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('~'):
    temp_string = g_strdup_printf("%s%ctilde.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case ('|'):
    temp_string = g_strdup_printf("%s%cvbar.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (196):			/* A-umlaut finnish/swedish/german */
    temp_string = g_strdup_printf("%s%cA-diaeresis.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (197):			/* A-ring finnish/swedish/danish/norwegian */
    temp_string = g_strdup_printf("%s%cA-ring.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (198):			/* AE-diphtong danish/norwegian */
    temp_string = g_strdup_printf("%s%cAE-lig.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (214):			/* O-umlaut finnish/swedish/german */
    temp_string = g_strdup_printf("%s%cO-diaeresis.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (216):			/* O-slash danish/norwegian */
    temp_string = g_strdup_printf("%s%cO-slash.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (220):			/* U-umlaut german */
    temp_string = g_strdup_printf("%s%cU-diaeresis.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (228):			/* a-umlaut finnish/swedish/german */
    temp_string = g_strdup_printf("%s%ca_-diaeresis.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (229):			/* a-ring finnish/swedish/danish/norwegian */
    temp_string = g_strdup_printf("%s%ca_-ring.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (230):			/* ae-diphtong danish/norwegian */
    temp_string = g_strdup_printf("%s%cae_-lig.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (246):			/* o-umlaut finnish/swedish/german */
    temp_string = g_strdup_printf("%s%co_-diaeresis.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (248):			/* o-slash danish/norwegian */
    temp_string = g_strdup_printf("%s%co_-slash.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (252):			/* u-umlaut german */
    temp_string = g_strdup_printf("%s%cu_-diaeresis.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (225):			/* a-acute_accent spanish */
    temp_string = g_strdup_printf("%s%ca_-acute-accent.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (233):			/* e-acute_accent spanish */
    temp_string = g_strdup_printf("%s%ce_-acute-accent.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (237):			/* i-acute_accent spanish */
    temp_string = g_strdup_printf("%s%ci_-acute-accent.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (243):			/* o-acute_accent spanish */
    temp_string = g_strdup_printf("%s%co_-acute-accent.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (250):			/* u-acute_accent spanish */
    temp_string = g_strdup_printf("%s%cu_-acute-accent.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (193):			/* A-acute_accent spanish */
    temp_string = g_strdup_printf("%s%cA-acute-accent.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (201):			/* E-acute_accent spanish */
    temp_string = g_strdup_printf("%s%cE-acute-accent.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (205):			/* I-acute_accent spanish */
    temp_string = g_strdup_printf("%s%cI-acute-accent.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (211):			/* O-acute_accent spanish */
    temp_string = g_strdup_printf("%s%cO-acute-accent.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (218):			/* U-acute_accent spanish */
    temp_string = g_strdup_printf("%s%cU-acute-accent.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (241):			/* n-tilde spanish */
    temp_string = g_strdup_printf("%s%cn_-tilde.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (209):			/* N-tilde spanish */
    temp_string = g_strdup_printf("%s%cN-tilde.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (161):			/* open exclamation spanish */
    temp_string = g_strdup_printf("%s%cexcl-open.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  case (191):			/* open question spanish */
    temp_string = g_strdup_printf("%s%cquest-open.sym",
				  w_current->font_directory,
				  PATH_SEPARATER_CHAR);
    break;

  default:
    /* this is needed since WinNT file systems are 
     * case insensitive, and cannot tell the difference 
     * between A.sym and a.sym.  So we create a_.sym -  
     * z_.sym, this loads up the chars 
     */
    if (needed_char >= 'a' && needed_char <= 'z') {
      temp_string = g_strdup_printf("%s%c%c_.sym",
				    w_current->font_directory,
				    PATH_SEPARATER_CHAR, needed_char);
    } else {
      temp_string = g_strdup_printf("%s%c%c.sym",
				    w_current->font_directory,
				    PATH_SEPARATER_CHAR, needed_char);
    }
    break;

  }

  if (access(temp_string, R_OK) != 0) {
    s_log_message("Could not find character %c definition\n", needed_char,
		  temp_string);
    free(temp_string);
    temp_string =
	g_strdup_printf("%s%cquest.sym", w_current->font_directory,
			PATH_SEPARATER_CHAR);
    if (access(temp_string, R_OK) != 0) {
      fprintf(stderr,
	      "Could not load question font char -- check font-directory keyword\n");
      exit(-1);
    }
    not_found = TRUE;
  }

  /* printf("loading: %s\n", temp_string); */

  font_set[(unsigned char) needed_char].font_prim_objs = o_text_add_head();

  temp_parent = w_current->page_current->object_parent;
  /* set the addition of attributes to the head node */
  w_current->page_current->object_parent =
      font_set[(unsigned char) needed_char].font_prim_objs;

  font_set[(unsigned char) needed_char].font_prim_objs =
      o_read(w_current,
	     font_set[(unsigned char) needed_char].font_prim_objs,
	     temp_string);
  w_current->page_current->object_parent = temp_parent;

  font_set[(unsigned char) needed_char].font_prim_objs =
      return_head(font_set[(unsigned char) needed_char].font_prim_objs);

  if (not_found == TRUE) {
    /* set the font text size (width) to the question mark's size */
    /* yes, the question mark character was loaded instead of the real char */
    /* 63 == question mark character */
    font_set[(unsigned char) needed_char].font_text_size =
	font_set[63].font_text_size;
  }

  return (font_set[(unsigned char) needed_char].font_prim_objs);
}


#line 624 "../noweb/o_text_basic.nw"
int o_text_num_lines(char *string)
{
  int i, line_count = 0;

  if (string == NULL) {
    return 0;
  }

  /* if it's not null, then we have at least one line */
  line_count++;
  /* Count how many \n are in the string */
  for (i = 0; i <= strlen(string); i++) {
    if (string[i] == '\n')
      line_count++;
  }

  return (line_count);
}


#line 655 "../noweb/o_text_basic.nw"
int o_text_height(char *string, int size)
{
  int line_count = 0;

  if (string == NULL) {
    return 0;
  }

  /* Get the number of lines in the string */
  line_count = o_text_num_lines(string);

  /* 26 is the height of a single char (in mils) */
  /* which represents a character which is 2 pts high */
  /* So size has to be divided in half */
  /* and it's added the LINE_SPACING*character_height of each line */
  return (26 * size / 2 * (1 + LINE_SPACING * (line_count - 1)));
}


#line 685 "../noweb/o_text_basic.nw"
int o_text_width(TOPLEVEL * w_current, char *string, int size)
{
  int i;
  int len;
  int width = 0, max_width = 0;
  int size_of_tab_in_coord;

  /* Make sure TAB_CHAR_MODEL is loaded before trying to use its text */
  /* size */
  if (font_set[(unsigned char) TAB_CHAR_MODEL[0]].font_prim_objs == NULL) {
    o_text_load_font(w_current, (unsigned char) TAB_CHAR_MODEL[0]);
  }
  /* Get the maximum tab width's in coordinates */
  size_of_tab_in_coord = tab_in_chars * size *
      font_set[(unsigned char) TAB_CHAR_MODEL[0]].font_text_size;

  if (string == NULL) {
    return 0;
  }
  len = strlen(string);
  for (i = 0; i < len; i++) {
    if (string[i] == '\n') {
      width = 0;
      continue;
    }
    if (string[i] == '\t') {
      width += (size_of_tab_in_coord - (width % size_of_tab_in_coord));
      continue;

    }

    if (font_set[(unsigned char) string[i]].font_prim_objs == NULL) {
      o_text_load_font(w_current, (unsigned char) string[i]);
    }

    width =
	width + size * font_set[(unsigned char) string[i]].font_text_size;
    if (width > max_width)
      max_width = width;
  }

  /* the size is a fudge factor */
  /* Changed the return value according to a suggestion of Ales. */
  /* Yes, the -size*10 fudge factor should be removed. */
  /* return(max_width - size*10); */
  return (max_width);
}


#line 743 "../noweb/o_text_basic.nw"
OBJECT *o_text_create_string(TOPLEVEL * w_current, OBJECT * object_list,
			     char *string, int size, int color, int x,
			     int y, int alignment, int angle)
{
  OBJECT *temp_tail = NULL;
  OBJECT *temp_list;
  OBJECT *start_of_char;
  int i;
  int len;
  int x_offset;
  int y_offset;
  int text_width;
  int text_height;
  int char_height;
  int line_start_x, line_start_y;
  int sign = 1;

  temp_list = object_list;


  /* error condition hack */
  if (string == NULL) {
    return (NULL);
  }

  len = strlen(string);
  /* now read in the chars */
  temp_tail = w_current->page_current->object_tail;

  text_height = o_text_height(string, size);
  char_height = o_text_height("a", size);
  text_width = o_text_width(w_current, string, size / 2);

  switch (angle) {
  case (0):
    sign = -1;
    break;
  case (90):
    sign = 1;
    break;
  case (180):
    sign = 1;
    break;
  case (270):
    sign = -1;
    break;
  }

  if (angle == 0 || angle == 180) {
    y = y - o_text_height("a", size) + text_height;

    switch (alignment) {

    case (LOWER_LEFT):
      x_offset = x;
      y_offset = y;
      break;

    case (MIDDLE_LEFT):
      x_offset = x;
      y_offset = y + sign * 0.5 * text_height;
      break;

    case (UPPER_LEFT):
      x_offset = x;
      y_offset = y + sign * text_height;
      break;

    case (LOWER_MIDDLE):
      x_offset = x + sign * 0.5 * text_width;
      y_offset = y;
      break;

    case (MIDDLE_MIDDLE):
      x_offset = x + sign * 0.5 * text_width;
      y_offset = y + sign * 0.5 * text_height;
      break;

    case (UPPER_MIDDLE):
      x_offset = x + sign * 0.5 * text_width;
      y_offset = y + sign * text_height;

      break;

    case (LOWER_RIGHT):
      x_offset = x + sign * text_width;
      y_offset = y;
      break;

    case (MIDDLE_RIGHT):
      x_offset = x + sign * text_width;
      y_offset = y + sign * 0.5 * text_height;
      break;

    case (UPPER_RIGHT):
      x_offset = x + sign * text_width;
      y_offset = y + sign * text_height;
      break;

    default:
      fprintf(stderr, "Got an invalid text alignment [%d]\n", alignment);
      fprintf(stderr, "Defaulting to Lower Left");
      alignment = LOWER_LEFT;
      x_offset = x;
      y_offset = y;
      break;
    }
  } else {			/* angle is 90 or 270 */
    x = x + sign * (o_text_height("a", size) - text_height);

    switch (alignment) {

    case (LOWER_LEFT):
      x_offset = x;
      y_offset = y;
      break;

    case (MIDDLE_LEFT):
      x_offset = x + sign * 0.5 * text_height;
      y_offset = y;
      break;

    case (UPPER_LEFT):
      x_offset = x + sign * text_height;
      y_offset = y;
      break;

    case (LOWER_MIDDLE):
      x_offset = x;
      y_offset = y - sign * 0.5 * text_width;
      break;

    case (MIDDLE_MIDDLE):
      x_offset = x + sign * 0.5 * text_height;
      y_offset = y - sign * 0.5 * text_width;
      break;

    case (UPPER_MIDDLE):
      x_offset = x + sign * text_height;
      y_offset = y - sign * 0.5 * text_width;

      break;

    case (LOWER_RIGHT):
      x_offset = x;
      y_offset = y - sign * text_width;
      break;

    case (MIDDLE_RIGHT):
      x_offset = x + sign * 0.5 * text_height;
      y_offset = y - sign * text_width;
      break;

    case (UPPER_RIGHT):
      x_offset = x + sign * text_height;
      y_offset = y - sign * text_width;
      break;

    default:
      fprintf(stderr, "Got an invalid text alignment [%d]\n", alignment);
      fprintf(stderr, "Defaulting to Lower Left");
      alignment = LOWER_LEFT;
      x_offset = x;
      y_offset = y;
      break;
    }

  }

  switch (angle) {
  case (180):
    x_offset = x_offset - text_width;
    y_offset = y_offset - text_height;
    angle = 0;
    break;
  }

#if DEBUG
  printf("width: %d\n", o_text_width(w_current, string, size / 2));
  printf("1 %d %d\n", x_offset, y_offset);
#endif

  line_start_x = x_offset;
  line_start_y = y_offset;

  for (i = 0; i < len; i++) {

    if (font_set[(unsigned char) string[i]].font_prim_objs == NULL) {
      o_text_load_font(w_current, (unsigned char) string[i]);
    }

    start_of_char = temp_list;

    if (font_set[(unsigned char) string[i]].font_prim_objs->next) {
      int rel_char_coord;
      int size_of_tab_in_coord;

      /* Make sure TAB_CHAR_MODEL is loaded before trying to use its text */
      /* size */
      if (font_set[(unsigned char) TAB_CHAR_MODEL[0]].font_prim_objs ==
	  NULL) {
	o_text_load_font(w_current, (unsigned char) TAB_CHAR_MODEL[0]);
      }
      /* Get the maximum tab width's in coordinates */
      size_of_tab_in_coord = tab_in_chars *
	  o_text_width(w_current, TAB_CHAR_MODEL, size / 2);

      if (string[i] != '\n' && string[i] != '\t') {
	/* only add the character if it is not a newline or tab character */
	temp_list = o_list_copy_all(w_current,
				    font_set[(unsigned char) string[i]].
				    font_prim_objs->next, temp_list,
				    NORMAL_FLAG);
	start_of_char = start_of_char->next;
      }

      /* if the character is a newline or tab, this code will "continue" */
      switch (string[i]) {
      case '\n':
	switch (angle) {
	case 0:
	  x_offset = line_start_x;
	  y_offset = line_start_y - char_height * LINE_SPACING;
	  line_start_x = x_offset;
	  line_start_y = y_offset;
	  continue;
	  break;
	case 90:
	  x_offset = line_start_x + char_height * LINE_SPACING;
	  y_offset = line_start_y;
	  line_start_x = x_offset;
	  line_start_y = y_offset;
	  continue;
	  break;
	case 180:
	  x_offset = line_start_x;
	  y_offset = line_start_y + char_height * LINE_SPACING;
	  line_start_x = x_offset;
	  line_start_y = y_offset;
	  continue;
	  break;
	case 270:
	  x_offset = line_start_x - char_height * LINE_SPACING;
	  y_offset = line_start_y;
	  line_start_x = x_offset;
	  line_start_y = y_offset;
	  continue;
	  break;
	default:
	  fprintf(stderr, "o_text_create_string: Angle not supported\n");
	  break;
	}
      case '\t':
#if DEBUG
	printf("Found tab character.\n");
	printf("Tab size in coord: %i\n", size_of_tab_in_coord);
	printf("Line start: %i,%i\n", line_start_x, line_start_y);
	printf("Position: %i, %i\n", x_offset, y_offset);
#endif
	switch (angle) {
	case 0:
	case 180:
	  rel_char_coord = x_offset - line_start_x;
#if DEBUG
	  printf("Add: %i\n",
		 (size_of_tab_in_coord -
		  (rel_char_coord % size_of_tab_in_coord)));
#endif
	  x_offset +=
	      (size_of_tab_in_coord -
	       (rel_char_coord % size_of_tab_in_coord));
	  continue;
	  break;
	case 90:
	  rel_char_coord = y_offset - line_start_y;
#if DEBUG
	  printf("Add: %i\n",
		 (size_of_tab_in_coord -
		  (rel_char_coord % size_of_tab_in_coord)));
#endif
	  y_offset +=
	      (size_of_tab_in_coord -
	       (rel_char_coord % size_of_tab_in_coord));
	  continue;
	  break;
	case 270:
	  rel_char_coord = line_start_y - y_offset;
#if DEBUG
	  printf("Add: %i\n",
		 (size_of_tab_in_coord -
		  (rel_char_coord % size_of_tab_in_coord)));
#endif
	  y_offset -=
	      (size_of_tab_in_coord -
	       (rel_char_coord % size_of_tab_in_coord));
	  continue;
	  break;
	default:
	  fprintf(stderr, "o_text_create_string: Angle not supported\n");
	  break;
	}
      }

      /* does not get here if the character was a newline or tab.  This is */
      /* correct. */
      o_complex_set_color(start_of_char, color);
      o_scale(w_current, start_of_char, size / 2, size / 2);

      /* do this if you want to stack chars */
      /* we don't want to do that for now */
      o_text_rotate_lowlevel(w_current, x, y, angle, start_of_char);
      o_complex_world_translate(w_current,
				x_offset, y_offset, start_of_char);
    }

    switch (angle) {
    case (0):
      x_offset = (x_offset) +
	  size / 2 * font_set[(unsigned char) string[i]].font_text_size;
      break;

    case (90):
      y_offset = (y_offset) +
	  size / 2 * font_set[(unsigned char) string[i]].font_text_size;
      break;

    case (180):
      x_offset = (x_offset) -
	  size / 2 * font_set[(unsigned char) string[i]].font_text_size;
      break;

    case (270):
      y_offset = (y_offset) -
	  size / 2 * font_set[(unsigned char) string[i]].font_text_size;
      break;
    }
  }

  /* don't set the head */

  w_current->page_current->object_tail = temp_tail;

#if DEBUG
  printf("2 %d %d\n", x_offset, y_offset);
#endif
  return (object_list);
}


#line 1091 "../noweb/o_text_basic.nw"
OBJECT *o_text_add(TOPLEVEL * w_current, OBJECT * object_list,
		   char type, int color, int x, int y, int alignment,
		   int angle, char *string, int size,
		   int visibility, int show_name_value)
{
  OBJECT *new_node = NULL;
  OBJECT *temp_list = NULL;
  OBJECT *temp_parent = NULL;
  TEXT *text;
  int left, right, top, bottom;
  char *name = NULL;
  char *value = NULL;
  char *output_string = NULL;

  if (string == NULL) {
    return (NULL);
  }

  new_node = s_basic_init_object("text");
  new_node->type = type;

  text = (TEXT *) malloc(sizeof(TEXT));

  text->string = u_basic_strdup(string);
  text->length = strlen(string);
  text->size = size;
  text->alignment = alignment;
  text->x = x;
  text->y = y;
  WORLDtoSCREEN(w_current, x, y, &text->screen_x, &text->screen_y);
  text->angle = angle;

  new_node->text = text;

  /* TODO: questionable cast */
  new_node->draw_func = (void *) text_draw_func;
  /* TODO: questionable cast */
  new_node->sel_func = (void *) select_func;

  new_node->color = color;
  new_node->visibility = visibility;
  new_node->show_name_value = show_name_value;

  /* create the object in the main list */
  /* object_list points to the object */
  /* I use it below as a sanity check to make sure it was linked */
  /* properly */
  object_list = (OBJECT *) s_basic_link_object(new_node, object_list);

  /* fix up actual string here */
  if (o_attrib_get_name_value(string, &name, &value)) {

    switch (show_name_value) {
    case (SHOW_NAME_VALUE):
      output_string = g_strdup(string);
      break;

    case (SHOW_NAME):
      if (name[0] != '\0') {
	output_string = g_strdup(name);
      } else {
	/* you probably can remove this now... */
	/* since improper attributes will never get here */
	fprintf(stderr, "Got an improper attribute: %s\n", string);
	output_string = g_strdup("invalid");

      }
      break;

    case (SHOW_VALUE):
      if (value[0] != '\0') {
	output_string = g_strdup(value);
      } else {
	/* you probably can remove this now... */
	/* since improper attributes will never get here */
	fprintf(stderr, "Got an improper attribute: %s\n", string);
	output_string = g_strdup("invalid");
      }
      break;
    }
  } else {
    output_string = g_strdup(string);
  }


  /* now start working on the complex */
  temp_list = o_text_add_head();

  temp_parent = w_current->page_current->object_parent;
  /* set the addition of attributes to the head node */
  w_current->page_current->object_parent = temp_list;

  if (visibility == VISIBLE ||
      (visibility == INVISIBLE && w_current->show_hidden_text)) {
    object_list->text->prim_objs =
	o_text_create_string(w_current, temp_list,
			     output_string, size, color,
			     x, y, alignment, angle);
    object_list->text->displayed_width = o_text_width(w_current,
						      output_string,
						      size / 2);
    object_list->text->displayed_height =
	o_text_height(output_string, size);
  } else {
    object_list->text->prim_objs = NULL;
    object_list->text->displayed_width = 0;
    object_list->text->displayed_height = 0;
    s_delete(w_current, temp_list);
  }

  w_current->page_current->object_parent = temp_parent;

  get_text_bounds(w_current, object_list, &left, &top, &right, &bottom);

  /* set the new object's bounding box */
  object_list->left = left;
  object_list->top = top;
  object_list->right = right;
  object_list->bottom = bottom;

  if (name)
    free(name);
  if (value)
    free(value);
  if (output_string)
    free(output_string);
  return (object_list);
}


#line 1230 "../noweb/o_text_basic.nw"
void o_text_recalc(TOPLEVEL * w_current, OBJECT * o_current)
{
  int left, right, top, bottom;

  if (o_current->visibility == INVISIBLE && !w_current->show_hidden_text) {
    return;
  }

  get_complex_bounds(w_current, o_current->text->prim_objs,
		     &left, &top, &right, &bottom);
  o_current->left = left;
  o_current->top = top;
  o_current->right = right;
  o_current->bottom = bottom;

  WORLDtoSCREEN(w_current, o_current->text->x,
		o_current->text->y,
		&o_current->text->screen_x, &o_current->text->screen_y);
}


#line 1262 "../noweb/o_text_basic.nw"
OBJECT *o_text_read(TOPLEVEL * w_current, OBJECT * object_list,
		    char buf[], FILE * fp, unsigned int release_ver,
		    unsigned int fileformat_ver)
{
  char type;
  int x, y;
  int color;
  int size;
  int visibility;
  int show_name_value;
  int angle;
  int alignment;
  int num_lines = 0;
  int i;
  char *string = NULL;
  char *temp = NULL;
  char buffer[MAX_TEXT_LINE_LENGTH];

  if (fileformat_ver == 1) {
    sscanf(buf, "%c %d %d %d %d %d %d %d %d %d\n", &type, &x, &y,
	   &color, &size,
	   &visibility, &show_name_value, &angle, &alignment, &num_lines);
  } else if (release_ver < VERSION_20000220) {
    /* yes, above less than (not less than and equal) is correct. The format */
    /* change occurred in 20000220 */
    sscanf(buf, "%c %d %d %d %d %d %d %d\n", &type, &x, &y,
	   &color, &size, &visibility, &show_name_value, &angle);
    alignment = LOWER_LEFT;	/* older versions didn't have this */
    num_lines = 1;		/* only support a single line */
  } else {
    sscanf(buf, "%c %d %d %d %d %d %d %d %d\n", &type, &x, &y,
	   &color, &size,
	   &visibility, &show_name_value, &angle, &alignment);
    num_lines = 1;		/* only support a single line */
  }

  if (size == 0) {
    fprintf(stderr,
	    "Found a zero size text string [ %c %d %d %d %d %d %d %d %d ]\n",
	    type, x, y, color, size, visibility, show_name_value, angle,
	    alignment);
    s_log_message
	("Found a zero size text string [ %c %d %d %d %d %d %d %d %d ]\n",
	 type, x, y, color, size, visibility, show_name_value, angle,
	 alignment);
  }

  switch (angle) {

  case (0):
  case (90):
  case (180):
  case (270):
    break;

  default:
    fprintf(stderr,
	    "Found an unsupported text angle [ %c %d %d %d %d %d %d %d %d ]\n",
	    type, x, y, color, size, visibility, show_name_value, angle,
	    alignment);
    s_log_message
	("Found an unsupported text angle [ %c %d %d %d %d %d %d %d %d ]\n",
	 type, x, y, color, size, visibility, show_name_value, angle,
	 alignment);
    s_log_message("Setting angle to 0\n");
    angle = 0;
    break;

  }

  switch (alignment) {
  case (LOWER_LEFT):
  case (MIDDLE_LEFT):
  case (UPPER_LEFT):
  case (LOWER_MIDDLE):
  case (MIDDLE_MIDDLE):
  case (UPPER_MIDDLE):
  case (LOWER_RIGHT):
  case (MIDDLE_RIGHT):
  case (UPPER_RIGHT):

    break;

  default:
    fprintf(stderr,
	    "Found an unsupported text alignment [ %c %d %d %d %d %d %d %d %d ]\n",
	    type, x, y, color, size, visibility, show_name_value, angle,
	    alignment);
    s_log_message
	("Found an unsupported text alignment [ %c %d %d %d %d %d %d %d %d ]\n",
	 type, x, y, color, size, visibility, show_name_value, angle,
	 alignment);
    s_log_message("Setting alignment to LOWER_LEFT\n");
    alignment = LOWER_LEFT;
    break;
  }

  if (color < 0 || color > MAX_COLORS) {
    fprintf(stderr, "Found an invalid color [ %s ]\n", buf);
    s_log_message("Found an invalid color [ %s ]\n", buf);
    s_log_message("Setting color to WHITE\n");
    color = WHITE;
  }
#ifdef HAS_GTK12
  if (num_lines > 1) {
    s_log_message
	("Found a multi-line text item which is fully supported when using gtk+ 2.2.x.\n");
  }
#endif

  assert(num_lines && num_lines > 0);
  for (i = 0; i < num_lines; i++) {
    fgets(buffer, 1024, fp);

    if (string == 0) {
      string = u_basic_strdup(buffer);
    } else {
      temp = u_basic_strdup_multiple(string, buffer, NULL);
      free(string);
      string = temp;
    }
  }

  string = remove_last_nl(string);
  object_list = o_text_add(w_current, object_list, type, color, x, y,
			   alignment, angle, string,
			   size, visibility, show_name_value);
  free(string);

  return (object_list);
}


#line 1392 "../noweb/o_text_basic.nw"
void o_text_set_info_font(char buf[])
{
  char type;
  int width;
  char character;
  unsigned int temp;
  int special = 0;
  char *string;

  string = g_strdup(remove_nl(buf));

  /* the right way to do this is to use the ascii code in the character
   * field hack
   */
  sscanf(string, "%c %c %d %d\n", &type, &character, &width, &special);

  /* space */
  if (special == 1 && character == '_') {
    character = 32;
  }

  /* newline */
  if (special == 1 && character == 'n') {
    character = 10;
  }

  temp = (unsigned char) character;
  if (temp >= 0 && temp <= 255) {
    font_set[(unsigned char) character].font_text_size = width;
  }

  free(string);
}


#line 1437 "../noweb/o_text_basic.nw"
char *o_text_save(OBJECT * object)
{
  int x, y;
  int color;
  int size;
  char *string;
  char *buf;
  int num_lines;

  x = object->text->x;
  y = object->text->y;

  string = object->text->string;
  size = object->text->size;

  /* Use the right color */
  if (object->saved_color == -1) {
    color = object->color;
  } else {
    color = object->saved_color;
  }

  /* string can have multiple lines (seperated by \n's) */
  num_lines = o_text_num_lines(string);

  buf = g_strdup_printf("%c %d %d %d %d %d %d %d %d %d\n%s", object->type,
			x, y, color, size, object->visibility,
			object->show_name_value, object->text->angle,
			object->text->alignment, num_lines, string);

  return (buf);
}


#line 1481 "../noweb/o_text_basic.nw"
void o_text_recreate(TOPLEVEL * w_current, OBJECT * o_current)
{
  OBJECT *temp_parent;
  char *name = NULL;
  char *value = NULL;
  char *output_string = NULL;

  if (o_attrib_get_name_value(o_current->text->string, &name, &value)) {
    switch (o_current->show_name_value) {
    case (SHOW_NAME_VALUE):
      output_string = g_strdup(o_current->text->string);
      break;

    case (SHOW_NAME):
      if (name[0] != '\0') {
	output_string = g_strdup(name);
      } else {
	/* you probably can remove this now... */
	/* since improper attributes will never get here */
	fprintf(stderr,
		"Got an improper attribute: %s\n",
		o_current->text->string);
	output_string = g_strdup("invalid");
      }
      break;

    case (SHOW_VALUE):
      if (value[0] != '\0') {
	output_string = g_strdup(value);
      } else {
	/* you probably can remove this now... */
	/* since improper attributes will never get here */
	fprintf(stderr,
		"Got an improper attribute: %s\n",
		o_current->text->string);
	output_string = g_strdup("invalid");
      }
      break;
    }
  } else {
    output_string = g_strdup(o_current->text->string);
  }

  o_list_delete_rest(w_current, o_current->text->prim_objs);

  temp_parent = w_current->page_current->object_parent;
  /* set the addition of attributes to the head node */
  w_current->page_current->object_parent = o_current->text->prim_objs;

  if (o_current->visibility == VISIBLE ||
      (o_current->visibility == INVISIBLE
       && w_current->show_hidden_text)) {

    /* need to create that head node if complex is null */
    if (o_current->text->prim_objs == NULL) {
      o_current->text->prim_objs = o_text_add_head();
    }

    o_current->text->prim_objs =
	o_text_create_string(w_current,
			     o_current->text->prim_objs,
			     output_string,
			     o_current->text->size,
			     o_current->color,
			     o_current->text->x,
			     o_current->text->y,
			     o_current->text->alignment,
			     o_current->text->angle);

    o_complex_set_saved_color_only(o_current->text->prim_objs,
				   o_current->saved_color);
    o_current->text->displayed_width = o_text_width(w_current,
						    output_string,
						    o_current->text->size /
						    2);
    o_current->text->displayed_height =
	o_text_height(output_string, o_current->text->size);
  } else {
    /* make sure list is truely free */
    s_delete_list_fromstart(w_current, o_current->text->prim_objs);
    o_current->text->prim_objs = NULL;
    o_current->text->displayed_width = 0;
    o_current->text->displayed_height = 0;
  }

  w_current->page_current->object_parent = temp_parent;
  if (name)
    free(name);
  if (value)
    free(value);
  if (output_string)
    free(output_string);
}


#line 1581 "../noweb/o_text_basic.nw"
void
o_text_translate_world(TOPLEVEL * w_current, int x1, int y1,
		       OBJECT * o_current)
{
  int screen_x, screen_y;
  int left, right, top, bottom;


  o_current->text->x = o_current->text->x + x1;
  o_current->text->y = o_current->text->y + y1;

  /* update screen coords */
  WORLDtoSCREEN(w_current, o_current->text->x, o_current->text->y,
		&screen_x, &screen_y);

  o_current->text->screen_x = screen_x;
  o_current->text->screen_y = screen_y;

  o_complex_world_translate(w_current, x1, y1, o_current->text->prim_objs);

  /* update bounding box */
  /* do it */
  get_text_bounds(w_current, o_current, &left, &top, &right, &bottom);

  o_current->left = left;
  o_current->top = top;
  o_current->right = right;
  o_current->bottom = bottom;
}


#line 1620 "../noweb/o_text_basic.nw"
OBJECT *o_text_copy(TOPLEVEL * w_current, OBJECT * list_tail,
		    OBJECT * o_current)
{
  OBJECT *new_obj;
  int color;

  if (o_current->saved_color == -1) {
    color = o_current->color;
  } else {
    color = o_current->saved_color;
  }

  new_obj = o_text_add(w_current, list_tail, OBJ_TEXT,
		       color,
		       o_current->text->x, o_current->text->y,
		       o_current->text->alignment,
		       o_current->text->angle,
		       o_current->text->string,
		       o_current->text->size,
		       o_current->visibility, o_current->show_name_value);

  return (new_obj);
}


#line 1655 "../noweb/o_text_basic.nw"
void o_text_freeallfonts(TOPLEVEL * w_current)
{
  int i;

  for (i = 0; i < NUM_CHARS; i++) {
    if (font_set[i].font_prim_objs != NULL) {
      s_delete_list_fromstart(w_current, font_set[i].font_prim_objs);
      font_set[i].font_prim_objs = NULL;
    }
  }

}


#line 1679 "../noweb/o_text_basic.nw"
void o_text_print_text_width(FILE * fp, char *output_string)
{
  int len, i, j;
  int starting_character = 0;
  int num_lines, line_number;
  int max_len = -1;
  char *single_line = NULL;
  char *max_length_line = NULL;
  int single_len;

  /* break up the string and find the longest string */
  num_lines = o_text_num_lines(output_string);
  single_line = u_basic_strdup(output_string);	/* larger than needed */
  len = strlen(output_string);
  for (line_number = 0; line_number < num_lines; line_number++) {
    j = 0;
    /* break up the string into lines */
    for (i = starting_character; i < len; i++) {
      if (output_string[i] != '\n' && output_string[i] != '\0') {
	single_line[j] = output_string[i];
      } else {
	starting_character = i + 1;
	break;
      }
      j++;
    }
    single_line[j] = '\0';

    single_len = strlen(single_line);
    if (single_len > max_len) {
      max_len = strlen(single_line);
      if (max_length_line)
	free(max_length_line);
      max_length_line = u_basic_strdup(single_line);
    }
  }


  fprintf(fp, "(");
  len = strlen(max_length_line);
  for (i = 0; i < len; i++) {
    if (max_length_line[i] == '(' || max_length_line[i] == ')'
	|| max_length_line[i] == '\\') {
      fprintf(fp, "\\");
    }

    fprintf(fp, "%c", max_length_line[i]);
  }

  /* convert width to mils */
  /* .95 is a fudge factor */
  fprintf(fp, ") stringwidth pop\n");

  if (single_line)
    free(single_line);
  if (max_length_line)
    free(max_length_line);
}


#line 1751 "../noweb/o_text_basic.nw"
void o_text_print_text_height(FILE * fp, int size)
{
  fprintf(fp, "%f\n", (float) size);
}


#line 1767 "../noweb/o_text_basic.nw"
void o_text_print_text_height_full(FILE * fp, char *string, int size)
{
  int num_lines = o_text_num_lines(string);
  fprintf(fp, "%f\n", (float) (size * num_lines +
			       size * LINE_SPACING * (num_lines - 1)));
}


#line 1785 "../noweb/o_text_basic.nw"
void o_text_print_text_string(FILE * fp, char *string)
{
  int len, i;

  if (!string) {
    return;
  }

  len = strlen(string);

  fprintf(fp, "(");
  for (i = 0; i < len; i++) {
    if (string[i] == '(' || string[i] == ')' || string[i] == '\\') {
      fprintf(fp, "\\");
    }

    fprintf(fp, "%c", string[i]);
  }

  fprintf(fp, ") show\n");
}


#line 1822 "../noweb/o_text_basic.nw"
void
o_text_print(TOPLEVEL * w_current, FILE * fp, OBJECT * o_current,
	     int origin_x, int origin_y)
{
  char *output_string = NULL;
  char *name = NULL;
  char *value = NULL;
  int sign = 1;
  int len;
  int i, j;
  int x, y;
  int num_lines, line_number, starting_character;
  int full_text_height;
  char *single_line = NULL;

  if (!o_current->text->string) {
    return;
  }

  if (w_current->print_color) {
    f_print_set_color(fp, o_current->color);
  }

  fprintf(fp, "/Helvetica findfont\n");
  fprintf(fp, "%f scalefont\n", (float) o_current->text->size * 1.4);
  fprintf(fp, "setfont\n");
  fprintf(fp, "\n");

  /* fprintf(fp, "newpath\n"); */

  if (o_attrib_get_name_value(o_current->text->string, &name, &value)) {
    switch (o_current->show_name_value) {
    case (SHOW_NAME_VALUE):
      output_string = g_strdup(o_current->text->string);
      break;

    case (SHOW_NAME):
      if (name[0] != '\0') {
	output_string = g_strdup(name);
      } else {
	fprintf(stderr,
		"Got an improper attribute: %s\n",
		o_current->text->string);
	output_string = g_strdup("invalid");
      }
      break;

    case (SHOW_VALUE):
      if (value[0] != '\0') {
	output_string = g_strdup(value);
      } else {
	/* you probably can remove this now... */
	/* since improper attributes will never get here */
	fprintf(stderr,
		"Got an improper attribute: %s\n",
		o_current->text->string);
	output_string = g_strdup("invalid");
      }
      break;
    }
  } else {
    output_string = g_strdup(o_current->text->string);
  }

  switch (o_current->text->angle) {
  case (0):
    sign = -1;
    break;
  case (90):
    sign = 1;
    break;
  case (180):
    sign = 1;
    break;
  case (270):
    sign = -1;
    break;
  }

  full_text_height = o_text_height(output_string, o_current->text->size);
  num_lines = o_text_num_lines(output_string);

  x = o_current->text->x;
  y = o_current->text->y;
  if (o_current->text->angle == 0 || o_current->text->angle == 180) {

    /* not quite sure about the second part of next statement. */
    /* full_text_height should take the spacing into account, but it doesn't */
    y = y - o_text_height("a", o_current->text->size) + full_text_height +
	(num_lines - 1) * o_current->text->size * LINE_SPACING;

    switch (o_current->text->alignment) {
    case (LOWER_LEFT):
      fprintf(fp, "%d mils %d mils moveto\n", x, y);
      /*                              x_offset = x; */
      /*                              y_offset = y; */
      break;

    case (MIDDLE_LEFT):
      fprintf(fp, "%d mils %d mils\n", x, y);
      o_text_print_text_height_full(fp, output_string,
				    o_current->text->size);
      fprintf(fp, ".714285 mul\n");	/* height adjustment factor */
      fprintf(fp, ".5 mul %d mul add moveto\n", sign);
      /*                              x_offset = x; */
      /*                              y_offset = y + sign*0.5*text_height; */
      break;

    case (UPPER_LEFT):
      fprintf(fp, "%d mils %d mils\n", x, y);
      o_text_print_text_height_full(fp, output_string,
				    o_current->text->size);
      fprintf(fp, ".714285 mul\n");	/* height adjustment factor */
      fprintf(fp, "%d mul add moveto\n", sign);
      /*                              x_offset = x; */
      /*                              y_offset = y + sign*text_height; */
      break;

    case (LOWER_MIDDLE):
      fprintf(fp, "%d mils ", x);

      o_text_print_text_width(fp, output_string);
      fprintf(fp, ".5 mul %d mul add\n", sign);
      fprintf(fp, "%d mils\n", y);
      fprintf(fp, "moveto\n");
      /*                              x_offset = x + sign*0.5*text_width; */
      /*                              y_offset = y; */
      break;

    case (MIDDLE_MIDDLE):
      fprintf(fp, "%d mils ", x);

      o_text_print_text_width(fp, output_string);
      fprintf(fp, ".5 mul %d mul add\n", sign);
      fprintf(fp, "%d mils\n", y);
      o_text_print_text_height_full(fp, output_string,
				    o_current->text->size);
      fprintf(fp, ".714285 mul\n");	/* height adjustment factor */
      fprintf(fp, ".5 mul %d mul add moveto\n", sign);
      /*                              x_offset = x + sign*0.5*text_width; */
      /*                              y_offset = y + sign*0.5*text_height; */
      break;

    case (UPPER_MIDDLE):
      fprintf(fp, "%d mils ", x);

      o_text_print_text_width(fp, output_string);
      fprintf(fp, ".5 mul %d mul add\n", sign);
      fprintf(fp, "%d mils\n", y);
      o_text_print_text_height_full(fp, output_string,
				    o_current->text->size);
      fprintf(fp, ".714285 mul\n");	/* height adjustment factor */
      fprintf(fp, "%d mul add moveto\n", sign);
      /*                              x_offset = x + sign*0.5*text_width; */
      /*                              y_offset = y + sign*text_height; */
      break;

    case (LOWER_RIGHT):
      fprintf(fp, "%d mils ", x);

      o_text_print_text_width(fp, output_string);
      fprintf(fp, "%d mul add\n", sign);
      fprintf(fp, "%d mils\n", y);
      fprintf(fp, "moveto\n");
      /*                              x_offset = x + sign*text_width; */
      /*                              y_offset = y; */
      break;

    case (MIDDLE_RIGHT):
      fprintf(fp, "%d mils ", x);

      o_text_print_text_width(fp, output_string);
      fprintf(fp, "%d mul add\n", sign);
      fprintf(fp, "%d mils\n", y);
      o_text_print_text_height_full(fp, output_string,
				    o_current->text->size);
      fprintf(fp, ".714285 mul\n");	/* height adjustment factor */
      fprintf(fp, ".5 mul %d mul add moveto\n", sign);
      /*                              x_offset = x + sign*text_width; */
      /*                              y_offset = y + sign*0.5*text_height; */
      break;

    case (UPPER_RIGHT):
      fprintf(fp, "%d mils ", x);

      o_text_print_text_width(fp, output_string);
      fprintf(fp, "%d mul add\n", sign);
      fprintf(fp, "%d mils\n", y);
      o_text_print_text_height_full(fp, output_string,
				    o_current->text->size);
      fprintf(fp, ".714285 mul\n");	/* height adjustment factor */
      fprintf(fp, "%d mul add moveto\n", sign);
      /*                              x_offset = x + sign*text_width; */
      /*                              y_offset = y + sign*text_height; */
      break;
    }

  } else if (o_current->text->angle == 90 || o_current->text->angle == 270) {

    /* not quite sure about the second part of next statement. */
    /* full_text_height should take the spacing into account, but it doesn't */
    x = x + sign * (o_text_height("a", o_current->text->size) -
		    full_text_height - (num_lines -
					1) * o_current->text->size *
		    LINE_SPACING);

    switch (o_current->text->alignment) {

    case (LOWER_LEFT):
      fprintf(fp, "%d mils %d mils moveto\n", x, y);
      /*                                x_offset = x; */
      /*                                y_offset = y; */
      break;

    case (MIDDLE_LEFT):
      fprintf(fp, "%d mils\n", x);

      o_text_print_text_height_full(fp, output_string,
				    o_current->text->size);
      fprintf(fp, ".714285 mul\n");	/* height adjustment factor */
      fprintf(fp, ".5 mul %d mul add\n", sign);
      fprintf(fp, "%d mils moveto\n", y);
      /*                                x_offset = x + sign*0.5*text_height; */
      /*                                y_offset = y; */
      break;

    case (UPPER_LEFT):
      fprintf(fp, "%d mils\n", x);

      o_text_print_text_height_full(fp, output_string,
				    o_current->text->size);
      fprintf(fp, ".714285 mul\n");	/* height adjustment factor */
      fprintf(fp, "%d mul add\n", sign);
      fprintf(fp, "%d mils moveto\n", y);
      /*                                x_offset = x + sign*text_height; */
      /*                                y_offset = y; */
      break;

    case (LOWER_MIDDLE):
      fprintf(fp, "%d mils\n", x);
      fprintf(fp, "%d mils\n", y);
      o_text_print_text_width(fp, output_string);
      fprintf(fp, "0.5 mul %d mul sub moveto\n", sign);
      /*                                x_offset = x; */
      /*                                y_offset = y - sign*0.5*text_width; */
      break;

    case (MIDDLE_MIDDLE):
      fprintf(fp, "%d mils\n", x);
      o_text_print_text_height_full(fp, output_string,
				    o_current->text->size);
      fprintf(fp, ".714285 mul\n");	/* height adjustment factor */
      fprintf(fp, "0.5 mul %d mul add\n", sign);
      fprintf(fp, "%d mils\n", y);
      o_text_print_text_width(fp, output_string);
      fprintf(fp, "0.5 mul %d mul sub moveto\n", sign);
      /*                                x_offset = x + sign*0.5*text_height; */
      /*                                y_offset = y - sign*0.5*text_width; */
      break;

    case (UPPER_MIDDLE):
      fprintf(fp, "%d mils\n", x);
      o_text_print_text_height_full(fp, output_string,
				    o_current->text->size);
      fprintf(fp, ".714285 mul\n");	/* height adjustment factor */
      fprintf(fp, "%d mul add\n", sign);
      fprintf(fp, "%d mils\n", y);
      o_text_print_text_width(fp, output_string);
      fprintf(fp, "0.5 mul %d mul sub moveto\n", sign);
      /*                                x_offset = x + sign*text_height; */
      /*                                y_offset = y - sign*0.5*text_width; */

      break;

    case (LOWER_RIGHT):
      fprintf(fp, "%d mils\n", x);
      fprintf(fp, "%d mils\n", y);
      o_text_print_text_width(fp, output_string);
      fprintf(fp, "%d mul sub moveto\n", sign);
      /*                                x_offset = x; */
      /*                                y_offset = y - sign*text_width; */
      break;

    case (MIDDLE_RIGHT):
      fprintf(fp, "%d mils\n", x);
      o_text_print_text_height_full(fp, output_string,
				    o_current->text->size);
      fprintf(fp, ".714285 mul\n");	/* height adjustment factor */
      fprintf(fp, "0.5 mul %d mul add\n", sign);
      fprintf(fp, "%d mils\n", y);
      o_text_print_text_width(fp, output_string);
      fprintf(fp, "%d mul sub moveto\n", sign);
      /*                                x_offset = x + sign*0.5*text_height; */
      /*                                y_offset = y - sign*text_width; */
      break;

    case (UPPER_RIGHT):
      fprintf(fp, "%d mils\n", x);
      o_text_print_text_height_full(fp, output_string,
				    o_current->text->size);
      fprintf(fp, ".714285 mul\n");	/* height adjustment factor */
      fprintf(fp, "%d mul add\n", sign);
      fprintf(fp, "%d mils\n", y);
      o_text_print_text_width(fp, output_string);
      fprintf(fp, "%d mul sub moveto\n", sign);
      /*                                x_offset = x + sign*text_height; */
      /*                                y_offset = y - sign*text_width; */
      break;
    }

  }

  /* correct 180 degree rotated text so it is in the right position */
  if (o_current->text->angle == 180) {
    o_text_print_text_width(fp, output_string);
    fprintf(fp, "-1.0 mul\n");	/* x distance back */
    o_text_print_text_height_full(fp, output_string,
				  o_current->text->size);
    fprintf(fp, ".714285 mul\n");	/* height adjustment factor */
    fprintf(fp, "-1.0 mul\n");	/* y distance down */
    fprintf(fp, "rmoveto\n");
  }

  starting_character = 0;
  single_line = u_basic_strdup(output_string);	/* larger than needed */
  len = strlen(output_string);
  for (line_number = 0; line_number < num_lines; line_number++) {

    j = 0;
    /* break up the string into lines */
    for (i = starting_character; i < len; i++) {
      if (output_string[i] != '\n' && output_string[i] != '\0') {
	single_line[j] = output_string[i];
      } else {
	starting_character = i + 1;
	break;
      }
      j++;
    }
    single_line[j] = '\0';

    /* Push the current position onto the stack */
    fprintf(fp, "currentpoint\n");

    if (o_current->text->angle == 90 || o_current->text->angle == 270) {
      fprintf(fp, "gsave\n%d rotate\n", o_current->text->angle);
    }

    /* output the single line */
    o_text_print_text_string(fp, single_line);

    if (o_current->text->angle == 90 || o_current->text->angle == 270) {
      fprintf(fp, "grestore\n");
    }

    /* Correct the position on the stack to go to the next line */
    if (line_number != num_lines - 1) {
      /* only do this if there are more text lines to output */

      switch (o_current->text->angle) {

      case 0:
	/* stack: x y */
	o_text_print_text_height(fp, o_current->text->size);
	fprintf(fp, "-1 mul %f mul add moveto\n", LINE_SPACING);
	/* stack: empty */

	/* x = line_start_x; */
	/* y = line_start_y - char_height * 2; */
	break;

      case 90:
	fprintf(fp, "exch\n");	/* stack: y x */
	o_text_print_text_height(fp, o_current->text->size);
	fprintf(fp, "%f mul add exch moveto\n", LINE_SPACING);
	/* stack: empty */
	/* x = line_start_x + char_height * 2; */
	/* y = line_start_y; */
	break;

      case 180:
	/* stack: x y */
	o_text_print_text_height(fp, o_current->text->size);
	fprintf(fp, "%f mul sub moveto\n", LINE_SPACING);
	/* stack: empty */

	/* x = line_start_x; */
	/* y = line_start_y + char_height * 2; */
	break;


      case 270:
	fprintf(fp, "exch\n");	/* stack: y x */
	o_text_print_text_height(fp, o_current->text->size);
	fprintf(fp, "%f mul sub exch moveto\n", LINE_SPACING);
	/* stack: empty */

	/* x = line_start_x - char_height * 2; */
	/* y = line_start_y; */
	break;

      default:
	fprintf(stderr, "o_text_create_string: Angle not supported\n");
	break;
      }

    } else {
      /* get rid of the last unused current point */
      fprintf(fp, "pop pop\n");

    }
  }

  if (output_string)
    free(output_string);
  if (name)
    free(name);
  if (value)
    free(value);
  if (single_line)
    free(single_line);
}


#line 2251 "../noweb/o_text_basic.nw"
/* takes world coords as the center point as well as a true angle */
void
o_text_rotate_lowlevel(TOPLEVEL * w_current, int world_centerx,
		       int world_centery, int angle, OBJECT * object)
{
  OBJECT *o_current = NULL;

  /* translate object to origin */
  /* o_text_translate_world(w_current, -world_centerx, -world_centery, object); */

  /* rotate_point_90(object->text->x, object->text->y, &newx, &newy); */

  /* o_text_translate_world(w_current, world_centerx, world_centery, object); */

  o_current = object;

  while (o_current != NULL) {
    switch (o_current->type) {
    case (OBJ_LINE):
      o_line_rotate_world(w_current, 0, 0, angle, o_current);
      break;
    }
    o_current = o_current->next;
  }
}


#line 2287 "../noweb/o_text_basic.nw"
void
o_text_rotate_world(TOPLEVEL * w_current, int world_centerx,
		    int world_centery, int angle, int angle_change,
		    OBJECT * object)
{
  int newx, newy;
  int origx, origy;
  int x, y;

  origx = object->text->x;
  origy = object->text->y;

  object->text->angle = (object->text->angle + angle_change) % 360;

#if DEBUG
  printf("rotating text angle: %d\n", angle);
  printf("rotating text angle_change: %d\n", angle_change);
  printf("final text angle: %d\n", object->text->angle);
#endif

  x = origx + (-world_centerx);
  y = origy + (-world_centery);

  rotate_point_90(x, y, angle_change, &newx, &newy);

  x = newx + (world_centerx);
  y = newy + (world_centery);

  o_text_translate_world(w_current, x - object->text->x,
			 y - object->text->y, object);

  o_text_recreate(w_current, object);
}


#line 2329 "../noweb/o_text_basic.nw"
void
o_text_rotate(TOPLEVEL * w_current, int centerx, int centery, int angle,
	      int angle_change, OBJECT * object)
{
  int newx, newy;
  int origx, origy;
  int world_centerx, world_centery;
  int x, y;

  SCREENtoWORLD(w_current, centerx, centery,
		&world_centerx, &world_centery);


  origx = object->text->x;
  origy = object->text->y;

  object->text->angle = angle;

  x = origx + (-world_centerx);
  y = origy + (-world_centery);

  rotate_point_90(x, y, angle_change, &newx, &newy);

  x = newx + (world_centerx);
  y = newy + (world_centery);

  o_text_translate_world(w_current, x - object->text->x,
			 y - object->text->y, object);

  o_text_recreate(w_current, object);
}



#line 44 "../noweb/o_text_basic.nw"
#if 0				/* code which is no longer needed, replaced by new functions below */
#line 2371 "../noweb/o_text_basic.nw"
void
o_text_mirror_old(TOPLEVEL * w_current, int centerx, int centery,
		  OBJECT * object)
{
  int newx = 0, newy = 0;
  int origx, origy;
  int world_centerx, world_centery;
  int x, y;
  char *output_string = NULL;
  char *name = NULL;
  char *value = NULL;
  int height_mod = 0;
  int sign = 1;

  SCREENtoWORLD(w_current, centerx, centery,
		&world_centerx, &world_centery);

  origx = object->text->x;
  origy = object->text->y;

  x = origx + (-world_centerx);
  y = origy + (-world_centery);

  if (o_attrib_get_name_value(object->text->string, &name, &value)) {
    switch (object->show_name_value) {
    case (SHOW_NAME_VALUE):
      output_string = g_strdup(object->text->string);
      break;

    case (SHOW_NAME):
      if (name[0] != '\0') {
	output_string = g_strdup(name);
      } else {
	/* you probably can remove this now... */
	/* since improper attributes will never get here */
	fprintf(stderr,
		"Got an improper attribute: %s\n", object->text->string);
	output_string = g_strdup("invalid");
      }
      break;

    case (SHOW_VALUE):
      if (value[0] != '\0') {
	output_string = g_strdup(value);
      } else {
	/* you probably can remove this now... */
	/* since improper attributes will never get here */
	fprintf(stderr,
		"Got an improper attribute: %s\n", object->text->string);
	output_string = g_strdup("invalid");
      }
      break;
    }
  } else {
    output_string = g_strdup(object->text->string);
  }

  switch (object->text->alignment) {
  case (LOWER_LEFT):
    if (object->text->angle == 0 || object->text->angle == 180) {
      sign = 1;
      height_mod = 0;
    } else if (object->text->angle == 90 || object->text->angle == 270) {
      height_mod = 0;
    }
    break;

  case (MIDDLE_LEFT):
    if (object->text->angle == 0 || object->text->angle == 180) {
      sign = 1;
      height_mod = 0;
    } else if (object->text->angle == 90 || object->text->angle == 270) {
      height_mod = o_text_height(object->text->string, object->text->size);
    }
    break;

  case (UPPER_LEFT):
    if (object->text->angle == 0 || object->text->angle == 180) {
      sign = 1;
      height_mod = 0;
    } else if (object->text->angle == 90 || object->text->angle == 270) {
      height_mod = 2 * o_text_height(object->text->string,
				     object->text->size);
    }
    break;

  case (LOWER_MIDDLE):
    if (object->text->angle == 0 || object->text->angle == 180) {
      sign = 0.5;
      height_mod = 0;
    } else if (object->text->angle == 90 || object->text->angle == 270) {
      sign = 1;
      height_mod = 0;
    }
    break;

  case (MIDDLE_MIDDLE):
    if (object->text->angle == 0 || object->text->angle == 180) {
      sign = 0.5;
      height_mod = 0;
    } else if (object->text->angle == 90 || object->text->angle == 270) {
      sign = 1;
      height_mod = o_text_height(object->text->string, object->text->size);
    }
    break;

  case (UPPER_MIDDLE):
    if (object->text->angle == 0 || object->text->angle == 180) {
      sign = 0.5;
      height_mod = 0;
    } else if (object->text->angle == 90 || object->text->angle == 270) {
      sign = 1;
      height_mod = 2 * o_text_height(object->text->string,
				     object->text->size);
    }
    break;

  case (LOWER_RIGHT):
    if (object->text->angle == 0 || object->text->angle == 180) {
      sign = -1;
      height_mod = 0;
    } else if (object->text->angle == 90 || object->text->angle == 270) {
      sign = 1;
      height_mod = 0;
    }
    break;

  case (MIDDLE_RIGHT):
    if (object->text->angle == 0 || object->text->angle == 180) {
      sign = -1;
      height_mod = 0;
    } else if (object->text->angle == 90 || object->text->angle == 270) {
      sign = 1;
      height_mod = o_text_height(object->text->string, object->text->size);
    }
    break;

  case (UPPER_RIGHT):
    if (object->text->angle == 0 || object->text->angle == 180) {
      sign = -1;
      height_mod = 0;
    } else if (object->text->angle == 90 || object->text->angle == 270) {
      sign = 1;
      height_mod = 2 * o_text_height(object->text->string,
				     object->text->size);
    }
    break;
  }

  switch (object->text->angle) {

  case (0):
    newx = -(x + sign * o_text_width(w_current,
				     output_string,
				     object->text->size / 2));
    break;

  case (90):
    newx = -(x - sign * o_text_height(object->text->string,
				      object->text->size) + height_mod);
    break;

  case (180):
    newx = -(x - sign * o_text_width(w_current,
				     output_string,
				     object->text->size / 2));
    break;

  case (270):
    newx = -(x + sign * o_text_height(object->text->string,
				      object->text->size) - height_mod);
    break;


  default:
    fprintf(stderr, "Invalid angle specified!\n");
    return;
    break;

  }

  newy = y;

  x = newx + (world_centerx);
  y = newy + (world_centery);

  /* don't know if this is needed? */
  o_text_translate_world(w_current, x - object->text->x,
			 y - object->text->y, object);

  o_text_recreate(w_current, object);
  if (output_string)
    free(output_string);
  if (name)
    free(name);
  if (value)
    free(value);
}


#line 2584 "../noweb/o_text_basic.nw"
void
o_text_mirror_world_old(TOPLEVEL * w_current, int world_centerx,
			int world_centery, OBJECT * object)
{
  int newx = 0, newy = 0;
  int origx, origy;
  int x, y;
  char *output_string = NULL;
  char *name = NULL;
  char *value = NULL;
  int sign = 1;
  int height_mod = 0;

  origx = object->text->x;
  origy = object->text->y;

  /* translate to origin */
  x = origx + (-world_centerx);
  y = origy + (-world_centery);


  if (o_attrib_get_name_value(object->text->string, &name, &value)) {
    switch (object->show_name_value) {
    case (SHOW_NAME_VALUE):
      output_string = g_strdup(object->text->string);
      break;

    case (SHOW_NAME):
      if (name[0] != '\0') {
	output_string = g_strdup(name);
      } else {
	/* you probably can remove this now... */
	/* since improper attributes will never get here */
	fprintf(stderr,
		"Got an improper attribute: %s\n", object->text->string);
	output_string = g_strdup("invalid");
      }
      break;

    case (SHOW_VALUE):
      if (value[0] != '\0') {
	output_string = g_strdup(value);
      } else {
	/* you probably can remove this now... */
	/* since improper attributes will never get here */
	fprintf(stderr,
		"Got an improper attribute: %s\n", object->text->string);
	output_string = g_strdup("invalid");
      }
      break;
    }
  } else {
    output_string = g_strdup(object->text->string);
  }

  switch (object->text->alignment) {
  case (LOWER_LEFT):
    if (object->text->angle == 0 || object->text->angle == 180) {
      sign = 1;
      height_mod = 0;
    } else if (object->text->angle == 90 || object->text->angle == 270) {
      height_mod = 0;
    }
    break;

  case (MIDDLE_LEFT):
    if (object->text->angle == 0 || object->text->angle == 180) {
      sign = 1;
      height_mod = 0;
    } else if (object->text->angle == 90 || object->text->angle == 270) {
      height_mod = o_text_height(object->text->string, object->text->size);
    }
    break;

  case (UPPER_LEFT):
    if (object->text->angle == 0 || object->text->angle == 180) {
      sign = 1;
      height_mod = 0;
    } else if (object->text->angle == 90 || object->text->angle == 270) {
      height_mod = 2 * o_text_height(object->text->string,
				     object->text->size);
    }
    break;

  case (LOWER_MIDDLE):
    if (object->text->angle == 0 || object->text->angle == 180) {
      sign = 0.5;
      height_mod = 0;
    } else if (object->text->angle == 90 || object->text->angle == 270) {
      sign = 1;
      height_mod = 0;
    }
    break;

  case (MIDDLE_MIDDLE):
    if (object->text->angle == 0 || object->text->angle == 180) {
      sign = 0.5;
      height_mod = 0;
    } else if (object->text->angle == 90 || object->text->angle == 270) {
      sign = 1;
      height_mod = o_text_height(object->text->string, object->text->size);
    }
    break;

  case (UPPER_MIDDLE):
    if (object->text->angle == 0 || object->text->angle == 180) {
      sign = 0.5;
      height_mod = 0;
    } else if (object->text->angle == 90 || object->text->angle == 270) {
      sign = 1;
      height_mod = 2 * o_text_height(object->text->string,
				     object->text->size);
    }
    break;

  case (LOWER_RIGHT):
    if (object->text->angle == 0 || object->text->angle == 180) {
      sign = -1;
      height_mod = 0;
    } else if (object->text->angle == 90 || object->text->angle == 270) {
      sign = 1;
      height_mod = 0;
    }
    break;

  case (MIDDLE_RIGHT):
    if (object->text->angle == 0 || object->text->angle == 180) {
      sign = -1;
      height_mod = 0;
    } else if (object->text->angle == 90 || object->text->angle == 270) {
      sign = 1;
      height_mod = o_text_height(object->text->string, object->text->size);
    }
    break;

  case (UPPER_RIGHT):
    if (object->text->angle == 0 || object->text->angle == 180) {
      sign = -1;
      height_mod = 0;
    } else if (object->text->angle == 90 || object->text->angle == 270) {
      sign = 1;
      height_mod = 2 * o_text_height(object->text->string,
				     object->text->size);
    }
    break;
  }

  switch (object->text->angle) {

  case (0):
    newx = -(x + sign * o_text_width(w_current,
				     output_string,
				     object->text->size / 2));

    break;

  case (90):
    newx = -(x - sign * o_text_height(object->text->string,
				      object->text->size) + height_mod);
    break;

  case (180):
    newx = -(x - sign * o_text_width(w_current,
				     output_string,
				     object->text->size / 2));
    break;

  case (270):
    newx = -(x + sign * o_text_height(object->text->string,
				      object->text->size) + height_mod);
    break;

  default:
    fprintf(stderr, "Invalid angle specified!\n");
    return;
    break;

  }

  newy = y;

  x = newx + (world_centerx);
  y = newy + (world_centery);

  object->text->x = x;
  object->text->y = y;

  /* don't know if this is needed ? */
  /* o_text_translate_world(w_current, x-object->text->x, y-object->text->y, object); */
  o_text_recreate(w_current, object);
  if (output_string)
    free(output_string);
  if (name)
    free(name);
  if (value)
    free(value);
}


#line 47 "../noweb/o_text_basic.nw"
#endif

#if 0				/* interesting, but currently unused code */
#line 2794 "../noweb/o_text_basic.nw"
void
o_text_return_center(TOPLEVEL * w_current, OBJECT * o_current,
		     int *centerx, int *centery)
{
  int text_height;
  int text_width;

  text_height = o_text_height(o_current->text->string,
			      o_current->text->size);

  /* this will NOT NOT NOT work with attributes */
  text_width = o_text_width(w_current, o_current->text->string,
			    o_current->text->size / 2);

  switch (o_current->text->angle) {
  case (0):
    *centerx = o_current->text->x + text_width / 2;
    *centery = o_current->text->y + text_height / 2;
    break;

  case (90):
    *centerx = o_current->text->x - text_height / 2;
    *centery = o_current->text->y + text_width / 2;
    break;

  case (180):
    *centerx = o_current->text->x - text_width / 2;
    *centery = o_current->text->y - text_height / 2;
    break;

  case (270):
    *centerx = o_current->text->x + text_height / 2;
    *centery = o_current->text->y - text_width / 2;
    break;
  }
}


#line 2841 "../noweb/o_text_basic.nw"
/* the complex here is the complex of a complex object */
o_text_change_angle(TOPLEVEL * w_current, OBJECT * prim_objs,
		    int new_angle)
{
  OBJECT *o_current;
  int centerx, centery;

  o_current = prim_objs;

  while (o_current != NULL) {
    if (o_current->type == OBJ_TEXT) {
      o_current->text->angle = new_angle;

      /* change world to non */
      o_text_return_center(w_current, o_current, &centerx, &centery);

      o_text_translate_world(w_current, -centerx, -centery, o_current);

      o_text_mirror_world(w_current, 0, 0, o_current);
      /* 
         o_text_rotate_world(w_current, 0, 0, 
         new_angle, 180, o_current);
       */

      /*                        o_text_rotate_world(w_current, 0, 0, new_angle, 
         180, o_current) */

      o_text_translate_world(w_current, centerx, centery, o_current);

      /*                        o_text_rotate_lowlevel(w_current, 
         0, 0, new_angle, 180, o_current->text->prim_objs); */

#if 0
      w_current->override_color = w_current->background_color;
      o_text_draw(w_current, o_current);
      w_current->override_color = -1;
#endif

      o_text_recreate(w_current, o_current);
    }
    o_current = o_current->next;
  }
}


#line 52 "../noweb/o_text_basic.nw"
#endif

#line 2898 "../noweb/o_text_basic.nw"
void
o_text_mirror_world(TOPLEVEL * w_current, int world_centerx,
		    int world_centery, OBJECT * object)
{
  int origx, origy;
  int x, y;

  origx = object->text->x;
  origy = object->text->y;

  x = origx + (-world_centerx);
  y = origy + (-world_centery);

  if ((object->text->angle % 180) == 0) {
    switch (object->text->alignment) {
    case (LOWER_LEFT):
      object->text->alignment = LOWER_RIGHT;
      break;

    case (MIDDLE_LEFT):
      object->text->alignment = MIDDLE_RIGHT;
      break;

    case (UPPER_LEFT):
      object->text->alignment = UPPER_RIGHT;
      break;

    case (LOWER_RIGHT):
      object->text->alignment = LOWER_LEFT;
      break;

    case (MIDDLE_RIGHT):
      object->text->alignment = MIDDLE_LEFT;
      break;

    case (UPPER_RIGHT):
      object->text->alignment = UPPER_LEFT;
      break;

    default:
      break;
    }
  } else {
    switch (object->text->alignment) {
    case (LOWER_LEFT):
      object->text->alignment = UPPER_LEFT;
      break;

    case (UPPER_LEFT):
      object->text->alignment = LOWER_LEFT;
      break;

    case (LOWER_RIGHT):
      object->text->alignment = UPPER_RIGHT;
      break;

    case (UPPER_RIGHT):
      object->text->alignment = LOWER_RIGHT;
      break;

    case (LOWER_MIDDLE):
      object->text->alignment = UPPER_MIDDLE;
      break;

    case (UPPER_MIDDLE):
      object->text->alignment = LOWER_MIDDLE;
      break;

    default:
      break;
    }
  }

  object->text->x = -x + (world_centerx);
  object->text->y = y + (world_centery);

  o_text_recreate(w_current, object);
}


#line 2987 "../noweb/o_text_basic.nw"
void
o_text_mirror(TOPLEVEL * w_current, int centerx, int centery,
	      OBJECT * object)
{
  int world_centerx, world_centery;

  SCREENtoWORLD(w_current, centerx, centery,
		&world_centerx, &world_centery);

  o_text_mirror_world(w_current, world_centerx, world_centery, object);
}
