// SEQBUILD.CPP

// Copyright (C) 1998 Tommi Hassinen, Geoff Hutchison.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "libghemicalconfig2.h"
#include "seqbuild.h"

#include "v3d.h"

#include <string.h>

#include <iomanip>
#include <iostream>
#include <sstream>
using namespace std;

/*################################################################################################*/

sb_chain_descriptor::sb_chain_descriptor(bool m1l)
{
	mode_1_letter = m1l;
	
	seq1 = NULL;
	seq3 = NULL;
	
	curr_res = NOT_DEFINED;
	c_crd_BGN = c_crd_END = NOT_DEFINED;
	c_tor_BGN = c_tor_END = NOT_DEFINED;
	
	if (mode_1_letter) seq1 = new vector<char>;
	else seq3 = new vector<char *>;
}

sb_chain_descriptor::~sb_chain_descriptor(void)
{
	if (seq1 != NULL)
	{
		delete seq1;
		seq1 = NULL;
	}
	
	if (seq3 != NULL)
	{
		for (int n1 = 0;n1 < (int) seq3->size();n1++)
		{
			char * tmps = seq3->operator[](n1);
			delete[] tmps;
		}
		
		delete seq3;
		seq3 = NULL;
	}
}

int sb_chain_descriptor::AddRes1(char r1l)
{
	seq1->push_back(r1l);
	return (int) seq1->size();
}

int sb_chain_descriptor::AddRes3(const char * r3l)
{
	if (strlen(r3l) != 3)
	{
		cout << "callEXIT : sb_chain_descriptor::AddRes3() ; bad input" << endl;
		exit(EXIT_FAILURE);
	}
	
	char * tmps = new char[4];
	strcpy(tmps, r3l);
	
	seq3->push_back(tmps);
	return (int) seq3->size();
}

// add an XYZ constraint for an atom in a residue which was last added.
// USE THIS WITH GREAT CARE! ; this has been added for a special purpose.
// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
// the constraints are forced to be added in sequence order...

void sb_chain_descriptor::AddCRD(int atm_id, float x, float y, float z)
{
	int lastres = ((int) (mode_1_letter ? seq1->size() : seq3->size())) - 1;
	
	sb_constraint_crd newcrd;
	newcrd.pos = lastres;
	
	newcrd.atm_id = atm_id;
	newcrd.crdX = x; newcrd.crdY = y; newcrd.crdZ = z;
	
	c_crd_v.push_back(newcrd);
}

// add a torsional constraint for a residue which was last added.
// USE THIS WITH GREAT CARE! ; this has been added for a special purpose.
// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
// the constraints are forced to be added in sequence order...

void sb_chain_descriptor::AddTOR(int tor_ind, float tor_val)
{
	int lastres = ((int) (mode_1_letter ? seq1->size() : seq3->size())) - 1;
	
	sb_constraint_tor newtor;
	newtor.pos = lastres;
	
	newtor.tor_ind = tor_ind;
	newtor.tor_val = tor_val;
	
	c_tor_v.push_back(newtor);
}

/*################################################################################################*/

sequencebuilder::sequencebuilder(chn_info::chn_type ct, char * filename)
{
	type = ct;
	
	ifstream file;
	file.unsetf(ios::dec | ios::oct | ios::hex);
	
	model::OpenLibDataFile(file, false, filename);
	
	// read information about the main chain...
	
	while (file.peek() != 'M') file.getline(buffer, sizeof(buffer));
	file.getline(buffer, sizeof(buffer));
	
	while (file.peek() != 'E')
	{
		if (file.peek() == 'T')
		{
			sb_data_td newtd; file >> newtd;
			file.getline(buffer, sizeof(buffer));
			td_mc_store.push_back(newtd);
			continue;
		}
		
		if (file.peek() == 'A')
		{
			sb_data_atm newatm; file >> newatm;
			while (file.peek() != '(') file.get();
			newatm.tr = new typerule(& file, & cout);
			file.getline(buffer, sizeof(buffer));
			main_vector.push_back(newatm);
			continue;
		}
		
		file.getline(buffer, sizeof(buffer));
	}
	
	while (file.peek() != 'C') file.getline(buffer, sizeof(buffer));
	file.getline(buffer, sizeof(buffer));
	
	while (file.peek() != 'E')
	{
		if (file.peek() == 'T')
		{
			sb_data_td newtd; file >> newtd;
			file.getline(buffer, sizeof(buffer));
			td_mc_store.push_back(newtd);
			continue;
		}
		
		if (file.peek() == 'A')
		{
			sb_data_atm newatm; file >> newatm;
			while (file.peek() != '(') file.get();
			newatm.tr = new typerule(& file, & cout);
			file.getline(buffer, sizeof(buffer));
			conn_vector.push_back(newatm);
			continue;
		}
		
		file.getline(buffer, sizeof(buffer));
	}
	
	// read descriptions about the chain initiation...
	
	while (file.peek() != 'H') file.getline(buffer, sizeof(buffer));
	file.getline(buffer, sizeof(buffer));
	
	while (file.peek() != 'E')
	{
		typerule newrule = typerule(& file, & cout);
		file.getline(buffer, sizeof(buffer));
		head_vector.push_back(newrule);
	}
	
	// read descriptions about the chain termination...
	
	while (file.peek() != 'T') file.getline(buffer, sizeof(buffer));
	file.getline(buffer, sizeof(buffer));
	
	while (file.peek() != 'E')
	{
		typerule newrule = typerule(& file, & cout);
		file.getline(buffer, sizeof(buffer));
		tail_vector.push_back(newrule);
	}
	
	// read the default modifications for residues...
	
	while (file.peek() != 'B') file.getline(buffer, sizeof(buffer));	// default...
	file.getline(buffer, sizeof(buffer)); mod[0] = new sb_data_res; mod[0]->ReadModification(file);
	if (mod[0]->td_vector.size() > 0)
	{
		cout << "callEXIT : BODY_MOD should not have any TORDEF lines!" << endl;
		exit(EXIT_FAILURE);
	}
	
	while (file.peek() != 'H') file.getline(buffer, sizeof(buffer));	// head-mods
	file.getline(buffer, sizeof(buffer)); mod[1] = new sb_data_res; mod[1]->ReadModification(file);
	if (mod[1]->td_vector.size() > 0)
	{
		cout << "callEXIT : HEAD_MOD should not have any TORDEF lines!" << endl;
		exit(EXIT_FAILURE);
	}
	
	while (file.peek() != 'T') file.getline(buffer, sizeof(buffer));	// tail-mods
	file.getline(buffer, sizeof(buffer)); mod[2] = new sb_data_res; mod[2]->ReadModification(file);
	if (mod[2]->td_vector.size() > 0)
	{
		cout << "callEXIT : TAIL_MOD should not have any TORDEF lines!" << endl;
		exit(EXIT_FAILURE);
	}
	
	// read descriptions for the residues...
	
	while (file.peek() != 'E')
	{
		if (file.peek() == 'R')
		{
			sb_data_res newres; file >> newres;
			resi_vector.push_back(newres);
		}
		else file.getline(buffer, sizeof(buffer));
	}
	
	// ready!!!
	
	file.close();
}

sequencebuilder::~sequencebuilder(void)
{
	delete mod[0];
	delete mod[1];
	delete mod[2];
}

// here we use only the first coordinate set, no matter how many there are...

void sequencebuilder::Build(model * mdl, sb_chain_descriptor * chde, bool do_centering)
{
	bool bad_chde = false;
	if (chde->seq1 == NULL && chde->seq3 == NULL) bad_chde = true;
	if (chde->mode_1_letter)
	{
		if (chde->seq1 == NULL) bad_chde = true;
	}
	else
	{
		if (chde->seq3 == NULL) bad_chde = true;
	}
	
	if (bad_chde)
	{
		cout << "callEXIT : sequencebuilder::Build() ; bad input" << endl;
		exit(EXIT_FAILURE);
	}
	
	const int num_res = (chde->mode_1_letter ? chde->seq1->size() : chde->seq3->size());
	const int num_cs = mdl->GetCRDSetCount();
	
	fGL b1crd[3] = { 1.0, 1.0, 0.0 };		// these should be const...
	fGL b2crd[3] = { 1.0, 0.0, 0.0 };
	fGL b3crd[3] = { 0.0, 0.0, 0.0 };
	
	atom base[3] =
	{
		atom(element(), b1crd, num_cs),		// these should be const...
		atom(element(), b2crd, num_cs),
		atom(element(), b3crd, num_cs)
	};
	
	id_vector.resize(0);
	atom_vector.resize(0);
	allatm_vector.resize(0);
	
	id_vector.push_back(main_vector[0].prev[0]); atom_vector.push_back(& base[0]);
	id_vector.push_back(main_vector[0].prev[1]); atom_vector.push_back(& base[1]);
	id_vector.push_back(main_vector[0].prev[2]); atom_vector.push_back(& base[2]);
	
	for (i32u n1 = 0;n1 < num_res;n1++)
	{
		bool first_res = (n1 == 0);
		bool last_res = (n1 == (num_res - 1));
		
		chde->curr_res = n1;
		
		chde->c_crd_BGN = chde->c_crd_END = NOT_DEFINED;
		for (i32u n2 = 0;n2 < chde->c_crd_v.size();n2++)
		{
			if (chde->c_crd_BGN < 0 && chde->c_crd_v[n2].pos == n1) chde->c_crd_BGN = n2;
			if (chde->c_crd_BGN >= 0 && chde->c_crd_END < 0 && chde->c_crd_v[n2].pos > n1) chde->c_crd_END = n2;
		}	if (chde->c_crd_BGN >= 0 && chde->c_crd_END < 0) chde->c_crd_END = chde->c_crd_v.size();
		
		chde->c_tor_BGN = chde->c_tor_END = NOT_DEFINED;
		for (i32u n2 = 0;n2 < chde->c_tor_v.size();n2++)
		{
			if (chde->c_tor_BGN < 0 && chde->c_tor_v[n2].pos == n1) chde->c_tor_BGN = n2;
			if (chde->c_tor_BGN >= 0 && chde->c_tor_END < 0 && chde->c_tor_v[n2].pos > n1) chde->c_tor_END = n2;
		}	if (chde->c_tor_BGN >= 0 && chde->c_tor_END < 0) chde->c_tor_END = chde->c_tor_v.size();
		
		td_v.resize(0);
		for (i32u n2 = 0;n2 < td_mc_store.size();n2++)
		{
			td_v.push_back(td_mc_store[n2]);
		}
		
		// build main chain.
		
		for (i32u n2 = 0;n2 < main_vector.size();n2++)
		{
			fGL crd[3];
			Convert(chde, & main_vector[n2], crd);
			
			id_vector.push_back(main_vector[n2].id);
			atom newatom(main_vector[n2].el, crd, num_cs);
			
			newatom.formal_charge = main_vector[n2].f_chrg;
			
			mdl->AddAtom(newatom);
			atom_vector.push_back(& mdl->GetLastAtom());
			allatm_vector.push_back(& mdl->GetLastAtom());
			
			if (!first_res || n2 != 0)
			{
				i32s p_ind = 0;
				while (id_vector[p_ind] != main_vector[n2].prev[0]) p_ind++;
				atom * prev = atom_vector[p_ind];
				
				bond newbond(prev, & mdl->GetLastAtom(), main_vector[n2].bt);
				mdl->AddBond(newbond);
			}
		}
		
		// build connection (if needed).
		
		if (!last_res)
		{
			for (i32u n2 = 0;n2 < conn_vector.size();n2++)
			{
				fGL crd[3];
				Convert(chde, & conn_vector[n2], crd);
				
				id_vector.push_back(conn_vector[n2].id);
				atom newatom(conn_vector[n2].el, crd, num_cs);
				
				newatom.formal_charge = conn_vector[n2].f_chrg;
				
				mdl->AddAtom(newatom);
				atom_vector.push_back(& mdl->GetLastAtom());
				allatm_vector.push_back(& mdl->GetLastAtom());
				
				i32s p_ind = 0;
				while (id_vector[p_ind] != conn_vector[n2].prev[0]) p_ind++;
				atom * prev = atom_vector[p_ind];
				
				bond newbond(prev, & mdl->GetLastAtom(), conn_vector[n2].bt);
				mdl->AddBond(newbond);
			}
		}
		
		if (first_res)		// head-mods
		{
			BuildResidue(chde, mdl, mod[1]);
			
			// a hardcoded fix implemented:
			// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^
			// amino : need to set amino-terminal N formal_charge.
			
			if (type == chn_info::amino_acid)
			{
				i32u index = 0;
				while (index < id_vector.size())
				{
					if (id_vector[index] == 0x0000) break;
					else index++;
				}
				
				if (index >= id_vector.size())
				{
					cout << "callEXIT : ERROR : amino-terminal N missing?" << endl;
					exit(EXIT_FAILURE);
				}
				
				atom_vector[index]->formal_charge = +1;
			}
		}
		else if (last_res)	// tail-mods
		{
			BuildResidue(chde, mdl, mod[2]);
		}
		else			// default...
		{
			BuildResidue(chde, mdl, mod[0]);
		}
		
		i32u res = 0;
		while (true)
		{
			if (chde->mode_1_letter)
			{
				if (resi_vector[res].symbol1 == chde->seq1->operator[](n1)) break;
			}
			else
			{
				const char * seq3str = chde->seq3->operator[](n1);
				if (!strcmp(resi_vector[res].symbol3, seq3str)) break;
			}
			
			if (res != resi_vector.size()) res++;
			else break;
		}
		
		if (res != resi_vector.size())
		{
			for (i32u n2 = 0;n2 < resi_vector[res].td_vector.size();n2++)
			{
				td_v.push_back(resi_vector[res].td_vector[n2]);
			}
			
			BuildResidue(chde, mdl, & resi_vector[res]);
		}
		else
		{
			ostringstream str;
			str << "unknown residue ";
			
			if (chde->mode_1_letter)
			{
				str << chde->seq1->operator[](n1);
			}
			else
			{
				str << chde->seq3->operator[](n1);
			}
			
			str << endl << ends;
			if (mdl->verbosity >= 2) mdl->PrintToLog(str.str().c_str());
		}
		
		vector<i32s> tmpv1; atmr_vector tmpv2;
		for (i32u n2 = 0;n2 < id_vector.size();n2++)
		{
			if (id_vector[n2] & 0xFF00) continue;
			
			tmpv1.push_back(id_vector[n2]);
			tmpv2.push_back(atom_vector[n2]);
		}
		
		id_vector.resize(0);
		atom_vector.resize(0);
		
		for (i32u n2 = 0;n2 < tmpv1.size();n2++)
		{
			id_vector.push_back(tmpv1[n2] + 0xFF00);
			atom_vector.push_back(tmpv2[n2]);
		}
	}
	
	id_vector.resize(0);
	atom_vector.resize(0);
	
	// add hydrogens for a relevant range of atoms...
	
	for (i32u n1 = 0;n1 < allatm_vector.size();n1++)
	{
		mdl->AddHydrogens(allatm_vector[n1]);
	}
	
	allatm_vector.resize(0);
	
	// do centering (if asked). can't affect current_eng object
	// since above many atoms/bonds added -> current_eng deleted!
	
	if (do_centering) mdl->CenterCRDSet(0, true);
}

// here we will identify the sequences (if there is any), write down the sequences
// into model::ref_civ and update the /mol/chn/res-numbering...

// only model::UpdateChains() should call this???
// only model::UpdateChains() should call this???
// only model::UpdateChains() should call this???

void sequencebuilder::Identify(model * mdl)
{
	if (!mdl->IsGroupsClean())
	{
		cout << "callEXIT : is_groups_clean was false at sequencebuilder::Identify()." << endl;
		exit(EXIT_FAILURE);	//mdl->UpdateGroups();
	}
	
	if (mdl->ref_civ == NULL)
	{
		cout << "callEXIT : mdl->ref_civ == NULL at sequencebuilder::Identify()." << endl;
		exit(EXIT_FAILURE);
	}
	
	// here we will find all possible chains and identify them.
	
	for (i32s n1 = 0;n1 < mdl->nmol;n1++)
	{
		iter_al range[2];
		mdl->GetRange(0, n1, range);
		
		vector<atom *> head_atoms;
		vector<atom *> tail_atoms;
		
		for (iter_al it1 = range[0];it1 != range[1];it1++)
		{
			i32s tmp1 = (* it1).el.GetAtomicNumber();
			
			if (tmp1 == main_vector.front().el.GetAtomicNumber())		// look for heads...
			{
				for (i32u n2 = 0;n2 < head_vector.size();n2++)
				{
					bool flag = head_vector[n2].Check(mdl, & (* it1), 0);
					if (!flag) continue; head_atoms.push_back(& (* it1)); break;
				}
			}
			
			if (tmp1 == main_vector.back().el.GetAtomicNumber())		// look for tails...
			{
				for (i32u n2 = 0;n2 < tail_vector.size();n2++)
				{
					bool flag = tail_vector[n2].Check(mdl, & (* it1), 0);
					if (!flag) continue; tail_atoms.push_back(& (* it1)); break;
				}
			}
		}
		
		// now we have all possible head/tail atoms. next we have to check
		// all possible paths between them to find all possible main chains...
		
		path_vector.resize(0);
		for (i32u n2 = 0;n2 < head_atoms.size();n2++)
		{
			for (i32u n3 = 0;n3 < tail_atoms.size();n3++)
			{
				FindPath(mdl, head_atoms[n2], tail_atoms[n3]);
			}
		}
		
		if (path_vector.size())
		{
			ostringstream str;
			str << path_vector.size() << " chains:" << endl << ends;
			if (mdl->verbosity >= 3) mdl->PrintToLog(str.str().c_str());
		}
		
		for (i32s n2 = 0;n2 < (i32s) path_vector.size();n2++)
		{
			for (i32s n3 = 0;n3 < ((i32s) path_vector[n2].size()) - 1;n3++)		// tag the main-chain bonds...
			{
				iter_cl it1 = path_vector[n2][n3]->GetConnRecsBegin();
				while ((* it1).atmr != path_vector[n2][n3 + 1]) it1++;
				(* it1).bndr->flags[0] = true;
			}
			
			vector<char> tmpseq1;
			vector<char *> tmpseq3;
			
			i32u acount = 0;	// atom counter
			i32u rcount = 0;	// residue counter
			
			while (acount < path_vector[n2].size())
			{
				// rsize is the residue size (how many atoms it will chop away from the
				// path_vector). we use it to determine whether we have reached the last
				// residue, and to increment acount.
				
				int head_el = NOT_DEFINED;
				int tail_el = NOT_DEFINED;
				
				bool is_first = !rcount;
				if (is_first && type == chn_info::amino_acid)
				{
					// an extra test for an ACE-like block group.
					// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
					// -NH2 or -NH3 group -> -NH-X- group
					
					atom * refA = path_vector[n2][acount + 0];
					atom * refB = path_vector[n2][acount + 1];
					atom * refX = NULL;
					
					for (iter_cl itX = refA->GetConnRecsBegin();itX != refA->GetConnRecsEnd();itX++)
					{
						if ((* itX).atmr == refB) continue;
						if ((* itX).atmr->el.GetAtomicNumber() <= 1) continue;
						
						refX = (* itX).atmr;
					}
					
					if (refX != NULL)
					{
						cout << "found an ACE-like-block!" << endl;
						//exit(EXIT_FAILURE);
						
						head_el = refX->el.GetAtomicNumber();
						is_first = false;
					}
				}
				
				bool is_last = (acount + main_vector.size()) >= path_vector[n2].size();
				if (is_last && type == chn_info::amino_acid)
				{
					// an extra test for an NME-like block group.
					// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
					// O=C-O group -> O=C-X group (where X != O)
					
					atom * refA = path_vector[n2][acount + 2];
					atom * refB = path_vector[n2][acount + 1];
					atom * refX = NULL;
					
					for (iter_cl itX = refA->GetConnRecsBegin();itX != refA->GetConnRecsEnd();itX++)
					{
						if ((* itX).atmr == refB) continue;
						if ((* itX).bndr->bt.GetValue() != 1) continue;
						if ((* itX).atmr->el.GetAtomicNumber() == 8) continue;
						
						refX = (* itX).atmr;
					}
					
					if (refX != NULL)
					{
						cout << "found an NME-like-block!" << endl;
						//exit(EXIT_FAILURE);
						
						tail_el = refX->el.GetAtomicNumber();
						is_last = false;
					}
				}
				
				i32u rsize = main_vector.size();
				if (!is_last) rsize += conn_vector.size();
				
				// now we start identifying the residues.
				
				// we make a simple template for each of them, compare those to
				// the residue descriptions we have, and pick the closest match.
				// there may not be any missing atoms, and the closest match is
				// the one with largest number of correct atoms identified.
				
				i32s tmp1[2] = { NOT_DEFINED, NOT_DEFINED };
				for (i32s n4 = 0;n4 < (i32s) resi_vector.size();n4++)
				{
					vector<sb_tdata> tdata;
					BuildTemplate(tdata, n4, is_first, is_last);
					
					// do the template mods if blockgroups!!!
					// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
					
					if (head_el != NOT_DEFINED)
					{
						if (is_first)
						{
							cout << "callEXIT : bad head_el" << endl;
							exit(EXIT_FAILURE);
						}
						
						// ACE needs no modifications...
					}
					
					if (tail_el != NOT_DEFINED)
					{
						if (is_last)
						{
							cout << "callEXIT : bad tail_el" << endl;
							exit(EXIT_FAILURE);
						}
						
						for (i32s n5 = 0;n5 < (i32s) tdata.size();n5++)
						{
							if (tdata[n5].id[0] == 0x11)
							{
								tdata[n5].el = element(tail_el);
							}
						}
					}
					
					// set the main chain stuff...
					
					for (i32s n5 = 0;n5 < (i32s) main_vector.size();n5++)
					{
						tdata[n5].ref = path_vector[n2][acount + n5];
					}
					
					// set the connection stuff...
					
					if (!is_last)
					{
						const int mvsz = main_vector.size();
						for (i32s n5 = 0;n5 < (i32s) conn_vector.size();n5++)
						{
							tdata[mvsz + n5].ref = path_vector[n2][acount + mvsz + n5];
						}
					}
					
					bool result = CheckTemplate(tdata, 0);
///////////////////////////////////////////////////////////////////////////////////////////
//cout << "checktemplate " << n4 << " " << is_first << " " << is_last << " ; ";
//cout << "result = " << result << endl; int zzz; cin >> zzz;
///////////////////////////////////////////////////////////////////////////////////////////
					if (result && (tmp1[0] == NOT_DEFINED || ((i32s) tdata.size()) > tmp1[1]))
					{
						tmp1[0] = n4;
						tmp1[1] = (i32s) tdata.size();
					}
				}
				
				if (tmp1[0] == NOT_DEFINED)
				{
					ostringstream str;
					str << "WARNING : residue " << rcount << " was of unknown type!!!" << endl << ends;
					if (mdl->verbosity >= 2) mdl->PrintToLog(str.str().c_str());
					
					tmpseq1.push_back('?');		// handle the sequence...
					
					char * tmps = new char[4];
					strcpy(tmps, "???");
					
					tmpseq3.push_back(tmps);	// handle the sequence...
					
					for (i32u n3 = 0;n3 < main_vector.size();n3++)		// handle the numbers...
					{
						path_vector[n2][acount + n3]->builder_res_id = NOT_DEFINED;	// WHAT HERE???
						
						path_vector[n2][acount + n3]->id[1] = mdl->ref_civ->size();
						path_vector[n2][acount + n3]->id[2] = rcount;
					}
				}
				else
				{
					tmpseq1.push_back(resi_vector[tmp1[0]].symbol1);
					
					char * tmps = new char[4];
					strcpy(tmps, resi_vector[tmp1[0]].symbol3);
					
					tmpseq3.push_back(tmps);
					
					// rebuild the best matching template and use that to
					// set up the atom ID numbers...
					
					vector<sb_tdata> tdata;
					BuildTemplate(tdata, tmp1[0], is_first, is_last);
					
					for (i32s n4 = 0;n4 < (i32s) main_vector.size();n4++)
					{
						tdata[n4].ref = path_vector[n2][acount + n4];
					}
					
					if (!is_last)
					{
						const int mvsz = main_vector.size();
						for (i32s n5 = 0;n5 < (i32s) conn_vector.size();n5++)
						{
							tdata[mvsz + n5].ref = path_vector[n2][acount + mvsz + n5];
						}
					}
					
					CheckTemplate(tdata, 0);
					for (i32s n4 = 0;n4 < (i32s) tdata.size();n4++)
					{
						tdata[n4].ref->builder_res_id = (resi_vector[tmp1[0]].id << 8) + tdata[n4].id[0];
						
						tdata[n4].ref->id[1] = mdl->ref_civ->size();
						tdata[n4].ref->id[2] = rcount;
					}
					
/////////////////////////////////////////////////////////////////////////// exceptions
/////////////////////////////////////////////////////////////////////////// exceptions

// a hardcoded fix implemented:
// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^
// amino : fix the unambiguous LEU residues.

if (type == chn_info::amino_acid && resi_vector[tmp1[0]].symbol1 == 'L')
{
	atom * ref20 = NULL;
	atom * ref21 = NULL;
	atom * ref22 = NULL;
	atom * ref23 = NULL;
	
	for (i32s n4 = 0;n4 < (i32s) tdata.size();n4++)
	{
		if ((tdata[n4].ref->builder_res_id & 0xFF) == 0x20) ref20 = tdata[n4].ref;
		if ((tdata[n4].ref->builder_res_id & 0xFF) == 0x21) ref21 = tdata[n4].ref;
		if ((tdata[n4].ref->builder_res_id & 0xFF) == 0x22) ref22 = tdata[n4].ref;
		if ((tdata[n4].ref->builder_res_id & 0xFF) == 0x23) ref23 = tdata[n4].ref;
	}
	
	if (!ref20) { cout << "callEXIT : LEU-fix : ref20 not found!" << endl; exit(EXIT_FAILURE); }
	if (!ref21) { cout << "callEXIT : LEU-fix : ref21 not found!" << endl; exit(EXIT_FAILURE); }
	if (!ref22) { cout << "callEXIT : LEU-fix : ref22 not found!" << endl; exit(EXIT_FAILURE); }
	if (!ref23) { cout << "callEXIT : LEU-fix : ref23 not found!" << endl; exit(EXIT_FAILURE); }
	
	v3d<fGL> v1 = v3d<fGL>(ref21->GetCRD(0), ref20->GetCRD(0));
	v3d<fGL> v2 = v3d<fGL>(ref21->GetCRD(0), ref22->GetCRD(0));
	v3d<fGL> v3 = v3d<fGL>(ref22->GetCRD(0), ref23->GetCRD(0));
	fGL tor = v1.tor(v2, v3);
	
	if (tor < 0.0)	// should be +/- 120 deg (about +/- 2.1 rad).
	{
		cout << "LEU-fix!!!" << endl;
		
		// ok, this seems to be "wrong" way...
		// fix by exhanging 22<->23.
		
		int tmpid;
		
		tmpid = ref22->builder_res_id;
		ref22->builder_res_id = ref23->builder_res_id;
		ref23->builder_res_id = tmpid;
	}
}

// a hardcoded fix implemented:
// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^
// amino : fix the unambiguous VAL residues.

if (type == chn_info::amino_acid && resi_vector[tmp1[0]].symbol1 == 'V')
{
	atom * ref01 = NULL;
	atom * ref20 = NULL;
	atom * ref21 = NULL;
	atom * ref22 = NULL;
	
	for (i32s n4 = 0;n4 < (i32s) tdata.size();n4++)
	{
		if ((tdata[n4].ref->builder_res_id & 0xFF) == 0x01) ref01 = tdata[n4].ref;
		if ((tdata[n4].ref->builder_res_id & 0xFF) == 0x20) ref20 = tdata[n4].ref;
		if ((tdata[n4].ref->builder_res_id & 0xFF) == 0x21) ref21 = tdata[n4].ref;
		if ((tdata[n4].ref->builder_res_id & 0xFF) == 0x22) ref22 = tdata[n4].ref;
	}
	
	if (!ref01) { cout << "callEXIT : VAL-fix : ref01 not found!" << endl; exit(EXIT_FAILURE); }
	if (!ref20) { cout << "callEXIT : VAL-fix : ref20 not found!" << endl; exit(EXIT_FAILURE); }
	if (!ref21) { cout << "callEXIT : VAL-fix : ref21 not found!" << endl; exit(EXIT_FAILURE); }
	if (!ref22) { cout << "callEXIT : VAL-fix : ref22 not found!" << endl; exit(EXIT_FAILURE); }
	
	v3d<fGL> v1 = v3d<fGL>(ref20->GetCRD(0), ref01->GetCRD(0));
	v3d<fGL> v2 = v3d<fGL>(ref20->GetCRD(0), ref21->GetCRD(0));
	v3d<fGL> v3 = v3d<fGL>(ref21->GetCRD(0), ref22->GetCRD(0));
	fGL tor = v1.tor(v2, v3);
	
	if (tor < 0.0)	// should be +/- 120 deg (about +/- 2.1 rad).
	{
		cout << "VAL-fix!!!" << endl;
		
		// ok, this seems to be "wrong" way...
		// fix by exhanging 21<->22.
		
		int tmpid;
		
		tmpid = ref21->builder_res_id;
		ref21->builder_res_id = ref22->builder_res_id;
		ref22->builder_res_id = tmpid;
	}
}

// a hardcoded fix implemented:
// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^
// amino : fix the unambiguous TRP residues.

if (type == chn_info::amino_acid && resi_vector[tmp1[0]].symbol1 == 'W')
{
	atom * ref23 = NULL;
	atom * ref25 = NULL;
	atom * ref26 = NULL;
	atom * ref28 = NULL;
	atom * ref29 = NULL;
	
	for (i32s n4 = 0;n4 < (i32s) tdata.size();n4++)
	{
		if ((tdata[n4].ref->builder_res_id & 0xFF) == 0x23) ref23 = tdata[n4].ref;
		if ((tdata[n4].ref->builder_res_id & 0xFF) == 0x25) ref25 = tdata[n4].ref;
		if ((tdata[n4].ref->builder_res_id & 0xFF) == 0x26) ref26 = tdata[n4].ref;
		if ((tdata[n4].ref->builder_res_id & 0xFF) == 0x28) ref28 = tdata[n4].ref;
		if ((tdata[n4].ref->builder_res_id & 0xFF) == 0x29) ref29 = tdata[n4].ref;
	}
	
	if (!ref23) { cout << "callEXIT : TRP-fix : ref23 not found!" << endl; exit(EXIT_FAILURE); }
	if (!ref25) { cout << "callEXIT : TRP-fix : ref25 not found!" << endl; exit(EXIT_FAILURE); }
	if (!ref26) { cout << "callEXIT : TRP-fix : ref26 not found!" << endl; exit(EXIT_FAILURE); }
	if (!ref28) { cout << "callEXIT : TRP-fix : ref28 not found!" << endl; exit(EXIT_FAILURE); }
	if (!ref29) { cout << "callEXIT : TRP-fix : ref29 not found!" << endl; exit(EXIT_FAILURE); }
	
	bond b1 = bond(ref23, ref25, bondtype('S'));
	iter_bl itb1 = find(mdl->GetBondsBegin(), mdl->GetBondsEnd(), b1);
	bool b1exists = (itb1 != mdl->GetBondsEnd());
	
	bond b2 = bond(ref23, ref29, bondtype('S'));
	iter_bl itb2 = find(mdl->GetBondsBegin(), mdl->GetBondsEnd(), b2);
	bool b2exists = (itb2 != mdl->GetBondsEnd());
	
	if (b1exists == b2exists) { cout << "callEXIT : TRP-fix error #1" << endl; exit(EXIT_FAILURE); }
	if (b1exists && !b2exists)
	{
		cout << "TRP-fix!!!" << endl;
		
		// ok, this seems to be "wrong" way...
		// fix by exhanging 25<->29 and 26<->28.
		
		int tmpid;
		
		tmpid = ref25->builder_res_id;
		ref25->builder_res_id = ref29->builder_res_id;
		ref29->builder_res_id = tmpid;
		
		tmpid = ref26->builder_res_id;
		ref26->builder_res_id = ref28->builder_res_id;
		ref28->builder_res_id = tmpid;
	}
}

/////////////////////////////////////////////////////////////////////////// exceptions
/////////////////////////////////////////////////////////////////////////// exceptions
					
				}
				
				rcount++;
				acount += rsize;
			}
			
			// next we will write down the sequence...
			
			if (tmpseq1.size() != tmpseq3.size())
			{
				cout << "callEXIT : the seq1/seq3 arrays are bad!" << endl;
				exit(EXIT_FAILURE);
			}
			
			chn_info newinfo(type, tmpseq1.size());
			
			newinfo.id_mol = n1;
			newinfo.id_chn = mdl->ref_civ->size();
			
			for (i32u n3 = 0;n3 < tmpseq1.size();n3++)
			{
				newinfo.sequence1[n3] = tmpseq1[n3];				// a char...
				
				newinfo.sequence3[n3] = new char[strlen(tmpseq3[n3]) + 1];	// an array...
				strcpy(newinfo.sequence3[n3], tmpseq3[n3]);			// an array...
			}
			
			mdl->ref_civ->push_back(newinfo);
			
			// and make some output...
			
			ostringstream str;
			str << "chain " << newinfo.id_chn;
			str << ", length " << newinfo.length << ":" << endl;
			
			for (i32u n3 = 0;n3 < tmpseq1.size();n3++)
			{
				str << tmpseq3[n3] << " ";
				
				bool is_break = !((n3 + 1) % 10);
				bool is_end = ((n3 + 1) == tmpseq1.size());
				
				if (is_break || is_end) str << endl;
				
				// now free the text memory blocks...
				
				delete[] tmpseq3[n3];
				tmpseq3[n3] = NULL;
			}
			
			str << ends;
			if (mdl->verbosity >= 3) mdl->PrintToLog(str.str().c_str());
			
			// finally un-tag the main-chain bonds...
			
			for (i32s n3 = 0;n3 < ((i32s) path_vector[n2].size()) - 1;n3++)
			{
				iter_cl it1 = path_vector[n2][n3]->GetConnRecsBegin();
				while ((* it1).atmr != path_vector[n2][n3 + 1]) it1++;
				(* it1).bndr->flags[0] = false;
			}
		}
		
		// now check the HYDROGENS that are handed by model::AddHydrogens() and not
		// (except in some cases) by the sequence builder ; for hydrogens copy the res/atm
		// ID's from the heavy atom and set a special builder_res_id lower part (0xFF).
		
		// now for example a "select residue" operation will not skip the hydroges...
		
		for (iter_al it1 = range[0];it1 != range[1];it1++)
		{
			if ((* it1).el.GetAtomicNumber() != 1) continue;
			
			if ((* it1).GetConnRecCount() != 1)
			{
				cout << "WARNING : seqbuild : H atom with abnormal connectivity found." << endl;
				continue;
			}
			
			bool already_has_id = true;
			if ((* it1).id[1] < 0) already_has_id = false;
			if ((* it1).id[2] < 0) already_has_id = false;
			
			if (already_has_id != true)
			{
				atom * heavy = (* it1).GetFirstConnRec().atmr;
				if (heavy->id[1] < 0) continue;		// id < 0 means NOT_DEFINED...
				if (heavy->id[2] < 0) continue;		// id < 0 means NOT_DEFINED...
				
				(* it1).builder_res_id = (heavy->builder_res_id | 0xFF);
				
				(* it1).id[1] = heavy->id[1];		// chn id
				(* it1).id[2] = heavy->id[2];		// res id
			}
		}
	}
	
	path_vector.resize(0);
	
	// traditionally, we sorted the atom list here to make the newly assigned chains in nice order.
	// nowadays, this is done at model::UpdateChains() since there might be many different seqbuilder
	// objects (like for amino/nucleic acid variants) and we like to use them all...
}

void sequencebuilder::BuildResidue(sb_chain_descriptor * chde, model * mdl, sb_data_res * res)
{
	const int num_cs = mdl->GetCRDSetCount();
	
	for (i32u n1 = 0;n1 < res->atm_vector.size();n1++)
	{
		fGL crd[3];
		Convert(chde, & res->atm_vector[n1], crd);
		
		id_vector.push_back(res->atm_vector[n1].id);
		atom newatom(res->atm_vector[n1].el, crd, num_cs);
		
		newatom.formal_charge = res->atm_vector[n1].f_chrg;
		
		mdl->AddAtom(newatom);
		atom_vector.push_back(& mdl->GetLastAtom());
		allatm_vector.push_back(& mdl->GetLastAtom());
		
		i32s p_ind = 0;
		while (id_vector[p_ind] != res->atm_vector[n1].prev[0]) p_ind++;
		atom * prev = atom_vector[p_ind];
		
		bond newbond(prev, & mdl->GetLastAtom(), res->atm_vector[n1].bt);
		mdl->AddBond(newbond);
	}
	
	for (i32u n1 = 0;n1 < res->bnd_vector.size();n1++)
	{
		atom * tmp1[2]; i32s tmp2;
		tmp2 = 0; while (id_vector[tmp2] != res->bnd_vector[n1].atm[0]) tmp2++; tmp1[0] = atom_vector[tmp2];
		tmp2 = 0; while (id_vector[tmp2] != res->bnd_vector[n1].atm[1]) tmp2++; tmp1[1] = atom_vector[tmp2];
		
		bond newbond(tmp1[0], tmp1[1], res->bnd_vector[n1].bt);
		mdl->AddBond(newbond);
	}
}

void sequencebuilder::Convert(sb_chain_descriptor * chde, sb_data_atm * adata, fGL * crd)
{
//	cout << "Convert() begins..." << endl;
//	for (i32u n1 = 0;n1 < id_vector.size();n1++)
//	{
//		cout << "id_vector[" << n1 << "] ";
//		cout << "0x" << hex << id_vector[n1] << " ";
//		cout << atom_vector[n1] << endl;
//	}	int pause ; cin >> pause;
	
///////////////////////////////////////////////////////////////////////////
	
	atom * prev[3]; f64 ic[3];
	for (i32s n1 = 0;n1 < 3;n1++)
	{
		i32u n2 = 0;
		while (n2 < id_vector.size())
		{
			if (id_vector[n2] == adata->prev[n1]) break;
			else n2++;
		}
		
		if (n2 >= id_vector.size())
		{
			cout << "callEXIT : prev not found!" << endl;
			exit(EXIT_FAILURE);
		}
		
		prev[n1] = atom_vector[n2];
		ic[n1] = adata->ic1[n1];
	}
	
	const int tor_i = adata->ic2;
//	int tor_i = adata->ic2;			// for debug only...
	
	bool store = false;
	i32s p0i = NOT_DEFINED;
	i32s p1i = NOT_DEFINED;
	i32s p2i = NOT_DEFINED;
	
	if (tor_i > -1)
	{
		if (tor_i >= td_v.size())
		{
			cout << "callEXIT : td_v is incomplete ; " << tor_i << endl;
			exit(EXIT_FAILURE);
		}
		
		if (!td_v[tor_i].flag)
		{
			if (adata->id != td_v[tor_i].id[0])
			{
				cout << "callEXIT : id mismatch ; ";
				cout << hex << adata->id << " != ";
				cout << hex << td_v[tor_i].id[0] << endl;
				exit(EXIT_FAILURE);
			}
			
			td_v[tor_i].flag = true;
			store = true;
			
			if (fabs(ic[2]) > 0.0001)
			{
				cout << "callEXIT : tor error" << endl;
				exit(EXIT_FAILURE);
			}
		}
		
		if (type == chn_info::amino_acid)
		{
			if (adata->id == 0x10) goto skip_checks;	// a known issue...
			if (adata->id == 0x11) goto skip_checks;	// a known issue...
		}
		
		if (type == chn_info::nucleic_acid)
		{
			if (adata->id == 0x10) goto skip_checks;	// a known issue...
			if (adata->id == 0x11) goto skip_checks;	// a known issue...
		}
		
		// check prev0
		
		p0i = 0; while (p0i < (i32s) id_vector.size())
		{
			if (id_vector[p0i] == td_v[tor_i].id[1]) break;
			else p0i++;
		}
		
		if (p0i >= (i32s) id_vector.size())
		{
			cout << "callEXIT : p0i not found" << endl;
			exit(EXIT_FAILURE);
		}
		else if (atom_vector[p0i] != prev[0])
		{
			cout << "callEXIT : p0i mismatch ; ";
			cout << "0x" << hex << id_vector[p0i] << " " << prev[0] << endl;
			
		cout << "tordef " << hex << td_v[tor_i].id[0] << " " << hex << td_v[tor_i].id[1] << " ";
		cout << hex << td_v[tor_i].id[2] << " " << hex << td_v[tor_i].id[3] << endl;
			
			exit(EXIT_FAILURE);
		}
		
		// check prev1
		
		p1i = 0; while (p1i < (i32s) id_vector.size())
		{
			if (id_vector[p1i] == td_v[tor_i].id[2]) break;
			else p1i++;
		}
		
		if (p1i >= (i32s) id_vector.size())
		{
			cout << "callEXIT : p1i not found" << endl;
			exit(EXIT_FAILURE);
		}
		else if (atom_vector[p1i] != prev[1])
		{
			cout << "callEXIT : p1i mismatch ; ";
			cout << "0x" << hex << id_vector[p1i] << " " << prev[1] << endl;
			
		cout << "tordef " << hex << td_v[tor_i].id[0] << " " << hex << td_v[tor_i].id[1] << " ";
		cout << hex << td_v[tor_i].id[2] << " " << hex << td_v[tor_i].id[3] << endl;
			
			exit(EXIT_FAILURE);
		}
		
		// check prev2
		
		p2i = 0; while (p2i < (i32s) id_vector.size())
		{
			if (id_vector[p2i] == td_v[tor_i].id[3]) break;
			else p2i++;
		}
		
		if (p2i >= (i32s) id_vector.size())
		{
			cout << "callEXIT : p2i not found" << endl;
			exit(EXIT_FAILURE);
		}
		else if (atom_vector[p2i] != prev[2])
		{
			cout << "callEXIT : p2i mismatch ; ";
			cout << "0x" << hex << id_vector[p2i] << " " << prev[2] << endl;
			
		cout << "tordef " << hex << td_v[tor_i].id[0] << " " << hex << td_v[tor_i].id[1] << " ";
		cout << hex << td_v[tor_i].id[2] << " " << hex << td_v[tor_i].id[3] << endl;
			
			exit(EXIT_FAILURE);
		}
	}
	else
	{
	/*	for (i32u debug = 0;debug < td_v.size();debug++)	// for debug only...
		{
			bool all_match = true;
			tor_i = debug;
			
			// check prev0
			p0i = 0; while (p0i < (i32s) id_vector.size())
			{
				if (id_vector[p0i] == td_v[tor_i].id[1]) break;
				else p0i++;
			}
			
		if (p0i >= (i32s) id_vector.size() || atom_vector[p0i] != prev[0]) all_match = false;
		
			// check prev1
			p1i = 0; while (p1i < (i32s) id_vector.size())
			{
				if (id_vector[p1i] == td_v[tor_i].id[2]) break;
				else p1i++;
			}
			
		if (p1i >= (i32s) id_vector.size() || atom_vector[p1i] != prev[1]) all_match = false;
			
			// check prev2
			p2i = 0; while (p2i < (i32s) id_vector.size())
			{
				if (id_vector[p2i] == td_v[tor_i].id[3]) break;
				else p2i++;
			}
			
		if (p2i >= (i32s) id_vector.size() || atom_vector[p2i] != prev[2]) all_match = false;
			
			if (all_match)
			{
				cout << "callEXIT : all_match ; tor_i = " << tor_i << endl;
				exit(EXIT_FAILURE);
			}
		}
		
		tor_i = NOT_DEFINED;	*/
	}
	
	skip_checks:
	
	int c_crd_index = chde->c_crd_BGN;
	while (c_crd_index < chde->c_crd_END)
	{
		if (adata->id == chde->c_crd_v[c_crd_index].atm_id) break;
		else c_crd_index++;
	}
	
	if (c_crd_index < chde->c_crd_END)
	{
		// take the coordinates from constraints...
		// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
		
		crd[0] = chde->c_crd_v[c_crd_index].crdX;
		crd[1] = chde->c_crd_v[c_crd_index].crdY;
		crd[2] = chde->c_crd_v[c_crd_index].crdZ;
		
		if (tor_i > -1 && store)
		{
			v3d<fGL> v1 = v3d<fGL>(prev[0]->GetCRD(0), crd);
			v3d<fGL> v2 = v3d<fGL>(prev[0]->GetCRD(0), prev[1]->GetCRD(0));
			v3d<fGL> v3 = v3d<fGL>(prev[1]->GetCRD(0), prev[2]->GetCRD(0));
			fGL tor = v1.tor(v2, v3);
			
			if (tor_i >= chde->def_tor.size())
			{
				cout << "callEXIT : cannot store a torsion." << endl;
				exit(EXIT_FAILURE);
			}
			else
			{
				chde->def_tor[tor_i] = tor;
			}
		}
	}
	else
	{
		// calculate the coordinates from internal coordinates...
		// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
		
		if (tor_i > -1)
		{
			int c_tor_index = chde->c_tor_BGN;
			while (c_tor_index < chde->c_tor_END)
			{
				if (tor_i == chde->c_tor_v[c_tor_index].tor_ind) break;
				else c_tor_index++;
			}
			
			if (c_tor_index < chde->c_tor_END)
			{
				// use the constraint value, if available.
				
				ic[2] += chde->c_tor_v[c_tor_index].tor_val;
			}
			else if (tor_i < chde->def_tor.size())
			{
				// use the default table value, if available.
				
				ic[2] += chde->def_tor[tor_i];
			}
			else
			{
				// set a default value if the above fails;
				// the default geometry is the extended one,
				// EXCEPT for proline (15.0 deg and 0.0 deg).
				
				bool is_pro = false;
				if (chde->mode_1_letter)
				{
					if (chde->seq1->operator[](chde->curr_res) == 'P') is_pro = true;
				}
				else
				{
					if (!strcmp(chde->seq3->operator[](chde->curr_res), "PRO")) is_pro = true;
				}
				
				if (is_pro)
				{
					switch (tor_i)
					{
						case +3:
						ic[2] += 15.0 * M_PI / 180.0;
						break;
						
						case +4:
						ic[2] += 0.0 * M_PI / 180.0;
						break;
						
						default:
						ic[2] += 180.0 * M_PI / 180.0;
					}
				}
				else
				{
					ic[2] += 180.0 * M_PI / 180.0;
				}
			}
		}
		
		v3d<fGL> v1 = v3d<fGL>(prev[0]->GetCRD(0), prev[1]->GetCRD(0));
		
		v3d<fGL> tmp1 = v3d<fGL>(prev[1]->GetCRD(0), prev[2]->GetCRD(0));
		v3d<fGL> tmp2 = v1 * (tmp1.spr(v1) / v1.spr(v1));
		v3d<fGL> v2 = tmp1 - tmp2;
		
		v3d<fGL> v3 = v2.vpr(v1);
		
		v1 = v1 * (ic[0] * cos(ic[1]) / v1.len());
		v2 = v2 * (ic[0] * sin(ic[1]) * cos(ic[2]) / v2.len());
		v3 = v3 * (ic[0] * sin(ic[1]) * sin(ic[2]) / v3.len());
		
		v3d<fGL> tmp3 = v3d<fGL>(prev[0]->GetCRD(0)); tmp3 = tmp3 + (v1 + (v2 + v3));
		for (i32s n1 = 0;n1 < 3;n1++) crd[n1] = tmp3.data[n1];
	}
}

void sequencebuilder::BuildTemplate(vector<sb_tdata> & tdata, i32s res, bool is_first, bool is_last)
{
	BuildPartialT(tdata, main_vector);
	if (!is_last) BuildPartialT(tdata, conn_vector);
	
	if (is_first) BuildPartialT(tdata, mod[1]->atm_vector);		// head-mods
	else if (is_last) BuildPartialT(tdata, mod[2]->atm_vector);	// tail-mods
	else BuildPartialT(tdata, mod[0]->atm_vector);			// default...
	
	BuildPartialT(tdata, resi_vector[res].atm_vector);
}

void sequencebuilder::BuildPartialT(vector<sb_tdata> & tdata, vector<sb_data_atm> & adata)
{
	for (i32u n1 = 0;n1 < adata.size();n1++)
	{
		sb_tdata newdata;
		newdata.id[0] = adata[n1].id;
		
		if (adata[n1].prev[0] & 0xFF00) newdata.id[1] = NOT_DEFINED;
		else newdata.id[1] = adata[n1].prev[0];
		
		newdata.el = adata[n1].el;
		newdata.bt = adata[n1].bt;
		newdata.ref = NULL;
		
		tdata.push_back(newdata);
	}
}

// idea is that this path-searching function should be used only inside a molecule.
// this way we are sure to find always at least one path. we collect references to
// atoms for all paths and collect this way all possible paths.

// different flags should be used in type checking and in this path-searching...

void sequencebuilder::FindPath(model * mdl, atom * ref1, atom * ref2, i32u index)
{
	if (index >= main_vector.size() + conn_vector.size()) index = 0;
	
	if (index < main_vector.size())
	{
		if (ref1->el.GetAtomicNumber() != main_vector[index].el.GetAtomicNumber()) return;
		if (!main_vector[index].tr->Check(mdl, ref1, 0)) return;
	}
	else
	{
		i32u index2 = index - main_vector.size();
		
		if (ref1->el.GetAtomicNumber() != conn_vector[index2].el.GetAtomicNumber()) return;
		if (!conn_vector[index2].tr->Check(mdl, ref1, 0)) return;
	}
	
	tmpatm_vector.push_back(ref1);
	
//for (i32u debug = 0;debug < tmpatm_vector.size();debug++)
//{
//	cout << tmpatm_vector[debug]->el.GetSymbol();
//}	cout << endl;
	
	if (ref1 == ref2) path_vector.push_back(tmpatm_vector);
	else
	{
		for (iter_cl it1 = ref1->GetConnRecsBegin();it1 != ref1->GetConnRecsEnd();it1++)
		{
			if ((* it1).bndr->flags[2]) continue;
			
			(* it1).bndr->flags[2] = true;
			FindPath(mdl, (* it1).atmr, ref2, (index + 1));
			(* it1).bndr->flags[2] = false;
		}
	}
	
	tmpatm_vector.pop_back();
}

bool sequencebuilder::CheckTemplate(vector<sb_tdata> & tdata, i32s flag)
{
	vector<i32s> tmpv1;
	
	// look for a suitable starting point to start matching in the template.
	// if there is no such point we assume the template fits. however, the template
	// could match since it is too small (for exaple, the template for glycine will
	// match for all residues). therefore we have to test all templates and choose
	// the biggest matching one!!!
	
	i32u index = 0;
	while (index < tdata.size())
	{
		if (tdata[index].ref != NULL)
		{
			tmpv1.resize(0);
			for (i32u n1 = 0;n1 < tdata.size();n1++)
			{
				bool test1 = (tdata[n1].ref == NULL);
				bool test2 = (tdata[n1].id[1] == tdata[index].id[0]);
				if (test1 && test2) tmpv1.push_back(n1);
			}
			
			if (tmpv1.size()) break;
		}
		
		index++;
	}
	
	// if there is no suitable starting point, we have tested all atoms and
	// none of them is missing -> template fits -> TRUE!!!
	
	if (index == tdata.size()) return true;
	
	// find all possible bonds that we can use to continue testing...
	
	vector<crec> tmpv2; iter_cl it1;
	for (it1 = tdata[index].ref->GetConnRecsBegin();it1 != tdata[index].ref->GetConnRecsEnd();it1++)
	{
		bool test1 = (* it1).bndr->flags[flag];
		if (!test1) tmpv2.push_back((* it1));
	}
	
	// if there are less bonds than we need, the template can't match -> FALSE!!!
	
	if (tmpv2.size() < tmpv1.size()) return false;
	
	vector<i32s> tmpv3; tmpv3.resize(tmpv2.size());
	for (i32u n1 = 0;n1 < tmpv3.size();n1++) tmpv3[n1] = n1;
	
	// then we will check all bonds against the template in all possible permutations.
	// if some combination is a match then leave the ID-numbers untouched, clean the
	// bond flags and leave -> TRUE!!!
	
	while (true)
	{
		bool test = true;
		for (i32u n1 = 0;n1 < tmpv1.size();n1++)
		{
			i32s el = tdata[tmpv1[n1]].el.GetAtomicNumber();
			if (el != NOT_DEFINED && el != tmpv2[tmpv3[n1]].atmr->el.GetAtomicNumber()) test = false;
			
			i32s bt = tdata[tmpv1[n1]].bt.GetValue();
			if (bt != NOT_DEFINED && bt != tmpv2[tmpv3[n1]].bndr->bt.GetValue()) test = false;
		}
		
		if (test)
		{
			for (i32u n1 = 0;n1 < tmpv1.size();n1++)
			{
				tmpv2[tmpv3[n1]].bndr->flags[flag] = true;
				tdata[tmpv1[n1]].ref = tmpv2[tmpv3[n1]].atmr;
			}
			
			bool result = CheckTemplate(tdata, flag);
			
			for (i32u n1 = 0;n1 < tmpv1.size();n1++)
			{
				tmpv2[tmpv3[n1]].bndr->flags[flag] = false;
				if (!result) tdata[tmpv1[n1]].ref = NULL;
			}
			
			if (result) return true;
		}
		
		if (!next_permutation(tmpv3.begin(), tmpv3.end())) break;
	}
	
	// none of those combinations matched -> the template doesn't fit -> FALSE!!!
	
	return false;
}

/*################################################################################################*/

sb_data_td::sb_data_td(void)
{
	id[0] = NOT_DEFINED;
	id[1] = NOT_DEFINED;
	id[2] = NOT_DEFINED;
	id[3] = NOT_DEFINED;
	
	flag = false;
}

sb_data_td::~sb_data_td(void)
{
}

istream & operator>>(istream & p1, sb_data_td & p2)
{
	char buffer[256];
	while (p1.get() != 'F');
	
	p1 >> p2.id[0];
	p1 >> p2.id[1];
	p1 >> p2.id[2];
	p1 >> p2.id[3];
	
	p2.flag = false;
	
	return p1;
}

/*################################################################################################*/

sb_data_atm::sb_data_atm(void)
{
	id = NOT_DEFINED;
	
	el = element(1);
	f_chrg = +0;
	
	bt = bondtype(1);
	
	ic2 = NOT_DEFINED;
	for (i32s n1 = 0;n1 < 3;n1++)
	{
		prev[n1] = NOT_DEFINED;
		ic1[n1] = 0.0;
	}
	
	tr = NULL;
}

sb_data_atm::sb_data_atm(const sb_data_atm & p1)
{
	id = p1.id;
	
	el = p1.el;
	f_chrg = p1.f_chrg;
	
	bt = p1.bt;
	
	ic2 = p1.ic2;
	for (i32s n1 = 0;n1 < 3;n1++)
	{
		prev[n1] = p1.prev[n1];
		ic1[n1] = p1.ic1[n1];
	}
	
	if (p1.tr != NULL) tr = new typerule(* p1.tr);
	else tr = NULL;
}

sb_data_atm::~sb_data_atm(void)
{
	if (tr != NULL) delete tr;
}

istream & operator>>(istream & p1, sb_data_atm & p2)
{
	char buffer[256];
	while (p1.get() != 'M');
	
	p1 >> p2.id;
	p1 >> buffer; p2.el = element(buffer);
	p1 >> p2.f_chrg;
	
	p1 >> p2.prev[0] >> p2.prev[1] >> p2.prev[2];
	p1 >> p2.ic1[0] >> p2.ic1[1] >> p2.ic2 >> p2.ic1[2];
	
	p2.ic1[1] = M_PI * p2.ic1[1] / 180.0;
	p2.ic1[2] = M_PI * p2.ic1[2] / 180.0;
	
	p1 >> buffer;
	p2.bt = bondtype(buffer[0]);
	
///////////////////////////////////////////////////////////////////////////////////////////////
if (p2.bt.GetValue() == 0) { cout << "callEXIT : bad bondtype A" << endl; exit(EXIT_FAILURE); }
///////////////////////////////////////////////////////////////////////////////////////////////
	
	return p1;
}

/*################################################################################################*/

sb_data_bnd::sb_data_bnd(void)
{
	atm[0] = NOT_DEFINED;
	atm[1] = NOT_DEFINED;
	bt = bondtype(1);
}

sb_data_bnd::~sb_data_bnd(void)
{
}

istream & operator>>(istream & p1, sb_data_bnd & p2)
{
	char buffer[256];
	while (p1.get() != 'D');
	
	p1 >> p2.atm[0] >> p2.atm[1];
	p1 >> buffer; p2.bt = bondtype(buffer[0]);
	
///////////////////////////////////////////////////////////////////////////////////////////////
if (p2.bt.GetValue() == 0) { cout << "callEXIT : bad bondtype B" << endl; exit(EXIT_FAILURE); }
///////////////////////////////////////////////////////////////////////////////////////////////
	
	return p1;
}

/*################################################################################################*/

sb_data_res::sb_data_res(void)
{
	symbol1 = '?';
	strcpy(symbol3, "???");
	
	description = NULL;
}

sb_data_res::sb_data_res(const sb_data_res & p1)
{
	id = p1.id;
	
	symbol1 = p1.symbol1;
	strcpy(symbol3, p1.symbol3);
	
	if (p1.description != NULL)
	{
		description = new char[strlen(p1.description) + 1];
		strcpy(description, p1.description);
	}
	else description = NULL;
	
	td_vector = p1.td_vector;
	atm_vector = p1.atm_vector;
	bnd_vector = p1.bnd_vector;
}

sb_data_res::~sb_data_res(void)
{
	if (description != NULL) delete[] description;
}

void sb_data_res::ReadModification(istream & p1)
{
	char buffer[256];
	while (p1.peek() != 'E')
	{
		if (p1.peek() == 'T')
		{
			cout << "callEXIT : xxxx_MOD should not have any TORDEF lines!" << endl;
			exit(EXIT_FAILURE);
		}
		
		if (p1.peek() == 'A')
		{
			sb_data_atm newatm; p1 >> newatm;
			p1.getline(buffer, sizeof(buffer));
			atm_vector.push_back(newatm);
			continue;
		}
		
		if (p1.peek() == 'B')
		{
			sb_data_bnd newbnd; p1 >> newbnd;
			p1.getline(buffer, sizeof(buffer));
			bnd_vector.push_back(newbnd);
			continue;
		}
		
		p1.getline(buffer, sizeof(buffer));
	}
	
	p1.getline(buffer, sizeof(buffer));
}

istream & operator>>(istream & p1, sb_data_res & p2)
{
	char buffer[256];
	
	while (p1.get() != 'S');
	
	p1 >> p2.id;
	
	while (p1.get() != ':');
	
	p1 >> buffer;
	if (strlen(buffer) != 3)
	{
		cout << "callEXIT : bad symbol3 : " << buffer << endl;
		exit(EXIT_FAILURE);
	}
	
	strcpy(p2.symbol3, buffer);
	p1 >> p2.symbol1;
	
	while (p1.get() != '"');
	p1.getline(buffer, sizeof(buffer), '"');
	
	p2.description = new char[strlen(buffer) + 1];
	strcpy(p2.description, buffer);
	
	p1.getline(buffer, sizeof(buffer));
	
	while (p1.peek() != 'E')
	{
		if (p1.peek() == 'T')
		{
			sb_data_td newtd; p1 >> newtd;
			p1.getline(buffer, sizeof(buffer));
			p2.td_vector.push_back(newtd);
			continue;
		}
		
		if (p1.peek() == 'A')
		{
			sb_data_atm newatm; p1 >> newatm;
			p1.getline(buffer, sizeof(buffer));
			p2.atm_vector.push_back(newatm);
			continue;
		}
		
		if (p1.peek() == 'B')
		{
			sb_data_bnd newbnd; p1 >> newbnd;
			p1.getline(buffer, sizeof(buffer));
			p2.bnd_vector.push_back(newbnd);
			continue;
		}
		
		p1.getline(buffer, sizeof(buffer));
	}
	
	p1.getline(buffer, sizeof(buffer));
	return p1;
}

/*################################################################################################*/

// eof
