#ifndef INDIBASE_H
#define INDIBASE_H

#include "indiapi.h"
#include "indidevapi.h"

#define MAXRBUF 2048

/**
 * \namespace INDI
   \brief Namespace to encapsulate INDI client, drivers, and mediator classes.
   Developers can subclass the base devices class to implement device specific functionality. This ensures interoperability and consistency among devices within the same family
   and reduces code overhead.

   <ul>
   <li>BaseClient: Base class for INDI clients. By subclassing BaseClient, client can easily connect to INDI server
   and handle device communication, command, and notifcation.</li>
   <li>BaseMediator: Abstract class to provide interface for event notifications in INDI::BaseClient.</li>
   <li>BaseDriver: Base class for all INDI virtual driver as handled and stored in INDI::BaseClient.</li>
   <li>DefaultDriver: INDI::BaseDriver with extended functionality such as debug, simulation, and configuration support.
       It is \e only used by drivers directly, it cannot be used by clients.</li>
   <li>FilterInterface: Basic interface for filter wheels functions.</li>
   <li>GuiderInterface: Basic interface for guider (ST4) port functions.</li>
   <li>CCD: Base class for CCD drivers. Provides basic support for single chip CCD and CCDs with a guide head as well.</li>
   <li>Telescope: Base class for telescope drivers.</li>
   <li>FilterWheel: Base class for Filter Wheels. It implements the FilterInterface.</li>
   <li>Focuser: Base class for focusers.</li>
   <li>USBDevice: Base class for USB devices for direct read/write/control over USB.</li>
   </ul>
 */
namespace INDI
{
    class BaseMediator;
    class BaseClient;
    class BaseDriver;
    class DefaultDriver;
    class FilterInterface;
    class GuiderInterface;
    class CCD;
    class Telescope;
    class FilterWheel;
    class Focuser;
    class USBDevice;   
}


/**
 * \class INDI::BaseMediator
   \brief Meditates event notification as generated by driver and passed to clients.
*/
class INDI::BaseMediator
{
public:

    /** \brief Emmited when a new device is created from INDI server.
        \param device_name Name of the new device
    */
    virtual void newDevice(const char *device_name)  =0;

    /** \brief Emmited when a new property is created for an INDI driver.
        \param device_name Name of the device
        \param property_name Name of the new property
    */
    virtual void newProperty(const char *device_name, const char *property_name)  =0;

    /** \brief Emmited when a new BLOB value arrives from INDI server.
        \param bp Pointer to filled and process BLOB.
    */
    virtual void newBLOB(IBLOB *bp) =0;

    /** \brief Emmited when a new switch value arrives from INDI server.
        \param svp Pointer to a switch vector property.
    */
    virtual void newSwitch(ISwitchVectorProperty *svp) =0;

    /** \brief Emmited when a new number value arrives from INDI server.
        \param nvp Pointer to a number vector property.
    */
    virtual void newNumber(INumberVectorProperty *nvp) =0;

    /** \brief Emmited when a new text value arrives from INDI server.
        \param tvp Pointer to a text vector property.
    */
    virtual void newText(ITextVectorProperty *tvp) =0;

    /** \brief Emmited when a new light value arrives from INDI server.
        \param lvp Pointer to a light vector property.
    */
    virtual void newLight(ILightVectorProperty *lvp) =0;

    /** \brief Emmited when the server is connected.
    */
    virtual void serverConnected() =0;

    /** \brief Emmited when the server gets disconnected.
    */
    virtual void serverDisconnected() =0;
};

#endif // INDIBASE_H
