/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/


#include <cstdio>
#include <cstdlib>
#include <iostream>

#include <libinstrudeo/isdvideocanvas.h>
#include <libinstrudeo/isdvideoproperties.h>
#include <libinstrudeo/isdseekbackcalculator.h>
#include <libinstrudeo/isdutils.h>
#include <libinstrudeo/isdrectangle.h>
#include <libinstrudeo/isdlogger.h>
#include <libinstrudeo/isddatafile.h>

//-----CONSTRUCTORS-----
ISDDataFile::ISDDataFile(string dataFileName, ISDVideoProperties*& videoProperties_, bool calcSeekBacks)
    : ISDObject(), fileName(dataFileName),
      rectListHead(NULL), rectListTail(NULL), currentRect(NULL), videoProperties(NULL)
{
    videoProperties_ = NULL;

    //open the file stream
    inputStream.open(dataFileName.c_str(), ios::binary);
    if (!inputStream.good()) {
	lastError = ISD_FILE_ERROR;
	return;
    }

    //this initializes the videoProperties variable too
    if (preprocessData()!=ISD_SUCCESS) {
	LOG_WARNING("Error while preprocessing data file.");
	lastError = ISD_INIT_ERROR;
	return;
    }

    //return the object
    videoProperties_ = this->videoProperties;
}

//-----DESTRUCTOR-----
ISDDataFile::~ISDDataFile()
{
    inputStream.close();
    
    //delete the rectlist
    ISDRectangle* iterRect = rectListHead;
    ISDRectangle* bakRect = rectListHead;
    while (iterRect!=NULL){
	bakRect = iterRect->getNext();
	delete iterRect;
	iterRect = bakRect;
    }
}

//-----PUBLIC STATIC METHODS-----
string ISDDataFile::getMagicString()
{
    return ISD_MAGIC;
}

//-----PUBLIC METHODS-----
string ISDDataFile::getFileName()
{
    return fileName;
}
ISDObject::ISDErrorCode ISDDataFile::updateCanvas(ISDVideoCanvas* videoCanvas, int pos, bool& noChange)
{
    ISDErrorCode retVal;

    //optimistic initialisation
    noChange = true;
    
    //check for errors
    if (rectListHead==NULL || currentRect==NULL || !inputStream.good()) {
	LOG_WARNING("Tried to access the videostream inside the datafile without initialising it.");
	RETURN_ERROR(ISD_INIT_ERROR);
    }

    //position must be between the video's bounds
    if (pos < 0 || pos > videoProperties->getLength()) {
	LOG_WARNING("Trying to read at a position outside the video bounds.");
	RETURN_ERROR(ISD_ARGS_ERROR);
    }
    
    //we need to let the first rect fall through
    if (pos==currentRect->frameTime && currentRect->getPrevious()!=NULL) {
	RETURN_SUCCESS;
    }

    /*
     * Check if the requested position if before or after the current position.
     * If the new pos is after the current one, we don't need to use seekbacks.
     */
    if (pos >= currentRect->frameTime) {
	//move forward, until we are at the right position
	while ((currentRect->getNext() != NULL && currentRect->getNext()->frameTime <= pos) ||
	       currentRect->frameTime==0) {
	    currentRect = currentRect->getNext();
	    char data[currentRect->dataLen];
	    
	    if ((retVal = readData(currentRect, data)) != ISD_SUCCESS) {
		LOG_WARNING("Error while reading rect-data from disk.");
		RETURN_ERROR(retVal);
	    }
	    if ((retVal = videoCanvas->rectangleUpdate(currentRect, data)) != ISD_SUCCESS) {
		LOG_WARNING("Error while rect-updateing image.");
		RETURN_ERROR(retVal);
	    }
	    
	    noChange = false;
	}
	
	RETURN_SUCCESS;
    }
    //requested position is before current position
    else {
	/*
	 * Search the rectangle at position pos,
	 * load the seekBack rect of that rect in the currentRect
	 * and call this method again. This time, the pos will be > currentRect->frameTime,
	 * and the image will be rebuild, starting from the seekBack rect.
	 *
	 * This has been improved, to support seeking of large skips by remembering the the smallest
	 * seekback we encounter when going from the current position to the target position.
	 */
	int seekbackPos = currentRect->getSeekBackRectPos();
	while (currentRect->getPrevious() != NULL && currentRect->getPrevious()->frameTime >= pos) {
	    currentRect = currentRect->getPrevious();
	    if (currentRect->getSeekBackRectPos() < seekbackPos) {
		seekbackPos = currentRect->getSeekBackRectPos();
	    }

	    noChange = false;
	}

	//if this is the first frame, don't stop
	if (noChange && currentRect->getPrevious() != NULL) {
	    RETURN_SUCCESS;
	}
	
	/*
	 * search the seekback rect in the map
	 */
	map<int, ISDRectangle*>::iterator iter;
	iter = rectMap.find(seekbackPos);
	if (iter==rectMap.end()) {
	    LOG_WARNING("Couldn't get seekback rect from rectMap register.");
	    RETURN_ERROR(ISD_FAILURE);
	}

	ISDRectangle* seekBackRect = (*iter).second;
		
	if (seekBackRect!=NULL) {
	    //load the seekback rect
	    char data[seekBackRect->dataLen];
	    if ((retVal = readData(seekBackRect, data)) != ISD_SUCCESS) {
		LOG_WARNING("Error while reading seekback rect-data from disk.");
		RETURN_ERROR(retVal);
	    }
	    
	    if ((retVal = videoCanvas->rectangleUpdate(seekBackRect, data)) != ISD_SUCCESS) {
		LOG_WARNING("Error while seekback-rect-updateing image.");
		RETURN_ERROR(retVal);
	    }

	    videoCanvas->display();

	    //this check prevents an infinite loop
	    if (seekBackRect==currentRect) {
		currentRect = seekBackRect;
		//don't use recursive call if we know what will happen.
		RETURN_SUCCESS;
	    }
	    else {
		currentRect = seekBackRect;
		//call this method again with the (loaded) seekback rect as the current rect
		return updateCanvas(videoCanvas, pos, noChange);
	    }
	}
	else {
	    LOG_WARNING("Couldn't get seekback rect from rectMap register.");
	    RETURN_ERROR(ISD_FAILURE);   
	}

	RETURN_ERROR(ISD_FAILURE);
    }

    RETURN_SUCCESS;
}
ISDRectangle* ISDDataFile::getFirstRectangle()
{
    if (rectMap.size()>0) {
	//the rectMap maps filepositions to rectangles and sorts on keys,
	//so the first one is the first rectangle
	return rectMap.begin()->second;
    }
    else {
	return NULL;
    }
}
ISDObject::ISDErrorCode ISDDataFile::getPixelData(ISDRectangle* rect, char* data)
{
    //just a wrapper around the protected method
    return readData(rect, data);
}

//-----PRIVATE METHODS-----
ISDObject::ISDErrorCode ISDDataFile::preprocessData()
{
    ISDObject::ISDErrorCode retVal;
    int minUpdate = -1;

    //read magic codes and check if compatible
    string magic = ISD_MAGIC_TEMPLATE;
    if ((retVal = readMagic(magic)) != ISD_SUCCESS) {
	RETURN_ERROR(retVal);
    }
	
    //note: version numbers are encoded in ascii and at most 1 char long each
    int majorVersion = static_cast<int>(magic[9])-48;
    int minorVersion = static_cast<int>(magic[11])-48;
    if (magic.substr(0, 9)!=string(ISD_MAGIC_TEMPLATE).substr(0, 9) || 
	majorVersion>MAX_MAJOR_SUPPORTED_VERSION || 
	minorVersion>MAX_MINOR_SUPPORTED_VERSION)
    {
	RETURN_ERROR(ISD_VERSION_ERROR);
    }

    //read in the file header
    if ((retVal = readFileHeader()) != ISD_SUCCESS) {
	RETURN_ERROR(retVal);
    }

    ISDRectangleHeader rectHeader;
    ISDRectangle* rect;
    while ((retVal = readRectHeader(rectHeader)) == ISD_SUCCESS) {
	//check if this is the first element
	if (rectListHead==NULL) {
	    //the seekback of the first rect is the first rect
	    //Note: we already read in the header; the filepos is before that header
	    rect = new ISDRectangle(rectHeader, (int)inputStream.tellg()-ISD_RECT_HEADER_SIZE, NULL, NULL);
	    //the list points to the first rect
	    rectListHead = rect;
	    rectListTail = rect;
	    //init the currentRect so it points to the first rect
	    currentRect = rect;

	    //skip the data
	    inputStream.seekg(rectHeader.datalen, ios::cur);
	}
	else {
	    //add a new rect to the end of the double linked list
	    //Note: we already read in the header; the filepos is before that header
	    rect = new ISDRectangle(rectHeader, (int)inputStream.tellg()-ISD_RECT_HEADER_SIZE, rectListTail, NULL);
	    rectListTail->setNext(rect);
	    rectListTail = rect;
	    
	    //skip the data
	    inputStream.seekg(rectHeader.datalen, ios::cur);

	    /*
	     * adjust the minimum-update interval if necessary
	     * if minUpdate == -1, the value hasn't been initialised before
	     * if the interval == 0, don't update the value; this just means the two rects
	     *          need to be updated simultaneously
	     */
	    int newInterval = rect->frameTime - rect->getPrevious()->frameTime;
	    if (newInterval!=0 && (minUpdate==-1 || newInterval<minUpdate)) {
		minUpdate = newInterval;
	    }
	}

	//add an entry to the rectMap for this new rect
	rectMap.insert(pair<int, ISDRectangle*>(rect->filePos, rect));
    }
    
    //the only good exit value is EOF
    if (retVal!=ISD_EOF_ERROR){
	RETURN_ERROR(retVal);
    }

    //init the video properties
    videoProperties = new ISDVideoProperties(fileHeader, rectListTail->frameTime, minUpdate);

    RETURN_SUCCESS;
}
ISDObject::ISDErrorCode ISDDataFile::readMagic(string& magic)
{
    if (inputStream.good()) {
	//magic always occurs at the beginning of the file
	inputStream.seekg(0);
	char buf[magic.length()];
	inputStream.read(buf, magic.length());
	magic.assign(buf, magic.length());
	RETURN_SUCCESS;
    }
    else {
	RETURN_ERROR(ISD_FILE_ERROR);
    }
}
ISDObject::ISDErrorCode ISDDataFile::readFileHeader()
{
    if (inputStream.good()) {
	inputStream.read(reinterpret_cast<char*>(&fileHeader), ISD_FILE_HEADER_SIZE);
       
	fileHeader.framebufferWidth = Swap16IfLE(fileHeader.framebufferWidth);
	fileHeader.framebufferHeight = Swap16IfLE(fileHeader.framebufferHeight);
	fileHeader.format.redMax = Swap16IfLE(fileHeader.format.redMax);
	fileHeader.format.greenMax = Swap16IfLE(fileHeader.format.greenMax);
	fileHeader.format.blueMax = Swap16IfLE(fileHeader.format.blueMax);

	RETURN_SUCCESS;
    }
    else {
	RETURN_ERROR(ISD_FILE_ERROR);
    }
}
ISDObject::ISDErrorCode ISDDataFile::readRectHeader(ISDRectangleHeader& header)
{
    if (inputStream.good()) {
	/*
	 * We can't use the header struct as a databuffer to read the whole header
	 * in one pass, because of the odd size (ISD_RECT_HEADER_SIZE bytes). Structs
	 * are aligned at 2 or 4 bytes in memory, corrupting the seekBackPos entry in the
	 * struct because of additional bytes.
	 */
	inputStream.read(reinterpret_cast<char*>(&header.frameTime), sizeof(U32));
       	inputStream.read(reinterpret_cast<char*>(&header.x), sizeof(U16));
	inputStream.read(reinterpret_cast<char*>(&header.y), sizeof(U16));
	inputStream.read(reinterpret_cast<char*>(&header.w), sizeof(U16));
	inputStream.read(reinterpret_cast<char*>(&header.h), sizeof(U16));
	inputStream.read(reinterpret_cast<char*>(&header.datalen), sizeof(U32));
	inputStream.read(reinterpret_cast<char*>(&header.encType), sizeof(U8));
	inputStream.read(reinterpret_cast<char*>(&header.seekBackPos), sizeof(U32));
	
	//check for EOF; clean return
	if (inputStream.eof()){
	    //don't save the errorcode, this is no error !!
	    //reset the error flags
	    inputStream.clear();
	    return ISD_EOF_ERROR;
	}

	header.frameTime = Swap32IfLE(header.frameTime);
	header.x = Swap16IfLE(header.x);
	header.y = Swap16IfLE(header.y);
	header.w = Swap16IfLE(header.w);
	header.h = Swap16IfLE(header.h);
	header.datalen = Swap32IfLE(header.datalen);
	header.seekBackPos = Swap32IfLE(header.seekBackPos);

	RETURN_SUCCESS;
    }
    else {
	RETURN_ERROR(ISD_FILE_ERROR);
    }
}
ISDObject::ISDErrorCode ISDDataFile::readData(ISDRectangle* rect, char* data)
{
    if (inputStream.good() && rect!=NULL && data != NULL) {
	//position the file pointer
	inputStream.seekg(rect->filePos+ISD_RECT_HEADER_SIZE);
	inputStream.read(data, rect->dataLen);

	RETURN_SUCCESS;
    }
    else {
	RETURN_ERROR(ISD_FILE_ERROR);
    }
}


