/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#ifndef ISDRECT_H
#define ISDRECT_H

/**
 * Class represents an item in a list of rectangles, used by the 
 * ISDDataFile class to preprocess the input file structure with the
 * goal to minimize seek times. It's data is identical to the data in the
 * ISDRectangleHeader struct, but it provides some useful methods for the above
 * purpose.
 *
 * @author Bram Biesbrouck <b@beligum.org>
 */

#include <list>

#include <libinstrudeo/isdobject.h>
#include <libinstrudeo/isddatafile.h>

#undef LOG_HEADER
#define LOG_HEADER "Error while accessing rect header object: \n"
#include <libinstrudeo/isdloggermacros.h>

using namespace std;

class ISDRectangle : ISDObject
{

 public:
    //-----CONSTRUCTORS-----
    /**
     * Constructs a standard header object.
     *
     * @param frameTime The time, relative to the first frame in the movie, in milliseconds.
     * @param x The x pos in the framebuffer.
     * @param y The y pos in the framebuffer.
     * @param width The width of the rectangle.
     * @param height The height of the rectangle.
     * @param dataLen The length of the (encoded) data on disk.
     * @param encType The type of encoding of the data on disk.
     * @param filePos The filepointer position where the header on disk starts.
     * @param previous The previous header in the video-sequence or NULL if this is the first.
     * @param next The next header in the video-sequence or NULL if this is the last.
     * @param seekBackPos The seekBack rectangle filepointer position (where the header starts) of this rect;
     *                     use the first rectangle in the file for default (slow) behaviour.
     */
    ISDRectangle(int frameTime, int x, int y, int width, int height,
	    int dataLen, int encType, int filePos,
	    ISDRectangle* previous, ISDRectangle* next, int seekBackPos);
    /**
     * Convenience-constructor that does the exact same as the one above.
     *
     * @param header The header structure.
     * @param filePos The filepointer position where the header on disk starts.
     * @param previous The previous header in the video-sequence or NULL if this is the first.
     * @param next The next header in the video-sequence or NULL if this is the last.
     */
    ISDRectangle(ISDDataFile::ISDRectangleHeader& header, int filePos, ISDRectangle* previous, ISDRectangle* next);
    
    /**
     * Copy constructor that makes a shallow copy of all member data.
    ISDRectangle(ISDRectangle rect)
    */

    ~ISDRectangle();
    
    //-----SETTERS-----
    /**
     * Sets the next node in the linked list, NULL if this is the end node
     *
     * @param rect The next element.
     */
    void setNext(ISDRectangle* rect);
    
    /**
     * Sets the previous node in the linked list, NULL if this is the begin node
     *
     * @param rect The previous element.
     */
    void setPrevious(ISDRectangle* rect);

    /**
     * Sets the node in the list from where the image must be rebuild to
     * get a valid image at this node-time. See ISDDataFile for details.
     * @see ISDDataFile::calcSeekBackRects()
     *
     * @param filePos The seekBack rectangle filepointer position (where the header starts).
     */
    void setSeekBackRectPos(int filePos);

    //-----GETTERS-----
    /**
     * Returns the next node in the linked list, NULL if this is the end node
     *
     * @return The next element.
     */
    ISDRectangle* getNext();
    /**
     * Returns the previous node in the linked list, NULL if this is the end node
     *
     * @return The previous element.
     */
    ISDRectangle* getPrevious();

    /**
     * Gets the node in the list from where the image must be rebuild to
     * get a valid image at this node-time. See ISDDataFile for details.
     * @see ISDDataFile::calcSeekBackRects()
     *
     * @return The seekBack rectangle filepointer position (where the header starts).
     */
    int getSeekBackRectPos();

    /**
     * Returns a ISDRectangleHeader structure, filled with the values supplied to this rect header.
     *
     * @return The ISDRectangleHeader structure.
     */
    ISDDataFile::ISDRectangleHeader getISDRectangleHeader();

    //-----PUBLIC VARIABLES-----
    /*
     * These are made public for the ease of some of the functions
     */
    int x;
    int y;
    int width;
    int height;
    /*
     * The relative time (in milliseconds) of this rect to the first rect of the video
     */
    int frameTime;
    int dataLen;
    int encType;
    /*
     * The file-pointer in the instrudeo file where the HEADER of this rect begins (not the data)
     */
    int filePos;
    int seekBackPos;

 protected:
    ISDRectangle* previous;
    ISDRectangle* next;
};

#endif
