/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#ifndef ISDVIDEOCANVAS_H
#define ISDVIDEOCANVAS_H

/**
 * The implementation of the main video canvas that holds
 * the pixeldata and provides functions to manipulate that data.
 *
 * @author Bram Biesbrouck <b@beligum.org>
 */

#include <string>

#include <libinstrudeo/isdglwidget.h>

using namespace std;

class ISDVideoProperties;
class ISDRectangle;

class ISDVideoCanvas : public ISDGLWidget
{
 public:
    enum ISDPixelOrder {
	ISD_PIXEL_RGBA,
	ISD_PIXEL_ARGB
    };

    //-----CONSTRUCTORS-----
    /**
     * Constructs a new video canvas and allocates buffers
     * according the provided properties.
     *
     * @param videoProperties The video properties.
     */
    ISDVideoCanvas(ISDVideoProperties* videoProperties);

    ~ISDVideoCanvas();

    //-----PUBLIC FUNCTIONS-----
    /**
     * Initialises the OpenGL structures and views.
     *
     * @return Returns a code that indicates success or failure.
     */
    virtual ISDErrorCode initGL();

    /**
     * This function is useful when the OpenGL context changed,
     * re-initialises the context of the canvas and all its children.
     * Note: only the children of the canvas are updated, if you want to
     *       update all (eg.) commentboxes, you'll need to do that yourself.
     *
     * @return Returns a code that indicates success or failure.
     */
    virtual ISDErrorCode reInitGL();

    /**
     * Overloaded function of ISDGLWidget that implements the
     * OpenGL display routines for the main video.
     * It draws the current image at position (0,0).
     * The size equals the real size of the recorded video.
     * If you need resizing, you'll have to implement it yourself using
     * the glPixelZoom() function.
     */
    virtual void display();

    /**
     * Tells the canvas to update the scale to the new scale.
     * This is done for pixel-drawings as for vector-calculations.
     *
     * @param newScale The new scale.
     */
    void updateScales(float newScale);

    /**
     * Inserts the data of this rectangle in the frameBuffer, overwriting existing pixeldata.
     * According to the rects encType, the data is first decoded or not (if raw).
     *
     * @param rect The rectangle.
     * @param data The (possibly encoded) data.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode rectangleUpdate(const ISDRectangle* rect, char* data);

    /**
     * Fills the pixelBuffer with the pixels of the current frame.
     * The buffer must be allocated large enough to contain a whole frame.
     * Note: this method return the entire frame, use getPixelData() to grab
     *       a specified region.
     * 
     * @param pixelData The buffer that will be filled.
     * @param order The order of the pixel-channels that will be returned.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode getCurrentFrame(char* pixelData, ISDPixelOrder order = ISD_PIXEL_ARGB);

    /**
     * Fills the supplied data buffer with the (compressed) pixeldata,
     * belonging to the rectangle. (buffer must be allocated)
     *
     * @param rect The rect header of the data.
     * @param pixelData Pointer to the buffer where the data will be read into.
     * @param offScreenBuffer An optional off-screen buffer to read from instead of using glReadPixels
     *                        Note that the order of this buffer is used and 'order' is neglected.
     * @param order The order of the pixel-channels that will be returned.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode getPixelData(ISDRectangle* rect, char* pixelData,
			      char* offScreenBuffer = NULL, ISDPixelOrder order = ISD_PIXEL_ARGB);

    /**
     * Returns the width of the canvas.
     *
     * @return The width in pixels.
     */
    int getWidth();

    /**
     * Returns the height of the canvas.
     *
     * @return The height in pixels.
     */
    int getHeight();

 protected:
    //-----METHODS-----

    //-----VARIABLES-----
    ISDVideoProperties* videoProperties;
    char* imageBuffer;
    char* decodeBuffer;
    bool vertMirrorCanvas;
};

#endif
