/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#ifndef ISDWSCOMMUNICATOR_H
#define ISDWSCOMMUNICATOR_H

/**
 * This class acts as a central (local) proxy for all calls to
 * the Instrudeo WebService. It uses the automatically generated classes
 * in the "webservice" subfolder (gsoap stubs).
 *
 * @author Bram Biesbrouck <b@beligum.org>
 */

#include <string>
#include <list>

#include <libinstrudeo/webservice/soapInstrudeoServiceBindingProxy.h>
#include <libinstrudeo/isdobject.h>

using namespace std;

#define DEFAULT_FTP_PORT 21
#define DEFAULT_SERVICE_ENDPOINT "http://captorials.instrudeo.com/services/InstrudeoService"

class ISDWSPlatform;
class ISDWSCategory;
class ISDWSFtpTransmitter;
class ISDProgressCallback;

class ISDWSCommunicator : public ISDObject
{
 public:
    //-----CONSTRUCTORS-----
    ISDWSCommunicator(string& serviceEndpoint);
    
    ~ISDWSCommunicator();
    
    //-----METHODS-----
    /**
     * Contacts the Instrudeo WebService and retrieves all the subplatforms
     * of the specified platform and puts them in returnList. If the specified
     * platform has an ID == ISDWSPlatform::TOP_LEVEL_PARENT_ID, all top-level
     * platforms are returned.
     * Note 1: objects in the returnList must be destroyed by the user of this method.
     * 
     * @param cat The platform to search the children for.
     * @param returnList The list to fill with subplatforms.
     * @param errorString Returns the error message from the webservice if something went wrong.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode getSubPlatforms(ISDWSPlatform* cat, list<ISDWSPlatform*>& returnList, string& errorString);

    /**
     * Contacts the Instrudeo WebService and retrieves all the subcategories
     * of the specified category and puts them in returnList. If the specified
     * category has an ID == ISDWSCategory::TOP_LEVEL_PARENT_ID, all top-level
     * categories are returned.
     * Note 1: objects in the returnList must be destroyed by the user of this method.
     * 
     * @param cat The category to search the children for.
     * @param returnList The list to fill with subcategories.
     * @param errorString Returns the error message from the webservice if something went wrong.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode getSubCategories(ISDWSCategory* cat, list<ISDWSCategory*>& returnList, string& errorString);

    /**
     * Contacts the Instrudeo WebService with the specified login-info
     * and retrieves a new ftp path to upload the recording to. If the
     * login was unsuccessful, the errorString will be filled in accordingly.
     *
     * @param username The username to login to the webservice.
     * @param password The password to login to the webservice.
     * @param ftpPath Returns the retrieved path. Unspecified if login was unsuccessful.
     * @param sessionId Returns the id, returned by the webservice, indicating the upload session id.
     * @param errorString Returns the error message from the webservice if something went wrong.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode getFtpPath(string& username, string& password, string& ftpPath, int& sessionId, string& errorString);

    /**
     * Sends a recording-file to the Instrudeo Webservice FTP server.
     * The username must be the sessionId, as returned by the getFtpPath() method.
     * When the callbackClass is not NULL, the procentSent() method of
     * the object will be called regularly, updating the percentage of 
     * the bytes sent.
     *
     * @param recordingFileName The filename of the recording to send.
     * @param ftpPath The ftpPath-string, as returned by the Instrudeo webservice.
     * @param username The FTP-username (the sessionId of the getFtpPath() method).
     * @param password The FTP-password.
     * @param callbackClass Can be set to monitor the progress or NULL to discard that.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode transmitRecording(string& recordingFileName, string& ftpPath,
				   string& username, string& password,
				   ISDProgressCallback* callbackClass);
    
    /**
     * Notifies the webservice that the upload is done.
     * If the upload was aborted, set the aborted flag to true.
     *
     * @param sessionId The id of the session, as returned by the webservice in getFtpPath()
     * @param aborted Set to true if the upload was aborted.
     * @param errorString Returns the error message from the webservice if something went wrong.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode setUploadDone(int sessionId, bool aborted, string& errorString);

 protected:
    //-----METHODS-----
    /**
     * This method is used to communicate with the FTP transmitter on the number
     * of bytes sent.
     */
    static int ftpCallback(void* callbackClass, double xfered, double total);
    
    /**
     * Is called to handle the callback call.
     */
    void handleCallback(double xfered);

    /**
     * Splits the ftpPath into its components
     *
     * @param ftpPath The ftp path string to split.
     * @param host Returns the hostname or IP address.
     * @param port Returns the FTP port.
     * @param dir Returns the remote directory.
     * @param file Returns the remote file name.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode splitFtpPathString(string& ftpPath, string& host, int& port, string& dir, string& file);

    //-----VARIABLES-----
    InstrudeoServiceBinding* wsStub;
    static ISDProgressCallback* callbackClass;
    off64_t transmitFileSize;
};

#endif
