/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#ifndef ISDWSFTPTRANSMITTER_H
#define ISDWSFTPTRANSMITTER_H

/**
 * Sends a recording to the Instrudeo Webservice FTP server,
 * using the libftppp and libftp library.
 *
 * @author Bram Biesbrouck <b@beligum.org>
 */

#include <string>

#include <curl/curl.h>

#include <libinstrudeo/isdobject.h>

using namespace std;

/**
 * Defines the number of bytes between successive callback-function calls
 * that reports the number of sent bytes.
 * Currently set to 10 KB.
 */
#define NUM_BYTES_CALLBACK_OFFSET 1024*10

typedef int (*FtpCallbackFunc)(void *arg, double xfered, double total);

class ISDWSCommunicator;

class ISDWSFtpTransmitter : public ISDObject
{
 public:
    //-----CONSTRUCTORS-----
    ISDWSFtpTransmitter();

    ~ISDWSFtpTransmitter();

    //-----METHODS-----
    /**
     * Send the specified recording to server:port.
     * The filename and directory specify the remote path.
     * The recording must be packed and saved before calling this.
     *
     * @param recFile The filename of the recording to send.
     * @param server The server hostname or IP address (without "ftp://")
     * @param port The ftp-port (usually 21).
     * @param directory The remote directory (created when necessary)
     * @param filename The remote filename.
     * @param username The FTP-username.
     * @param password The FTP-password.
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode sendRecording(string& recFile, string& server, int port,
			       string& directory, string& filename,
			       string& username, string& password);

    /**
     * Sets the function to call after every NUM_BYTES_CALLBACK_OFFSET
     * bytes sent.
     *
     * @param callbackClass The class where the function funcPtr is a member of.
     * @param funcPtr The function that has this signature: ftpCallback(double xfered, double total);
     * @return Returns a code that indicates success or failure.
     */
    ISDErrorCode setSentBytesCallback(ISDWSCommunicator* callbackClass, FtpCallbackFunc funcPtr);

 protected:
    //-----METHODS-----
    /**
     * Callback for this class that will delegate to the provided callback function
     */
    static int progressCallback(void *clientp, double dltotal, double dlnow, double ultotal, double ulnow);

    /**
     * Closes the connection if necessary and clean up.
     */
    void cleanup();

    //-----VARIABLES-----
    CURL* ftpHandle;
    static void* callbackClass;
    static FtpCallbackFunc callbackPtr;
};

#endif
