/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
//! @file inti/atk/value.h
//! @brief An AtkValue C++ wrapper interface.
//!
//! Provides AtkValue, an interface implemented by valuators and components which
//! display or select a value from a bounded range of values.

#ifndef INTI_ATK_VALUE_H
#define INTI_ATK_VALUE_H

#ifndef INTI_G_TYPE_H
#include <inti/glib/type.h>
#endif

#ifndef __ATK_VALUE_H__
#include <atk/atkvalue.h>
#endif

namespace Inti {

namespace G {
class Value;
}

namespace Atk {

class ValueIface;

//! @class Value value.h inti/atk/value.h
//! @brief An AtkValue C++ wrapper class.
//!
//! Value should be implemented for components which either display a value from a bounded range,
//! or which allow the user to specify a value from a bounded range, or both. For instance, most
//! sliders and range controls, as well as dials, should have Atk::Object representations which
//! implement Value on the component's behalf. Values may be read-only, in which case attempts
//! to alter the value return false to indicate failure.

class Value : public virtual G::TypeInterface
{
	friend class ValueIface;

protected:
//! @name Constructors
//! @{
	virtual ~Value() = 0;
	
//! @}
//  Override these do_ methods when you want to change the default behaviour of the AtkValue.
	
	virtual void do_get_current_value(GValue *value);

	virtual void do_get_maximum_value(GValue *value);

	virtual void do_get_minimum_value(GValue *value);

	virtual bool do_set_current_value(const GValue *value);

public:
//! @name Accessors
//! @{

	AtkValue* atk_value() const { return (AtkValue*)instance; }
	//!< Get a pointer to the AtkValue structure.

	AtkValueIface* atk_value_iface() const;
	//!< Get a pointer to the AtkValueIface structure.

	operator AtkValue* () const;
	//!< Conversion operator; safely converts a Value to an AtkValue pointer.

	void get_current_value(G::Value& value) const;
	//!< Gets the value of this object.
	//!< @param value A G::Value representing the current accessible value.

	void get_maximum_value(G::Value& value) const;
	//!< Gets the maximum value for this object.
	//!< @param value A G::Value representing the maximum accessible value.

	void get_minimum_value(G::Value& value) const;
	//!< Gets the minimum value for this object.
	//!< @param value A G::Value representing the minimum accessible value.

//! @}
//! @name Methods
//! @{

	bool set_current_value(const G::Value& value);
	//!< Sets the value of this object.
	//!< @param value A G::Value which is the desired new accessible value.
	//!< @return <EM>true</EM> if the new value is successfully set, <EM>false</EM> otherwise.
	
//! @}
};

} // namespace Atk

} // namespace Inti

#endif // INTI_ATK_VALUE_H

