/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2003 The Inti Development Team.
 *
 *  date.cc - A GDate C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "date.h"
#include <glib/gmem.h>

using namespace Inti;

/*  G::Date
 */

G::Date::Date()
: date_(g_date_new())
{
}

G::Date::Date(DateDay day, DateMonth month, DateYear year)
: date_(g_date_new_dmy(day, (GDateMonth)month, year))
{
}

G::Date::Date(unsigned int julian_day)
: date_(g_date_new_julian(julian_day))
{
}

G::Date::Date(const GDate& date)
: date_(g_date_new())
{
	*date_ = date;
}

G::Date::Date(const Date& src)
: date_(g_date_new())
{
	*date_ = *src.date_;
}

G::Date::~Date()
{
	if (date_)
	{
		g_date_free(date_);
		date_ = 0;
	}
}

G::Date&
G::Date::operator=(const Date& src)
{
	*date_ = *src.date_;
	return *this;
}

G::Date::operator GDate* () const
{
	return this ? date_ : 0;
}

bool
G::Date::valid() const
{
	return g_date_valid(g_date());
}

G::DateWeekday
G::Date::get_weekday() const
{
	return (DateWeekday)g_date_get_weekday(g_date());
}

G::DateDay
G::Date::get_day() const
{
	return g_date_get_day(g_date());
}

G::DateMonth
G::Date::get_month() const
{
	return (DateMonth)g_date_get_month(g_date());
}

G::DateYear
G::Date::get_year() const
{
	return g_date_get_year(g_date());
}

unsigned int
G::Date::get_julian() const
{
	return g_date_get_julian(g_date());
}

unsigned int
G::Date::get_day_of_year() const
{
	return g_date_get_day_of_year(g_date());
}

unsigned int
G::Date::get_monday_week_of_year() const
{
	return g_date_get_monday_week_of_year(g_date());
}

unsigned int
G::Date::get_sunday_week_of_year() const
{
	return g_date_get_sunday_week_of_year(g_date());
}

bool
G::Date::is_first_of_month() const
{
	return g_date_is_first_of_month(g_date());
}

bool
G::Date::is_last_of_month() const
{
	return g_date_is_last_of_month(g_date());
}

int
G::Date::compare(const Date& other) const
{
	return g_date_compare(g_date(), other.g_date());
}

bool
G::Date::operator==(const Date& other) const
{
	return compare(other) == 0;
}

bool
G::Date::operator!=(const Date& other) const
{
	return compare(other) != 0;
}

bool
G::Date::operator<(const Date& other) const
{
	return compare(other) < 0;
}

bool
G::Date::operator<=(const Date& other) const
{
	return compare(other) <= 0;
}

bool
G::Date::operator>(const Date& other) const
{
	return compare(other) > 0;
}

bool
G::Date::operator>=(const Date& other) const
{
	return compare(other) >= 0;
}

bool
G::Date::valid_day(DateDay day)
{
	return g_date_valid_day(day);
}

bool
G::Date::valid_month(DateMonth month)
{
	return g_date_valid_month((GDateMonth)month);
}

bool
G::Date::valid_year(DateYear year)
{
	return g_date_valid_year(year);
}

bool
G::Date::valid_weekday(DateWeekday weekday)
{
	return g_date_valid_weekday((GDateWeekday)weekday);
}

bool
G::Date::valid_julian(unsigned int julian_date)
{
	return g_date_valid_julian(julian_date);
}

bool
G::Date::valid_dmy(DateDay day, DateMonth month, DateYear year)
{
	return g_date_valid_dmy(day, (GDateMonth)month, year);
}

bool
G::Date::is_leap_year(DateYear year)
{
	return g_date_is_leap_year(year);
}

unsigned char
G::Date::get_days_in_month(DateMonth month, DateYear year)
{
	return g_date_get_days_in_month((GDateMonth)month, year);
}

unsigned char
G::Date::get_monday_weeks_in_year(DateYear year)
{
	return g_date_get_monday_weeks_in_year(year);
}

unsigned char
G::Date::get_sunday_weeks_in_year(DateYear year)
{
	return g_date_get_sunday_weeks_in_year(year);
}

void
G::Date::clear()
{
	g_date_clear(g_date(), 1);
}

void
G::Date::set_parse(const String& str)
{
	g_date_set_parse(g_date(), str.c_str());
}

void
G::Date::set_time(Time time)
{
	g_date_set_time(g_date(), time);
}

void
G::Date::set_day(DateDay day)
{
	g_date_set_day(g_date(), day);
}

void
G::Date::set_month(DateMonth month)
{
	g_date_set_month(g_date(), (GDateMonth)month);
}

void
G::Date::set_year(DateYear year)
{
	g_date_set_year(g_date(), year);
}

void
G::Date::set_dmy(DateDay day, DateMonth month, DateYear year)
{
	g_date_set_dmy(g_date(), day, (GDateMonth)month, year);
}

void
G::Date::set_julian(unsigned int julian_date)
{
	g_date_set_julian(g_date(), julian_date);
}

void
G::Date::add_days(unsigned int n_days)
{
	g_date_add_days(g_date(), n_days);
}

void
G::Date::subtract_days(unsigned int n_days)
{
	g_date_subtract_days(g_date(), n_days);
}

void
G::Date::add_months(unsigned int n_months)
{
	g_date_add_months(g_date(), n_months);
}

void
G::Date::subtract_months(unsigned int n_months)
{
	g_date_subtract_months(g_date(), n_months);
}

void
G::Date::add_years(unsigned int n_years)
{
	g_date_add_years(g_date(), n_years);
}

void
G::Date::subtract_years(unsigned int n_years)
{
	g_date_subtract_years(g_date(), n_years);
}

int
G::Date::days_between(const Date& date1, const Date& date2)
{
	return g_date_days_between(date1.g_date(), date2.g_date());
}

void
G::Date::to_struct_tm(struct tm *tm)
{
	g_date_to_struct_tm(g_date(), tm);
}

void
G::Date::clamp(const Date *min_date, const Date *max_date)
{
	g_date_clamp(g_date(), *min_date, *max_date);
}

void
G::Date::order(Date& other)
{
	g_date_order(g_date(), other.g_date());
}

String
G::Date::format(const String& date_format)
{
	String s;
	gsize result = 0;
	int buffersize = 64;

	do
	{
		char *buffer = static_cast<char*>(g_malloc(buffersize));
		result = g_date_strftime(buffer, buffersize, date_format.c_str(), g_date());
		if (result)
			s = buffer;
		g_free(buffer);
	}
	while (!result && (buffersize *= 2));

	return s;
}

