/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *
 *  scanner.cc - GScanner C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "scanner.h"
#include <glib/gmessages.h>
#include <glib/gstrfuncs.h>
#include <glib/gmem.h>

using namespace Inti;

G::Scanner::Scanner(const GScannerConfig *initial_settings)
: scanner_(g_scanner_new(initial_settings))
{
}	

G::Scanner::Scanner(GScanner *scanner)
: scanner_(scanner)
{
}

G::Scanner::~Scanner()
{
	if (scanner_)
	{
		g_scanner_destroy(scanner_);
		scanner_ = 0;
	}
}

G::Scanner::operator GScanner* () const
{
	return this ? scanner_ : 0;
}

GTokenType
G::Scanner::token() const
{
	return g_scanner_cur_token(scanner_);
}

GTokenValue
G::Scanner::value() const
{
	return g_scanner_cur_value(scanner_);
}

unsigned int
G::Scanner::line() const
{
	return g_scanner_cur_line(scanner_);
}

unsigned int
G::Scanner::position() const
{
	return g_scanner_cur_position(scanner_);
}

bool
G::Scanner::eof() const
{
	return g_scanner_eof(scanner_) != 0;
}

GTokenType
G::Scanner::next_token() const
{
	return scanner_->next_token;
}

GTokenValue
G::Scanner::next_value() const
{
	return scanner_->next_value;
}

unsigned int
G::Scanner::next_line() const
{
	return scanner_->next_line;
}

unsigned int
G::Scanner::next_position() const
{
	return scanner_->next_position;
}

GTokenType
G::Scanner::get_next_token() const
{
	return g_scanner_get_next_token(scanner_);
}

GTokenType
G::Scanner::peek_next_token() const
{
	return g_scanner_peek_next_token(scanner_);
}

void
G::Scanner::input_file(int input_fd)
{
	g_scanner_input_file(scanner_, input_fd);
}

void
G::Scanner::sync_file_offset()
{
	g_scanner_sync_file_offset(scanner_);
}

void
G::Scanner::input_text(const char *text, unsigned int length)
{
	g_scanner_input_text(scanner_, text, length);
}

void
G::Scanner::input_text(const String& text)
{
	g_scanner_input_text(scanner_, text.c_str(), text.size());
}

unsigned int
G::Scanner::set_scope(unsigned int scope_id)
{
	return g_scanner_set_scope(scanner_, scope_id);
}

void
G::Scanner::scope_add_symbol(unsigned int scope_id, const char *symbol, void *value)
{
	g_scanner_scope_add_symbol(scanner_, scope_id, symbol, value);
}

void
G::Scanner::scope_remove_symbol(unsigned int scope_id, const char *symbol)
{
	g_scanner_scope_remove_symbol(scanner_, scope_id, symbol);
}

void*
G::Scanner::scope_lookup_symbol(unsigned int scope_id, const char *symbol)
{
	return g_scanner_scope_lookup_symbol(scanner_, scope_id, symbol);
}

namespace {

void scope_foreach_symbol_slot(gpointer key, gpointer value, gpointer user_data)
{
	G::Scanner::ScopeForeachSymbolSlot *slot = static_cast<G::Scanner::ScopeForeachSymbolSlot*>(user_data);
	slot->call(key, value);
}

} // namespace

void
G::Scanner::scope_foreach_symbol(unsigned int scope_id, const ScopeForeachSymbolSlot *slot)
{
	g_scanner_scope_foreach_symbol(scanner_, scope_id, &scope_foreach_symbol_slot, (void*)slot);
}

void*
G::Scanner::lookup_symbol(const char *symbol)
{
	return g_scanner_lookup_symbol(scanner_, symbol);
}

void
G::Scanner::error(const char *format, ...)
{
	g_return_if_fail(format != 0);

	scanner_->parse_errors++;

	if (scanner_->msg_handler)
	{
		va_list args;
		va_start(args, format);
		char * msg = g_strdup_vprintf(format, args);
		va_end(args);
		scanner_->msg_handler(scanner_, msg, TRUE);
		g_free(msg);
	}
}

void
G::Scanner::warn(const char *format, ...)
{
	g_return_if_fail(format != 0);

	scanner_->parse_errors++;

	if (scanner_->msg_handler)
	{
		va_list args;
		va_start(args, format);
		char * msg = g_strdup_vprintf(format, args);
		va_end(args);
		scanner_->msg_handler(scanner_, msg, FALSE);
		g_free(msg);
	}
}

