/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *  Copyright (C) 2000 Red Hat, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
//! @file inti/gtk/celleditable.h
//! @brief A GtkCellEditable C++ wrapper interface.

#ifndef INTI_GTK_CELL_EDITABLE_H
#define INTI_GTK_CELL_EDITABLE_H

#ifndef INTI_G_TYPE_H
#include <inti/glib/type.h>
#endif

#ifndef __GTK_CELL_EDITABLE_H__
#include <gtk/gtkcelleditable.h>
#endif

namespace Inti {

namespace Gdk {
class Event;
}

namespace Gtk {

class CellEditableIface;

//! @class CellEditable celleditable.h inti/gtk/celleditable.h
//! @brief A GtkCellEditable C++ wrapper class.
//!
//! CellEditable is an abstract interface that Entry implements through
//! multiple inheritance. CellRenderer returns a smart pointer to a 
//! CellEditable when you call CellRenderer::start_editing(). Using this
//! pointer, you can monitor the editing process by connecting to either
//! its editing_done_signal or remove_widget_signal.

class CellEditable : public virtual G::TypeInterface
{
	friend class CellEditableIface;

protected:
//! @name Constructors
//! @{

	virtual ~CellEditable() = 0;
	//!< Destructor.
	
//! @}
//  Override this do_ method when you want to change the default behaviour of the GtkCellEditable.

	virtual void do_start_editing(GdkEvent *event);

//! @name Signal Handlers
//! @{

	virtual void on_editing_done();
	//!< Called when the cell renderer should update it's value from the cell.

	virtual void on_remove_widget();
	//!< Called when the cell is finished editing, and the editable widget may now be destroyed.

//! @}
//  Signals

	typedef G::Signal0<void> EditingDoneSignalType;
	typedef G::SignalProxy<TypeInstance, EditingDoneSignalType> EditingDoneSignalProxy;
	static const EditingDoneSignalType editing_done_signal;

	typedef G::Signal0<void> RemoveWidgetSignalType;
	typedef G::SignalProxy<TypeInstance, RemoveWidgetSignalType> RemoveWidgetSignalProxy;
	static const RemoveWidgetSignalType remove_widget_signal;

public:
//! @name Accessors
//! @{

	GtkCellEditable* gtk_cell_editable() const { return (GtkCellEditable*)instance; }
	//!< Get a pointer to the GtkCellEditable structure.

	GtkCellEditableIface* gtk_cell_editable_iface() const;
	//!< Get a pointer to the GtkCellEditableIface structure.

	operator GtkCellEditable* () const;
	//!< Conversion operator; safely converts a Calendar to a GtkCalendar pointer.

//! @}
//! @name Methods
//! @{

	void start_editing(const Gdk::Event *event);
	//!< Begins editing on a cell_editable. 
	//!< @param event A Gdk::Event, or null.
	//!<
	//!< <BR>The <EM>event</EM> is the Gdk::Event that began the editing process. It may be null,
	//!< in the instance that editing was initiated through programatic means.

	void editing_done();
	//!< Emits the "editing_done" signal.
	//!< This signal is a sign for the cell renderer to update it's value from the cell.

	void remove_widget();
	//!< Emits the "remove_widget" signal. 
	//!< This signal is meant to indicate that the cell is finished editing, and the
	//!< editable widget may now be destroyed.
	
//! @}
//! @name Signal Proxies
//! @{

	const EditingDoneSignalProxy sig_editing_done()
	{
		return EditingDoneSignalProxy(this, &editing_done_signal);
	}
	//!< Connect to the editing_done_signal; emitted when the cell renderer should
	//! update it's value from the cell.

	const RemoveWidgetSignalProxy sig_remove_widget()
	{
		return RemoveWidgetSignalProxy(this, &remove_widget_signal);
	}
	//!< Connect to the remove_widget_signal; emitted when the cell is finished editing,
	//!< and the editable widget may now be destroyed.

//! @}
};

} // namespace Gtk

} // namespace Inti

#endif // INTI_GTK_CELL_EDITABLE_H

