/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  editable.cc - GtkEditable C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "editable.h"
#include "private/editable_p.h"
#include "../glib/object.h"

using namespace Inti;

/*  Gtk::Editable
 */

Gtk::Editable::~Editable()
{
}
	
GtkEditableClass*
Gtk::Editable::gtk_editable_class() const
{
	return peek<GtkEditableClass>(GTK_TYPE_EDITABLE); 
}
	
Gtk::Editable::operator GtkEditable* () const 
{ 
	return this ? gtk_editable() : 0; 
}
	
bool
Gtk::Editable::get_selection_bounds(int *start, int *end) const
{
	return gtk_editable_get_selection_bounds(gtk_editable(), start, end);
}

String
Gtk::Editable::get_chars(int start_pos, int end_pos) const
{
	return gtk_editable_get_chars(gtk_editable(), start_pos, end_pos);
}

int
Gtk::Editable::get_position() const
{
	return gtk_editable_get_position(gtk_editable());
}

bool 
Gtk::Editable::get_editable() const
{
	return gtk_editable_get_editable(gtk_editable());
}

void 
Gtk::Editable::select_region (int start, int end)
{
	gtk_editable_select_region(gtk_editable(), start, end);
}

void 
Gtk::Editable::insert_text(const String& new_text, int& position)
{
	gtk_editable_insert_text(gtk_editable(), new_text.c_str(), new_text.size(), &position);
}

void 
Gtk::Editable::delete_text(int start_pos, int end_pos)
{
	gtk_editable_delete_text(gtk_editable(), start_pos, end_pos);
}

void 
Gtk::Editable::cut_clipboard()
{
	gtk_editable_cut_clipboard(gtk_editable());
}

void 
Gtk::Editable::copy_clipboard()
{
	gtk_editable_copy_clipboard(gtk_editable());
}

void 
Gtk::Editable::paste_clipboard()
{
	gtk_editable_paste_clipboard(gtk_editable());
}

void 
Gtk::Editable::delete_selection()
{
	gtk_editable_delete_selection(gtk_editable());
}

void 
Gtk::Editable::set_position(int position)
{
	gtk_editable_set_position(gtk_editable(), position);
}

void
Gtk::Editable::set_editable(bool is_editable)
{
	gtk_editable_set_editable(gtk_editable(), is_editable);
}

/*  Gtk::EditableClass
 */

void
Gtk::EditableClass::init(GtkEditableClass *g_iface)
{
	g_iface->insert_text = &insert_text_proxy;
	g_iface->delete_text = &delete_text_proxy;
	g_iface->changed = &changed_proxy;
	g_iface->do_insert_text = &do_insert_text_proxy;
	g_iface->do_delete_text = &do_delete_text_proxy;
	g_iface->get_chars = &get_chars_proxy;
	g_iface->set_selection_bounds = &set_selection_bounds_proxy;
	g_iface->get_selection_bounds = &get_selection_bounds_proxy;
	g_iface->set_position = &set_position_proxy;
	g_iface->get_position = &get_position_proxy;
}

void
Gtk::EditableClass::do_insert_text_proxy(GtkEditable *editable, const gchar *text, gint length, gint *position)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(editable));
	Gtk::Editable *tmp_editable = dynamic_cast<Gtk::Editable*>(object);
	if (tmp_editable)
		tmp_editable->do_insert_text(text, length, position);
	else
	{
		GtkEditableClass *g_iface = static_cast<GtkEditableClass*>(g_type_interface_peek_parent(GTK_EDITABLE_GET_CLASS(editable)));
		if (g_iface->do_insert_text)
			g_iface->do_insert_text(editable, text, length, position);
	}
}

void
Gtk::EditableClass::do_delete_text_proxy(GtkEditable *editable, gint start_pos, gint end_pos)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(editable));
	Gtk::Editable *tmp_editable = dynamic_cast<Gtk::Editable*>(object);
	if (tmp_editable)
		tmp_editable->do_delete_text(start_pos, end_pos);
	else
	{
		GtkEditableClass *g_iface = static_cast<GtkEditableClass*>(g_type_interface_peek_parent(GTK_EDITABLE_GET_CLASS(editable)));
		if (g_iface->do_delete_text)
			g_iface->do_delete_text(editable, start_pos, end_pos);
	}
}

gchar*
Gtk::EditableClass::get_chars_proxy(GtkEditable *editable, gint start_pos, gint end_pos)
{
	gchar *result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(editable));
	Gtk::Editable *tmp_editable = dynamic_cast<Gtk::Editable*>(object);
	if (tmp_editable)
		result = tmp_editable->do_get_chars(start_pos, end_pos);
	else
	{
		GtkEditableClass *g_iface = static_cast<GtkEditableClass*>(g_type_interface_peek_parent(GTK_EDITABLE_GET_CLASS(editable)));
		if (g_iface->get_chars)
			result = g_iface->get_chars(editable, start_pos, end_pos);
	}
	return result;
}

void
Gtk::EditableClass::set_selection_bounds_proxy(GtkEditable *editable, gint start_pos, gint end_pos)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(editable));
	Gtk::Editable *tmp_editable = dynamic_cast<Gtk::Editable*>(object);
	if (tmp_editable)
		tmp_editable->do_set_selection_bounds(start_pos, end_pos);
	else
	{
		GtkEditableClass *g_iface = static_cast<GtkEditableClass*>(g_type_interface_peek_parent(GTK_EDITABLE_GET_CLASS(editable)));
		if (g_iface->set_selection_bounds)
			g_iface->set_selection_bounds(editable, start_pos, end_pos);
	}
}

gboolean
Gtk::EditableClass::get_selection_bounds_proxy(GtkEditable *editable, gint *start_pos, gint *end_pos)
{
	gboolean result = FALSE;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(editable));
	Gtk::Editable *tmp_editable = dynamic_cast<Gtk::Editable*>(object);
	if (tmp_editable)
		result = tmp_editable->do_get_selection_bounds(start_pos, end_pos);
	else
	{
		GtkEditableClass *g_iface = static_cast<GtkEditableClass*>(g_type_interface_peek_parent(GTK_EDITABLE_GET_CLASS(editable)));
		if (g_iface->get_selection_bounds)
			result = g_iface->get_selection_bounds(editable, start_pos, end_pos);
	}
	return result;
}

void
Gtk::EditableClass::set_position_proxy(GtkEditable *editable, gint position)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(editable));
	Gtk::Editable *tmp_editable = dynamic_cast<Gtk::Editable*>(object);
	if (tmp_editable)
		tmp_editable->do_set_position(position);
	else
	{
		GtkEditableClass *g_iface = static_cast<GtkEditableClass*>(g_type_interface_peek_parent(GTK_EDITABLE_GET_CLASS(editable)));
		if (g_iface->set_position)
			g_iface->set_position(editable, position);
	}
}

gint
Gtk::EditableClass::get_position_proxy(GtkEditable *editable)
{
	gint result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(editable));
	Gtk::Editable *tmp_editable = dynamic_cast<Gtk::Editable*>(object);
	if (tmp_editable)
		result = tmp_editable->do_get_position();
	else
	{
		GtkEditableClass *g_iface = static_cast<GtkEditableClass*>(g_type_interface_peek_parent(GTK_EDITABLE_GET_CLASS(editable)));
		if (g_iface->get_position)
			result = g_iface->get_position(editable);
	}
	return result;
}

void
Gtk::EditableClass::insert_text_proxy(GtkEditable *editable, const gchar *text, gint length, gint *position)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(editable));
	Gtk::Editable *tmp_editable = dynamic_cast<Gtk::Editable*>(object);
	if (tmp_editable)
	{
		std::string tmp_text(text, length);
		tmp_editable->on_insert_text(tmp_text, position);
	}
	else
	{
		GtkEditableClass *g_iface = static_cast<GtkEditableClass*>(g_type_interface_peek_parent(GTK_EDITABLE_GET_CLASS(editable)));
		if (g_iface->insert_text)
			g_iface->insert_text(editable, text, length, position);
	}
}

void
Gtk::EditableClass::delete_text_proxy(GtkEditable *editable, gint start_pos, gint end_pos)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(editable));
	Gtk::Editable *tmp_editable = dynamic_cast<Gtk::Editable*>(object);
	if (tmp_editable)
		tmp_editable->on_delete_text(start_pos, end_pos);
	else
	{
		GtkEditableClass *g_iface = static_cast<GtkEditableClass*>(g_type_interface_peek_parent(GTK_EDITABLE_GET_CLASS(editable)));
		if (g_iface->delete_text)
			g_iface->delete_text(editable, start_pos, end_pos);
	}
}

void
Gtk::EditableClass::changed_proxy(GtkEditable *editable)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(editable));
	Gtk::Editable *tmp_editable = dynamic_cast<Gtk::Editable*>(object);
	if (tmp_editable)
		tmp_editable->on_changed();
	else
	{
		GtkEditableClass *g_iface = static_cast<GtkEditableClass*>(g_type_interface_peek_parent(GTK_EDITABLE_GET_CLASS(editable)));
		if (g_iface->changed)
			g_iface->changed(editable);
	}
}

/*  Overridable GtkEditable methods
 */

void
Gtk::Editable::do_insert_text(const char *text, int length, int *position)
{
	GtkEditableClass *g_iface = peek_parent<GtkEditableClass>(gtk_editable_class());
	if (g_iface->do_insert_text)
		g_iface->do_insert_text(gtk_editable(), text, length, position);
}

void
Gtk::Editable::do_delete_text(int start_pos, int end_pos)
{
	GtkEditableClass *g_iface = peek_parent<GtkEditableClass>(gtk_editable_class());
	if (g_iface->do_delete_text)
		g_iface->do_delete_text(gtk_editable(), start_pos, end_pos);
}

char*
Gtk::Editable::do_get_chars(int start_pos, int end_pos)
{
	char *result = 0;
	GtkEditableClass *g_iface = peek_parent<GtkEditableClass>(gtk_editable_class());
	if (g_iface->get_chars)
		result = g_iface->get_chars(gtk_editable(), start_pos, end_pos);
	return result;
}

void
Gtk::Editable::do_set_selection_bounds(int start_pos, int end_pos)
{
	GtkEditableClass *g_iface = peek_parent<GtkEditableClass>(gtk_editable_class());
	if (g_iface->set_selection_bounds)
		g_iface->set_selection_bounds(gtk_editable(), start_pos, end_pos);
}

bool
Gtk::Editable::do_get_selection_bounds(int *start_pos, int *end_pos)
{
	bool result = false;
	GtkEditableClass *g_iface = peek_parent<GtkEditableClass>(gtk_editable_class());
	if (g_iface->get_selection_bounds)
		result = g_iface->get_selection_bounds(gtk_editable(), start_pos, end_pos);
	return result;
}

void
Gtk::Editable::do_set_position(int position)
{
	GtkEditableClass *g_iface = peek_parent<GtkEditableClass>(gtk_editable_class());
	if (g_iface->set_position)
		g_iface->set_position(gtk_editable(), position);
}

int
Gtk::Editable::do_get_position()
{
	int result = 0;
	GtkEditableClass *g_iface = peek_parent<GtkEditableClass>(gtk_editable_class());
	if (g_iface->get_position)
		result = g_iface->get_position(gtk_editable());
	return result;
}

/*  Signal handlers
 */

void
Gtk::Editable::on_insert_text(const String& text, int *position)
{
	GtkEditableClass *g_iface = peek_parent<GtkEditableClass>(gtk_editable_class());
	if (g_iface->insert_text)
		g_iface->insert_text(gtk_editable(), text.c_str(), text.size(), position);
}

void
Gtk::Editable::on_delete_text(int start_pos, int end_pos)
{
	GtkEditableClass *g_iface = peek_parent<GtkEditableClass>(gtk_editable_class());
	if (g_iface->delete_text)
		g_iface->delete_text(gtk_editable(), start_pos, end_pos);
}

void
Gtk::Editable::on_changed()
{
	GtkEditableClass *g_iface = peek_parent<GtkEditableClass>(gtk_editable_class());
	if (g_iface->changed)
		g_iface->changed(gtk_editable());
}

/*  Signals
 */

const Gtk::Editable::InsertTextSignalType Gtk::Editable::insert_text_signal("insert_text");

const Gtk::Editable::DeleteTextSignalType Gtk::Editable::delete_text_signal("delete_text");

const Gtk::Editable::ChangedSignalType Gtk::Editable::changed_signal("changed");

