/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *  Copyright (C) 2000 Red Hat, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
//! @file inti/gtk/item.h
//! @brief A GtkItem C++ wrapper interface.
//!
//! Provides Item, a widget which is an abstract base class for MenuItem.

#ifndef INTI_GTK_ITEM_H
#define INTI_GTK_ITEM_H

#ifndef INTI_GTK_BIN_H
#include <inti/gtk/bin.h>
#endif

#ifndef __GTK_ITEM_H__
#include <gtk/gtkitem.h>
#endif

namespace Inti {

namespace Gtk {

class ItemClass;

//! @class Item item.h inti/gtk/item.h
//! @brief A GtkItem C++ wrapper class.
//!
//! The Item widget is an abstract base class for MenuItem.

class Item : public Bin
{
	friend class G::Object;
	friend class ItemClass;

	Item(const Item&);
	Item& operator=(const Item&);
	
protected:
//! @name Constructors
//! @{

	Item();
	//!< Construct a new Item.

	explicit Item(GtkItem *item, bool reference = false);
	//!< Construct a new Item from an existing GtkItem.
	//!< @param item A pointer to a GtkItem.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>item</EM> can be a newly created GtkItem or an existing
	//!< GtkItem (see G::Object::Object).

//! @}
//! @name Signal Handlers
//! @{

	virtual void on_select();
	//!< Called when the item is selected.

	virtual void on_deselect();
	//!< Called when the item is deselected.

	virtual void on_toggle();
	//!< Called when the item is toggled.

//! @}
//  Signals

	typedef G::Signal0<void> SelectSignalType;
	typedef G::SignalProxy<TypeInstance, SelectSignalType> SelectSignalProxy;
	static const SelectSignalType select_signal;

	typedef G::Signal0<void> DeselectSignalType;
	typedef G::SignalProxy<TypeInstance, DeselectSignalType> DeselectSignalProxy;
	static const DeselectSignalType deselect_signal;

	typedef G::Signal0<void> ToggleSignalType;
	typedef G::SignalProxy<TypeInstance, ToggleSignalType> ToggleSignalProxy;
	static const ToggleSignalType toggle_signal;

public:
//! @name Constructors
//! @{
	virtual ~Item();
	//!< Destructor.
	
//! @}
//! @name Accessors
//! @{

	GtkItem* gtk_item() const { return (GtkItem*)instance; }
	//!< Get a pointer to the GtkItem structure.

	GtkItemClass* gtk_item_class() const;
	//!< Get a pointer to the GtkItemClass structure.

	operator GtkItem* () const;
	//!< Conversion operator; safely converts an Item to a GtkItem pointer.

//! @}
//! @name Methods
//! @{

	void select();
	//!< Emits the "select" signal on Item.

	void deselect();
	//!< Emits the "deselect" signal on Item.

	void toggle();
	//!< Emits the "toggle" signal on Item.

//! @}
//! @name Signal Proxies
//! @{
	
	const SelectSignalProxy sig_select()
	{
		return SelectSignalProxy(this, &select_signal);
	}
	//!< Connect to the select_signal; emitted when the item is selected.


	const DeselectSignalProxy sig_deselect()
	{
		return DeselectSignalProxy(this, &deselect_signal);
	}
	//!< Connect to the deselect_signal; emitted when the item is deselected.

	const ToggleSignalProxy sig_toggle()
	{
		return ToggleSignalProxy(this, &toggle_signal);
	}
	//!< Connect to the toggle_signal; emitted when the item is toggled.

//! @}
};

} // namespace Gtk

} // namespace Inti

#endif // INTI_GTK_ITEM_H

