/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *  Copyright (C) 2000 Red Hat, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
//! @file inti/gtk/optionmenu.h
//! @brief A GtkOptionMenu C++ wrapper interface.
//!

#ifndef INTI_GTK_OPTION_MENU_H
#define INTI_GTK_OPTION_MENU_H

#ifndef INTI_GTK_BUTTON_H
#include <inti/gtk/button.h>
#endif

#ifndef __GTK_OPTION_MENU_H__
#include <gtk/gtkoptionmenu.h>
#endif

namespace Inti {

namespace Gtk {

class Menu;
class MenuItem;
class OptionMenuClass;

//! @class OptionMenu optionmenu.h inti/gtk/optionmenu.h
//! @brief A GtkOptionMenu C++ wrapper class.
//!
//! OptionMenu is a widget that allows the user to choose from a list of valid choices.
//! The OptionMenu displays the selected choice. When activated the OptionMenu displays
//! a popup Menu which allows the user to make a new choice.
//!
//! Using a OptionMenu is simple; construct a Menu, then append menu items to it with
//! Gtk::MenuShell::append(). Set that menu on the option menu with set_menu(). Set the
//! selected menu item with set_history(); connect to the "changed" signal on the option
//! menu; in the "changed" signal, check the new selected menu item with get_history().

class OptionMenu : public Button
{
	friend class G::Object;
	friend class OptionMenuClass;

	OptionMenu(const OptionMenu&);
	OptionMenu& operator=(const OptionMenu&);
	
protected:
//! @name Constructors
//! @{

	explicit OptionMenu(GtkOptionMenu *option_menu, bool reference = false);
	//!< Construct a new OptionMenu from an existing GtkOptionMenu.
	//!< @param option_menu A pointer to a GtkOptionMenu.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>option_menu</EM> can be a newly created GtkOptionMenu or an existing
	//!< GtkOptionMenu (see G::Object::Object).

//! @}
//! @name Signal Handlers
//! @{

	virtual void on_changed();
	//!< Called when the selected menu item changes. Call get_history() to get the
	//!< index of the selected menu item.

//! @}
//  Properties

	typedef G::Property<Menu*>  MenuPropertyType;
	typedef G::PropertyProxy<G::Object, MenuPropertyType> MenuPropertyProxy;
	static const MenuPropertyType menu_property;

//  Signals

	typedef G::Signal0<void> ChangedSignalType;
	typedef G::SignalProxy<TypeInstance, ChangedSignalType> ChangedSignalProxy;
	static const ChangedSignalType changed_signal;

public:
//! @name Constructors
//! @{

	OptionMenu();
	//!< Construct a new option menu.

	explicit OptionMenu(Menu& menu, unsigned int index = 0);
	//!< Construct a new option menu with a menu that is popped up to allow the user to choose a new value.
	//!< @param menu The Menu to associate with the option menu.
	//!< @param index The index of the menu item to select; index values are from 0 to n-1.

	virtual ~OptionMenu();
	//!< Destructor
	
//! @}
//! @name Accessors
//! @{

	GtkOptionMenu* gtk_option_menu() const { return (GtkOptionMenu*)instance; }
	//!< Get a pointer to the GtkOptionMenu structure.

	GtkOptionMenuClass* gtk_option_menu_class() const;
	//!< Get a pointer to the GtkOptionMenuClass structure.

	operator GtkOptionMenu* () const;
	//!< Conversion operator; safely converts an OptionMenu to a GtkOptionMenu pointer.

	Menu* get_menu() const;
	//!< Returns the Menu associated with the OptionMenu.
	
	int get_history() const;
	//!< Retrieves the index of the currently selected menu item.
	//!< @return The index of the selected menu item, or -1 if there are no menu items.
	//!<
	//!< <BR>The menu items are numbered from top to bottom, starting with 0.

//! @}
//! @name Methods
//! @{

	void set_menu(Menu& menu);
	//!< Sets the Menu that is popped up to allow the user to choose a new value.
	//!< @param menu The Menu to associate with the option menu.
	//!<
	//!< <BR>You should provide a simple menu avoiding the use of tearoff menu items,
	//!< submenus, and accelerators.

	void remove_menu();
	//!< Removes the popup menu from the option menu.

	void set_history(unsigned int index);
	//!< Selects the menu item specified by index making it the newly selected value
	//!< for the option menu.
	//!< @param index The index of the menu item to select; index values are from 0 to n-1.

//! @}
//! @name Property Proxies
//! @{

	const MenuPropertyProxy prop_menu()
	{
		return MenuPropertyProxy(this, &menu_property);
	}
	//!< The menu of options (Menu* : Read / Write).

//! @}
//! @name Signal Proxies
//! @{
	
	const ChangedSignalProxy sig_changed()
	{
		return ChangedSignalProxy(this, &changed_signal);
	}
	//!< Connect to the changed_signal; emitted when the selected menu item changes.
	//!< Call get_history() to get the index of the selected menu item.

//! @}
};

} // namespace Gtk

} // namespace Inti

#endif // INTI_GTK_OPTION_MENU_H

