/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
//! @file inti/gtk/radiomenuitem.h
//! @brief A GtkRadioMenuItem C++ wrapper interface.
//!
//! Provides RadioMenuItem, a menu item widget that belongs to a group. At each instant exactly one
//! of the radio menu items from a group is selected. 

#ifndef INTI_GTK_RADIO_MENU_ITEM_H
#define INTI_GTK_RADIO_MENU_ITEM_H

#ifndef INTI_GTK_MENU_CHECK_ITEM_H
#include <inti/gtk/checkmenuitem.h>
#endif

#ifndef __GTK_RADIO_MENU_ITEM_H__
#include <gtk/gtkradiomenuitem.h>
#endif

namespace Inti {

namespace Gtk {

class RadioMenuItemClass;

//! @class RadioMenuItem radiomenuitem.h inti/gtk/radiomenuitem.h
//! @brief A GtkRadioMenuItem C++ wrapper class.
//!
//! A radio menu item is a check menu item that belongs to a group. At each instant exactly one
//! of the radio menu items from a group is selected. You create a group radio menu items the 
//! same way you would create a group of buttons (see RadioButton).
//!
//! The first radio menu item in a group is constructed with one of the constructors that doesn't
//! have a group argument. These are:
//! @code
//! RadioMenuItem();
//! RadioMenuItem(const String& label, bool use_underline = false);
//! @endcode
//! Then subsequent radio menu items in the group are constructed with one of the constructors that
//! requires a group. These are:<BR>
//! @code
//! RadioMenuItem(Group *group);
//! RadioMenuItem(const RadioMenuItem *group);
//! RadioMenuItem(Group *group, const String& label, bool use_underline = false);
//! RadioMenuItem(const RadioMenuItem*group, const String& label, bool use_underline = false);
//! @endcode
//! There are two types of group arguments. The first, <EM>Group</EM>, is a typedef for GSList and
//! can be retrieved by calling get_group(). The second is just a pointer to a RadioMenuItem that
//! already has a group assigned to it. To remove a RadioMenuItem from one group and make it part of
//! a new one, use set_group().

class RadioMenuItem : public CheckMenuItem
{
	friend class G::Object;
	friend class RadioMenuItemClass;

	RadioMenuItem(const RadioMenuItem&);
	RadioMenuItem& operator=(const RadioMenuItem&);
	
protected:
//! @name Constructors
//! @{

	explicit RadioMenuItem(GtkRadioMenuItem *radio_menu_item, bool reference = false);
	//!< Construct a new RadioMenuItem from an existing GtkRadioMenuItem.
	//!< @param radio_menu_item A pointer to a GtkRadioMenuItem.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>radio_menu_item</EM> can be a newly created GtkRadioMenuItem or an existing
	//!< GtkRadioMenuItem (see G::Object::Object).

//! @}

public:
	typedef GSList Group;
	//!< The radio menu item group.

//! @name Constructors
//! @{

	RadioMenuItem();
	//!< Construct a new radio menu item.

	explicit RadioMenuItem(Group *group);
	//!< Construct a new radio menu item. 
	//!< @param group The group to which the radio menu item is to be attached. 
 
	explicit RadioMenuItem(const RadioMenuItem *group);
	//!< Construct a new radio menu item. 
	//!< @param group An exisitng radio menu item. 

	explicit RadioMenuItem(const String& label, bool use_underline = false);
	//!< Construct a new radio menu item whose child is a simple Label. 
	//!< @param label The text for the label 
	//!< @param use_underline Set <EM>true</EM> if label contains a mnemonic character.

	RadioMenuItem(Group *group, const String& label, bool use_underline = false);
	//!< Construct a new radio menu item whose child is a simple Label. 
	//!< @param group The group to which the radio menu item is to be attached. 
	//!< @param label The text for the label 
	//!< @param use_underline Set <EM>true</EM> if label contains a mnemonic character.

	RadioMenuItem(const RadioMenuItem *group, const String& label, bool use_underline = false);
	//!< Construct a new radio menu item whose child is a simple Label. 
	//!< @param group An exisitng radio menu item. 
	//!< @param label The text for the label 
	//!< @param use_underline Set <EM>true</EM> if label contains a mnemonic character.

	virtual ~RadioMenuItem();
	//!< Destructor.
	
//! @}
//! @name Accessors
//! @{

	GtkRadioMenuItem* gtk_radio_menu_item() const { return (GtkRadioMenuItem*)instance; }
	//!< Get a pointer to the GtkRadioMenuItem structure.
	
	GtkRadioMenuItemClass* gtk_radio_menu_item_class() const;
	//!< Get a pointer to the GtkRadioMenuItemClass structure.
	
	operator GtkRadioMenuItem* () const;
	//!< Conversion operator; safely converts a RadioMenuItem to a GtkRadioMenuItem pointer.
	
	Group* get_group() const;
	//!< Returns the group to which the radio menu item belongs.

//! @}
//! @name Methods
//! @{

	void set_group(Group *group);
	//!< Sets the group of a radio menu item, or changes it. 
	//!< @param group The new group. 
 	
//! @}
};

} // namespace Gtk

} // namespace Inti

#endif // INTI_GTK_RADIO_MENU_ITEM_H

