
/* Command line oriented batch and dialog tool which creates, loads,
   manipulates and burns ISO 9660 filesystem images.

   Copyright 2007-2009 Thomas Schmitt, <scdbackup@gmx.net>

   Provided under GPL version 2.

   This file contains inner declarations of xorriso.
   The public interface is in xorriso.h
*/


/* For now, #ifdef Xorriso_is_xorriso_selF  has no meaning.
   But it is already now to be set only by the xorriso.c module.
*/

#ifndef Xorriso_private_includeD
#define Xorriso_private_includeD yes

#define Xorriso_program_versioN "0.3.4"

/** The source code release timestamp */
#include "xorriso_timestamp.h"
#ifndef Xorriso_timestamP
#define Xorriso_timestamP "-none-given-"
#endif

/** The binary build timestamp is to be set externally by the compiler
    or by a macro definition in xorriso_buildstamp.h.
*/
#include "xorriso_buildstamp.h"
#ifndef Xorriso_build_timestamP
#define Xorriso_build_timestamP "-none-given-"
#endif

#include <regex.h>


#define Smem_malloC malloc
#define Smem_freE free
#define SfileadrL 4096

struct LinkiteM;          /* Trace of hops during symbolic link resolution */
struct ExclusionS;        /* List of -not_* conditions */
struct PermiteM;          /* Stack of temporarily altered access permissions */
struct SpotlisT;          /* List of intervals with different read qualities */
struct CheckmediajoB;     /* Parameters for Xorriso_check_media() */
struct SectorbitmaP;      /* Distiniction between valid and invalid sectors */


/* maximum number of history lines to be reported with -status:long_history */
#define Xorriso_status_history_maX 100

/* <<< ??? */
/* try to catch signals and ignore them during abort handling */
#define Xorriso_abort_handler_defaulT (1|(2<<4))


/** The list of startup file names */
#define Xorriso_rc_nuM 4


/* Default setting for the size limit of single data files:
   100 extents with 4 GB - 2 kB each = 400 GB - 200 kB
*/
#define Xorriso_default_file_size_limiT \
        (((off_t) 400) * ((off_t) 1024*1024*1024) - (off_t) 204800)


struct XorrisO { /* the global context of xorriso */

 int libs_are_started;

 /* source */
 char progname[SfileadrL];
 char initial_wdx[SfileadrL];
 int no_rc;

 /* Command line argument emulations:
      0=xorriso mode
      1=mkisofs mode
      2=cdrecord mode
 */
 int argument_emulation;

 /** List of startupfiles */
 char rc_filenames[Xorriso_rc_nuM][SfileadrL];
 int rc_filename_count;

 char wdi[SfileadrL];
 char wdx[SfileadrL];
 int did_something_useful;

 int add_plainly;
 off_t split_size;

 char list_delimiter[81];

 /* >>> put libisofs aspects here <<< */

 int do_joliet;

 int do_aaip; /* bit0= ACL in , bit1= ACL out , bit2= EA in , bit3= EA out
                 bit4= record dev,inode , bit5= check dev,inode
                 bit6= omit content check if dev,inode check is conclusive
                 bit7= omit dev check mit bit5
              */

 int relax_compliance;        /* opaque bitfield to be set by xorrisoburn */
 int do_follow_pattern;
 int do_follow_param;
 int do_follow_links;
 int follow_link_limit;
 int do_follow_mount;
 int do_global_uid;
 uid_t global_uid;
 int do_global_gid;
 gid_t global_gid;
 int do_global_mode;
 mode_t global_dir_mode;
 mode_t global_file_mode;

 int do_overwrite; /* 0=off, 1=on, 2=nondir */
 int do_reassure;  /* 0=off, 1=on, 2=tree */

 char volid[33];
 int volid_default;
 char loaded_volid[33];
 char assert_volid[SfileadrL];
 char assert_volid_sev[80];

 char publisher[129];
 char application_id[129];

 char session_logfile[SfileadrL];
 int session_lba;
 int session_blocks;

 /* >>> put libburn/isoburn aspects here */

 struct Xorriso_lsT *drive_blacklist;
 struct Xorriso_lsT *drive_greylist;
 struct Xorriso_lsT *drive_whitelist;

 int toc_emulation_flag; /* bit0= bit3 for isoburn_drive_aquire()
                                  scan -ROM profiles for ISO sessions
                            bit1= bit4 for isoburn_drive_aquire()
                                  do not emulate TOC on overwriteable media
                         */

 int image_start_mode;  /* From what address to load the ISO image
                           bit0-15= addressing mode
                            0= automatic lba as deduced from media
                            1= value is session number
                            2= value is track number
                            3= value is lba
                           bit16= with mode 3 : value is possibly 16 too high.
                                  Let isoburn_set_msc1() adjust it.
                           bit30= interference with normal msc1 processing
                                  is enabled. Without this bit,
                                  isoburn_set_msc1() will not be called.
                           bit31= image loading has happened,
                                  setting is kept for rollback only.
                                  Always apply as 0=auto.
                        */
 char image_start_value[81]; /* value according image_start_mode */

 char indev[SfileadrL];
 void *in_drive_handle;  /* interpreted only by xorrisoburn.c */
 void *in_volset_handle; /* interpreted only by xorrisoburn.c */
 char *in_charset;       /* The charset to interpret the filename bytes */

 int volset_change_pending; /* whether -commit would make sense */
 int no_volset_present;     /* set to 1 on first failure */

 struct CheckmediajoB *check_media_default;
 struct SectorbitmaP *in_sector_map; /* eventual sector validity bitmap */


 char outdev[SfileadrL];
 void *out_drive_handle; /* interpreted only by xorrisoburn.c */
 char *out_charset;      /* The charset to produce the filename bytes for */
 int dev_fd_1; /* The fd which substitutes for /dev/fd/1 and is
                     connected to externaly perveived stdout.
                  */

 int grow_blindly_msc2; /* if >= 0 this causes growing from drive to drive.
                           The value is used as block address offset for
                           image generation. Like in: mkisofs -C msc1,msc2
                         */

 int ban_stdio_write;
 int do_dummy;
 int do_close;
 int speed;     /* in libburn units : 1000 bytes/second , 0 = Max, -1 = Min */
 int fs;        /* fifo size in 2048 byte chunks : at most 1 GB */
 int padding;   /* number of bytes to add after ISO 9660 image */

 int alignment; /* if > 0 : image size alignment in 2048 byt blocks */
                /* <<< not sure whether to keep this after libisofs will have
                       learned to pad up MBR images to full MB */

 int do_stream_recording; /* 0=no, 1=yes, 2=for data, not for dir
                             >=16 means yes with number as start LBA  */

 int keep_boot_image;
 int patch_isolinux_image;
 char boot_image_bin_path[SfileadrL];
 int boot_image_emul; /* 0=no emulation
                        (1=emulation as hard disk)
                        (2=emulation as floppy)
                      */
 char boot_image_cat_path[SfileadrL];
 off_t boot_image_load_size;
 int boot_image_isohybrid; /* 0=off , 1=auto , 2=on , 3=force */

 /* LBA of boot image after image loading */
 int loaded_boot_bin_lba;
 /* Path of the catalog node after image loading */
 char loaded_boot_cat_path[SfileadrL];

 /* XORRISO options */
 int allow_graft_points;

 int allow_restore; /* 0= disallowed, 1=allowed, 2=device files allowed */
 int do_concat_split;  /* 1= restore complete split file directories as
                             regular files
                        */
 int do_auto_chmod;    /* 1= eventually temporarily open access permissions
                             of self-owned directories during restore
                        */

 int dialog;  /* 0=off , 1=single-line , 2=multi-line */

 int bsl_interpretation;
                     /* whether to run input through Sfile_bsl_interpreter():
                        bit0-1= dialog and quoted file reading
                          0= no interpretation, leave unchanged
                          1= only inside double quotes
                          2= outside single quotes
                          3= everywhere
                        bit2-3= reserved as future expansion of bit0-1
                        bit4= interpretation within program start arguments
                        bit5= perform backslash encoding with results
                        bit6= perform backslash encoding with info texts
                     */

 /* Pattern matching facility. It still carries legacy from scdbackup/askme.c
    but is fully functional for xorriso.
 */
 int search_mode;
 /* 0= start text
    1= fgrep , 
    2= regular expression
    3= (eventually structured) shell parser expression
    4= shell parser expression for leaf name
 */

 int structured_search;
 /* 0= flat text search
    1= '/' is a significant separator that cannot be matched by wildcards
    ( 2= like 1 : but report only occurence in tree, no payload, no location )
    ( 3= like 2 : but report first content level of matching directories )
    4= actually not structured but unique find mode (with search_mode 4)
 */

 int do_iso_rr_pattern; /* 0=off, 1=on, 2=ls */
 int do_disk_pattern;   /* 0=off, 1=on, 2=ls */

 int temp_mem_limit;

 off_t file_size_limit;

 struct ExclusionS *disk_exclusions;
 int disk_excl_mode; /* bit0= on (else off)
                        bit1= parameter too (else rekursion only)
                        bit2= whole subtree banned (else only exact path)
                        bit3= when comparing ignore excluded files rather
                              than to treat them as truely missing on disk
                     */

 int use_stdin; /* use raw stdin even if readline support is compiled */
 int result_page_length;
 int result_page_width;
 char mark_text[SfileadrL]; /* ( stdout+stderr, M: ) */
 int packet_output;
 char logfile[4][SfileadrL];
 FILE *logfile_fp[4];
 FILE *pktlog_fp;
 struct Xorriso_lsT *result_msglists[Xorriso_max_outlist_stacK];
 struct Xorriso_lsT *info_msglists[Xorriso_max_outlist_stacK];
 int msglist_flags[Xorriso_max_outlist_stacK]; /* bit0= result is redirected
                                                  bit1= info is redirected
                                               */
 int msglist_stackfill;

 int status_history_max; /* for -status long_history */
 

 char report_about_text[20];
 int report_about_severity;
 int library_msg_direct_print;
 char abort_on_text[20];
 int abort_on_severity; /* A severity rank number as threshold */
 int problem_status; /* Severity rank number. 0= no abort condition present */
 char problem_status_text[20];

 char errfile_log[SfileadrL]; /* for -errfile_log */
 int errfile_mode; /* bit0= marked */
 FILE *errfile_fp;

 int img_read_error_mode; /* 0=best_effort , 1=failure , 2=fatal */
 int extract_error_mode; /* 0=best_effort , 1=keep , 2=delete */

 char return_with_text[20];
 int return_with_severity;
 int return_with_value;
 int eternal_problem_status;
 char eternal_problem_status_text[20];

 /* temporary search facilities */
 regex_t *re;
 regmatch_t match[1]; 
 char **re_constants;
 int re_count;
 int re_fill;
 char reg_expr[2*SfileadrL];

 /* run state */
 int run_state; /* 0=preparing , 1=writing image */
 int is_dialog;
 int bar_is_fresh;
 char pending_option[SfileadrL]; /* eventual option entered at page prompt */
 int request_to_abort; /* abort a single operation like -ls, not the program */
 int request_not_to_ask; /* suppress reassure and pager */
 double idle_time;
 int re_failed_at; /* mismatch position with structured_search */
 int prepended_wd;
 double insert_count;
 double insert_bytes;
 double error_count; /* double will not roll over */

 /* pacifiers */
 int pacifier_style; /* 0= xorriso, 1=mkisofs 2=cdrecord */
 double pacifier_interval;
 double start_time;
 double last_update_time;
 /* optional global counters for brain reduced callback functions */
 off_t pacifier_count;
 off_t pacifier_total;
 off_t pacifier_byte_count; /* auxiliary counter for data bytes */

 void *pacifier_fifo;

 int find_compare_result; /* 1=everything matches , 0=mismatch , -1=error */ 

 struct PermiteM *perm_stack; /* Temporarily altered dir access permissions */

 /* result (stdout, R: ) */
 char result_line[10*SfileadrL];
 int result_line_counter;
 int result_page_counter;
 int result_open_line_len;


 /* info (stderr, I:) */
 char info_text[10*SfileadrL];

};


int Xorriso_prepare_regex(struct XorrisO *xorriso, char *adr, int flag);

int Xorriso_result(struct XorrisO *xorriso, int flag);

int Xorriso_info(struct XorrisO *xorriso, int flag);

int Xorriso_request_confirmation(struct XorrisO *xorriso, int flag);

int Xorriso_prescan_args(struct XorrisO *xorriso, int argc, char **argv,
                         int flag);

int Xorriso_execute_option(struct XorrisO *xorriso, char *line, int flag);

/* @return 0=match , else no match
*/
int Xorriso_regexec(struct XorrisO *xorriso, char *to_match, int *failed_at,
                    int flag);

int Xorriso_prepare_expansion_pattern(struct XorrisO *xorriso, char *pattern,
                                      int flag);

int Xorriso__mode_to_perms(mode_t st_mode, char perms[11], int flag);

int Xorriso_much_too_long(struct XorrisO *xorriso, int len, int flag);

int Xorriso_check_temp_mem_limit(struct XorrisO *xorriso, off_t mem, int flag);

int Xorriso_eval_nonmatch(struct XorrisO *xorriso, char *pattern,
                          int *nonconst_mismatches, off_t *mem, int flag);

/* @param flag bit0= a match count !=1 is a SORRY event
*/
int Xorriso_check_matchcount(struct XorrisO *xorriso,
                int count, int nonconst_mismatches, int num_patterns,
                char **patterns, int flag);

int Xorriso_no_pattern_memory(struct XorrisO *xorriso, off_t mem, int flag);

int Xorriso_alloc_pattern_mem(struct XorrisO *xorriso, off_t mem,
                              int count, char ***filev, int flag);

/* @param flag bit0= count results rather than storing them
   @return <=0 error , 1 is root (end processing) ,
                       2 is not root (go on processing)
*/
int Xorriso_check_for_root_pattern(struct XorrisO *xorriso,
              int *filec, char **filev, int count_limit, off_t *mem, int flag);

/* @param flag bit0= prepend wd only if name does not begin by '/'
               bit2= prepend wd (automatically done if wd[0]!=0)
*/
int Xorriso_make_abs_adr(struct XorrisO *xorriso, char *wd, char *name,
                             char adr[], int flag);

/* @param flag bit0= count result rather than storing it
               bit1= unexpected change of number is a FATAL event
*/
int Xorriso_register_matched_adr(struct XorrisO *xorriso,
                               char *adr, int count_limit,
                               int *filec, char **filev, off_t *mem, int flag);

int Xorriso_format_ls_l(struct XorrisO *xorriso, struct stat *stbuf, int flag);

/* @param flag bit0= simple readlink(): no normalization, no multi-hop
*/
int Xorriso_resolve_link(struct XorrisO *xorriso,
                     char *link_path, char result_path[SfileadrL], int flag);

/* @param flag bit0= for Xorriso_msgs_submit: use pager
*/
int Xorriso_hop_link(struct XorrisO *xorriso, char *link_path,
                   struct LinkiteM **link_stack, struct stat *stbuf, int flag);

/* reg_expr should be twice as large as bourne_expr ( + 2 to be exact) */
/* return: 2= bourne_expr is surely a constant */
int Xorriso__bourne_to_reg(char bourne_expr[], char reg_expr[], int flag);

int Xorriso_no_malloc_memory(struct XorrisO *xorriso, char **to_free,
                             int flag);

int Xorriso_pacifier_reset(struct XorrisO *xorriso, int flag);

/* This call is to be issued by long running workers in short intervals.
   It will check whether enough time has elapsed since the last pacifier
   message and eventually issue an update message.
   @param what_done  A sparse description of the action, preferrably in past 
                     tense. E.g. "done" , "files added".
   @param count The number of objects processed so far.
                Is ignored if <=0.
   @param todo  The number of objects to be done in total.
                Is ignored if <=0.
   @param current_object A string telling the object currently processed.
                         Ignored if "".
   @param flag  bit0= report unconditionally, no time check
*/
int Xorriso_pacifier_callback(struct XorrisO *xorriso, char *what_done,
                              off_t count, off_t todo, char *current_object,
                              int flag);

/* @param boss_iter Opaque handle to be forwarded to actions in ISO image
                    Set to NULL if calling this function from outside ISO world
   @param flag bit0= update rather than compare
*/
int Xorriso_find_compare(struct XorrisO *xorriso, void *boss_iter,
                         char *iso_path, char *iso_prefix, char *disk_prefix, 
                         int flag);

/* @param boss_iter Opaque handle to be forwarded to actions in ISO image
                    Set to NULL if calling this function from outside ISO world
*/
int Xorriso_update_interpreter(struct XorrisO *xorriso, void *boss_iter,
                               int compare_result, char *disk_path,
                               char *iso_rr_path, int flag);

int Xorriso_path_is_excluded(struct XorrisO *xorriso, char *path, int flag);

/* @param flag bit0= long format
               bit1= do not print count of nodes
               bit2= du format
               bit3= print directories as themselves (ls -d)
*/
int Xorriso_lsx_filev(struct XorrisO *xorriso, char *wd,
                      int filec, char **filev, off_t boss_mem, int flag);

/*
   @param flag >>> bit0= remove whole sub tree: rm -r
               bit1= remove empty directory: rmdir
               bit2= recursion: do not reassure in mode 2 "tree"
               bit3= this is for overwriting and not for plain removal
               bit4= count deleted files in xorriso->pacifier_count
               bit5= with bit0 only remove directory content, not the directory
   @return   <=0 = error
               1 = removed leaf file object
               2 = removed directory or tree
               3 = did not remove on user revocation
*/
int Xorriso_rmx(struct XorrisO *xorriso, off_t boss_mem, char *path, int flag);


int Xorriso_make_tmp_path(struct XorrisO *xorriso, char *orig_path,
                          char *tmp_path, int *fd, int flag);

/* @param flag bit0= path is a directory
               bit2= recursion: do not reassure in mode 2 "tree"
               bit3= this is for overwriting and not for plain removal
*/
int Xorriso_reassure_restore(struct XorrisO *xorriso, char *path, int flag);


int Xorriso_auto_chmod(struct XorrisO *xorriso, char *disk_path, int flag);

int Xorriso_protect_stdout(struct XorrisO *xorriso, int flag);


/* @param flag bit0= mark untested areas as valid
*/
int Xorriso_spotlist_to_sectormap(struct XorrisO *xorriso,
                                  struct SpotlisT *spotlist,
                                  int read_chunk,
                                  struct SectorbitmaP **map,
                                  int flag);

int Xorriso_toc_to_string(struct XorrisO *xorriso, char **toc_text, int flag);


/* @param flag bit0+1= what to aquire after giving up outdev
                       0=none, 1=indev, 2=outdev, 3=both
*/
int Xorriso_reaquire_outdev(struct XorrisO *xorriso, int flag);


/* Opens the -check_media data copy in for reading and writing
*/
int Xorriso_open_job_data_to(struct XorrisO *xorriso,
                             struct CheckmediajoB *job, int flag);

int Xorriso_no_findjob(struct XorrisO *xorriso, char *cmd, int flag);


int Xorriso_make_mount_cmd(struct XorrisO *xorriso, char *cmd,
                           int lba, int track, int session, char *volid,
                           char *devadr, char result[SfileadrL], int flag);


/* @param flag bit0= use env_path to find the desired program
*/
int Xorriso_execv(struct XorrisO *xorriso, char *cmd, char *env_path,
                  int *status, int flag);

int Xorriso_is_in_patternlist(struct XorrisO *xorriso,
                        struct Xorriso_lsT *patternlist, char *path, int flag);

char *Xorriso_get_pattern(struct XorrisO *xorriso,
                         struct Xorriso_lsT *patternlist, int index, int flag);


/* Normalize ACL and sort apart "access" ACL from "default" ACL.
 */
int Xorriso_normalize_acl_text(struct XorrisO *xorriso, char *in_text,
                    char **access_acl_text, char **default_acl_text, int flag);

int Xorriso_path_setfattr(struct XorrisO *xorriso, void *in_node, char *path,
                       char *name, size_t value_length, char *value, int flag);


int Sfile_str(char target[SfileadrL], char *source, int flag);

double Sfile_microtime(int flag);

int Sfile_add_to_path(char path[SfileadrL], char *addon, int flag);

int Sfile_scale(double value, char *result, int siz, double thresh, int flag);

int Sfile_destroy_argv(int *argc, char ***argv, int flag);

/*
 bit0= do not ignore trailing slash
 bit1= do not ignore empty components (other than the empty root name)
*/
int Sfile_count_components(char *path, int flag);

/*
 @param flag
 bit0= return -1 if file is missing
 bit1= return a hardlink with siblings as type 5
 bit2= evaluate eventual link target rather than the link object itself
 bit3= return a socket or a char device as types 7 or 8 rather than 0
 @return
  0=unknown
  1=regular
  2=directory
  3=symbolic link
  4=named pipe
  5=multiple hardlink (with bit1)
  6=block device
  7=socket (with bit3)
  8=character device (with bit3)
*/
int Sfile_type(char *filename, int flag);

/* @param flag bit0= only encode inside quotes
               bit1= encode < 32 outside quotes except 7, 8, 9, 10, 12, 13
               bit2= encode in any case above 126
               bit3= encode in any case shellsafe:
                     <=42 , 59, 60, 62, 63, 92, 94, 96, >=123
*/
int Sfile_bsl_encoder(char **result, char *text, size_t text_len, int flag);


struct Xorriso_lsT {
  char *text;
  struct Xorriso_lsT *prev,*next;
};

/** Create a new list item with arbitrary byte content.
    @param lstring  The newly created object or NULL on failure
    @param data     An array of bytes to be copied into the new object
    @param data_len Number of bytes to be copied
    @param link     Xorriso_lsT object to which the new object shall be linked
    @param flag Bitfield for control purposes
                bit0= insert before link rather than after it
                bit1= do not copy data (e.g. because *data is invalid)
    @return <=0 error, 1 ok
*/
int Xorriso_lst_new_binary(struct Xorriso_lsT **lstring, char *data,
                           int data_len, struct Xorriso_lsT *link, int flag);


/** Create a new list item with a 0-terminated text as content.
    @param lstring  The newly created object or NULL on failure
    @param text     A 0-terminated array of bytes
    @param link     Xorriso_lsT object to which the new object shall be linked
    @param flag Bitfield for control purposes
                bit0= insert before link rather than after it
    @return <=0 error, 1 ok
*/
int Xorriso_lst_new(struct Xorriso_lsT **lstring, char *text,
                    struct Xorriso_lsT *link, int flag);


/** Create a new list item at the end of a given list.
    @param lstring  Contains as input a pointer to a pointer to any existing
                    list item. As output this list item pointer will be
                    changed to the address of the new list item.
    @param data     An array of bytes to be copied into the new object
    @param data_len Number of bytes to be copied
    @param flag     unused yet, submit 0
    @return <=0 error, 1 ok
*/
int Xorriso_lst_append_binary(struct Xorriso_lsT **entry,
                              char *data, int data_len, int flag);


/** Destroy a single list item and connect its eventual list neighbors.
    @param lstring  pointer to the pointer to be freed and set to NULL
    @param flag     unused yet, submit 0
    @return 0= *lstring was alredy NULL, 1= ok
*/
int Xorriso_lst_destroy(struct Xorriso_lsT **lstring, int flag);


char *Text_shellsafe(char *in_text, char *out_text, int flag);

int Sort_argv(int argc, char **argv, int flag);

/* @param flag bit0= single letters */
char *Ftypetxt(mode_t st_mode, int flag);

/* @param flag bit0=with year and seconds
               bit1=timestamp format YYYY.MM.DD.hhmmss
*/
char *Ftimetxt(time_t t, char timetext[40], int flag);

int System_uname(char **sysname, char **release, char **version, 
                 char **machine, int flag);


struct DirseQ;

int Dirseq_new(struct DirseQ **o, char *adr, int flag);

int Dirseq_destroy(struct DirseQ **o, int flag);

int Dirseq_next_adr(struct DirseQ *o, char reply[SfileadrL], int flag);


int Linkitem_reset_stack(struct LinkiteM **o, struct LinkiteM *to, int flag);


struct FindjoB;


int Findjob_new(struct FindjoB **o, char *start_path, int flag);

int Findjob_destroy(struct FindjoB **job, int flag);


/* @return 0=no match , 1=match , <0 = error
*/
int Findjob_test(struct FindjoB *job, char *name, 
                 struct stat *boss_stbuf, struct stat *stbuf,
                 int depth, int flag);

/* @return <0 error, >=0 see xorriso.c struct FindjoB.action
*/
int Findjob_get_action(struct FindjoB *o, int flag);

/* @return <0 error, >=0 see xorriso.c struct FindjoB.action
*/ 
int Findjob_get_action_parms(struct FindjoB *o, char **target, char **text_2,
                             uid_t *user, gid_t *group,
                             mode_t *mode_and, mode_t *mode_or,
                             int *type, time_t *date, struct FindjoB **subjob,
                             int flag);

/* @param flag bit0= recursive
*/
int Findjob_set_action_target(struct FindjoB *o, int action, char *target,
                              int flag);

/* @param flag bit0= recursive
*/
int Findjob_set_action_chgrp(struct FindjoB *o, gid_t group, int flag);

/* @param flag bit0= recursive
*/
int Findjob_set_action_chmod(struct FindjoB *o,
                             mode_t mode_and, mode_t mode_or, int flag);

/* @param flag bit0= recursive
*/
int Findjob_set_action_ad(struct FindjoB *o, int type, time_t date, int flag);

int Findjob_set_start_path(struct FindjoB *o, char *start_path, int flag);

int Findjob_set_action_found_path(struct FindjoB *o, int flag);

int Findjob_get_start_path(struct FindjoB *o, char **start_path, int flag);

int Findjob_set_lba_range(struct FindjoB *o, int start_lba, int count,
                          int flag);

int Findjob_get_lba_damage_filter(struct FindjoB *o, int *start_lba,
                                  int *end_lba, int *damage_filter, int flag);

int Findjob_get_commit_filter(struct FindjoB *o, int *commit_filter, int flag);

int Findjob_get_acl_filter(struct FindjoB *o, int *acl_filter, int flag);

int Findjob_get_xattr_filter(struct FindjoB *o, int *xattr_filter, int flag);

int Findjob_get_aaip_filter(struct FindjoB *o, int *aaip_filter, int flag);

int Findjob_set_wanted_node(struct FindjoB *o, void *wanted_node, int flag);

int Findjob_get_wanted_node(struct FindjoB *o, void **wanted_node, int flag);

int Findjob_set_found_path(struct FindjoB *o, char *path, int flag);

int Findjob_get_found_path(struct FindjoB *o, char **path, int flag);


struct SplitparT;

int Splitparts_new(struct SplitparT **o, int count, int flag);

int Splitparts_destroy(struct SplitparT **o, int count, int flag);

int Splitparts_set(struct SplitparT *o, int idx,
                   char *name, int partno, int total_parts,
                   off_t offset, off_t bytes, off_t total_bytes, int flag);

int Splitparts_get(struct SplitparT *o, int idx, char **name, int *partno,
                   int *total_parts, off_t *offset, off_t *bytes,
                   off_t *total_bytes, int flag);

int Splitpart__parse(char *name, int *partno, int *total_parts,
                    off_t *offset, off_t *bytes, off_t *total_bytes, int flag);

int Splitpart__compose(char *adr, int partno, int total_parts,
                       off_t offset, off_t bytes, off_t total_bytes, int flag);

int Splitparts_sort(struct SplitparT *o, int count, int flag);


int Permstack_push(struct PermiteM **o, char *disk_path, struct stat *stbuf,
                   int flag);

int Permstack_pop(struct PermiteM **o, struct PermiteM *stopper,
                  struct XorrisO *xorriso, int flag);


int Spotlist_new(struct SpotlisT **o, int flag);

int Spotlist_destroy(struct SpotlisT **o, int flag);

int Spotlist_add_item(struct SpotlisT *o, int start_lba, int blocks,
                      int quality, int flag);

int Spotlist_count(struct SpotlisT *o, int flag);

int Spotlist_block_count(struct SpotlisT *o, int flag);

int Spotlist_sector_size(struct SpotlisT *o, int read_chunk, int flag);

int Spotlist_get_item(struct SpotlisT *o, int idx,
                      int *start_lba, int *blocks, int *quality, int flag);

char *Spotlist__quality_name(int quality, char name[80], int flag);


#define Xorriso_read_quality_gooD         0x7fffffff
#define Xorriso_read_quality_sloW         0x60000000
#define Xorriso_read_quality_partiaL      0x50000000
#define Xorriso_read_quality_valiD        0x40000000
#define Xorriso_read_quality_untesteD     0x3fffffff
#define Xorriso_read_quality_invaliD      0x3ffffffe
#define Xorriso_read_quality_tao_enD      0x28000000
#define Xorriso_read_quality_off_tracK    0x20000000
#define Xorriso_read_quality_unreadablE   0x00000000


int Checkmediajob_new(struct CheckmediajoB **o, int flag);

int Checkmediajob_destroy(struct CheckmediajoB **o, int flag);

int Checkmediajob_copy(struct CheckmediajoB *from, struct CheckmediajoB *to,
                       int flag);

int Sectorbitmap_new(struct SectorbitmaP **o, int sectors, int sector_size,
                     int flag);
int Sectorbitmap_destroy(struct SectorbitmaP **o, int flag);
int Sectorbitmap_from_file(struct SectorbitmaP **o, char *path, char *msg,
                           int *os_errno, int flag);
int Sectorbitmap_to_file(struct SectorbitmaP *o, char *path, char *info,
                         char *msg, int *os_errno, int flag);
int Sectorbitmap_set(struct SectorbitmaP *o, int sector, int flag);
int Sectorbitmap_set_range(struct SectorbitmaP *o,
                           int start_sector, int sectors, int flag);
int Sectorbitmap_is_set(struct SectorbitmaP *o, int sector, int flag);
int Sectorbitmap_bytes_are_set(struct SectorbitmaP *o,
                               off_t start_byte, off_t end_byte, int flag);

int Sectorbitmap_get_layout(struct SectorbitmaP *o,
                           int *sectors, int *sector_size, int flag);

int Sectorbitmap_copy(struct SectorbitmaP *from, struct SectorbitmaP *to,
                      int flag);

/* bit0= append (text!=NULL) */
int Sregex_string(char **handle, char *text, int flag);

/* @param flag bit0= only test expression whether compilable
*/
int Sregex_match(char *pattern, char *text, int flag);

#endif /* Xorriso_private_includeD */

