/*
 *  The OSS interface
 *
 */

#define _GNU_SOURCE

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/soundcard.h>
#include <sys/poll.h>
#include <fcntl.h>
#include <time.h>

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <dlfcn.h>
#include <sys/time.h>
#include <jack/jack.h>

#include "libjackasyn.h"
#include "libjackoss.h"

/* BSDI has this functionality, but not define :() */
#if defined(RTLD_NEXT)
#define REAL_LIBC RTLD_NEXT
#else
#define REAL_LIBC ((void *) -1L)
#endif


#define DEBUGCOND debug
//#define DEBUGCOND 1

#define DEBUG(x) if (DEBUGCOND) { x; }
#define DEBUG2(x) if (debug > 1) { x; }


#if 0
int (*_select)(int n, fd_set *readfds, fd_set *writefds, fd_set *exceptfds, struct timeval *timeout);
int (*_poll)(struct pollfd *ufds, unsigned int nfds, int timeout);
int (*_open)(const char *file, int oflag, ...);
int (*_close)(int fd);
ssize_t (*_write)(int fd, const void *buf, size_t n);
ssize_t (*_read)(int fd, void *buf, size_t n);
int (*_ioctl)(int fd, unsigned long request, ...);
int (*_fcntl)(int fd, int cmd, ...);
#endif


/* 
 *   
 * Open the JACK connection
 *
 */

#define SIZE_INT16 2


struct oss_dev {
  virdev* virdev;
  int open_count;
  int oss_fd;
};

static struct oss_dev* global_oss_dev = NULL;
static int debug = 0;

static int check_state(int handle) 
{
  return global_oss_dev &&  global_oss_dev->virdev &&
    (handle == global_oss_dev->oss_fd);
}



/*************************************************************
 *  Implementation of the open() function
 */

int jackoss_open(const char* name,int flags, ...)
{
  va_list args;
  struct oss_dev* h;
  mode_t mode;
  static int (*func) (const char *, int, mode_t) = NULL;
  int in,out;

  if (!func)
    func = (int (*) (const char *, int, mode_t)) dlsym (REAL_LIBC, "open");

  if (getenv("JACKASYN_DEBUG")) {
	  debug = atoi(getenv("JACKASYN_DEBUG"));
  }

  va_start (args, flags);
  mode = va_arg (args, mode_t);
  va_end (args);

  if ((strncmp(name,"/dev/dsp",8) && strncmp(name,"/dev/audio",8))
      ||getenv("JACKASYN_USE_OSS") ) {
       return (*func) (name, flags, mode);
  }

  DEBUG(fprintf(stderr,"%s\n",__FUNCTION__));

  h = global_oss_dev;
  if (!global_oss_dev) {
    h = global_oss_dev = (struct oss_dev*) malloc(sizeof(global_oss_dev));
    h->virdev=NULL;
    h->open_count = 0;
    h->oss_fd = -1;
    debug = 0;
  }
  if (h->open_count) {
    DEBUG(fprintf(stderr,"open failed, open count = %d\n",h->open_count));
    return h->oss_fd;
    //    return -1;
  }

  if (h->virdev) {
    virdev_reset(h->virdev);
    h->open_count++;
    return h->oss_fd;    
  }

  out = 2;
  in = 2;
  //  if (flags && O_WRONLY) out = 2,in = 0;
  //  if (flags && O_RDONLY) out = 0, in = 2;

  if (!(h->virdev = virdev_connect("oss",in,out))) {
    free(h);
    h = global_oss_dev= NULL;
    return (*func) (name, flags, mode);
  }

  h->open_count++;
  
  // TODO: What can we use as fd ??
  // it would be cool if linux IPC had a fd for synchronisation
  // like IRIX had. Can this be done with pipes, sockets ??
  // 
  // The problem is that pipes and sockets are either readable
  // or writeable, therefor we can not use them to fake a device
  // that is not readable and not writeable at the same time ....
  //
  // one solution would be to implement a kernel module for that
  // .... anyhow, now we have the select() and poll() hack that 
  // follows at the end of this file ...
  //

  h->oss_fd = open("/dev/zero",O_RDWR);
  virdev_start(h->virdev);

  DEBUG(fprintf(stderr,"%s done %d\n",__FUNCTION__,h->oss_fd);)
  return (long) h->oss_fd;
}

#define UNIMPLEMENTED(a) case a: fprintf(stderr,"ioctl: "#a" not implemented\n");break;


/*************************************************************
 *  Implementation of the ioctl() function
 */

int jackoss_ioctl(int handle, unsigned long int flag, ... )
{
  va_list args;
  void* data = NULL;
  audio_buf_info* bufinfo;
  count_info* cinfo;
  virdev* h;
  static int (*func) (long, int, void*) = NULL;


  va_start (args, flag);
  data = va_arg (args, void*);
  va_end (args);

  bufinfo = (audio_buf_info*) data;
  cinfo = (count_info*) data;

  if (!func)
    func = (int (*) (long, int, void*)) dlsym (REAL_LIBC, "ioctl");

  if (!check_state(handle)) return func(handle,flag,data); 
  DEBUG(fprintf(stderr,"%s\n",__FUNCTION__);)

  h = global_oss_dev->virdev;

  switch(flag) {
  case SNDCTL_DSP_RESET:
    DEBUG(fprintf(stderr,"RESET\n");)
    h->ofifo.writep = 0;
    h->ofifo.readp = 0;
    h->ififo.writep = 0;
    h->ififo.readp = 0;
    break;

  case SNDCTL_DSP_SYNC:
    DEBUG(fprintf(stderr,"SYNC\n");)
    atomic_write(h->ififo.writep,0);
    atomic_write(h->ififo.readp,0);
    atomic_write(h->ofifo.writep,0);
    atomic_write(h->ofifo.readp,0);
    break;

  case SOUND_PCM_READ_RATE:
  case SNDCTL_DSP_SPEED:    
    DEBUG(fprintf(stderr,"SPEED %d (wanted %d ) \n",h->rate,*((int*)data));)
    h->rate = *((int*)data);
    break;

  case SNDCTL_DSP_GETBLKSIZE:
    DEBUG(fprintf(stderr,"GETBLKSIZE\n");)
    *((int*)data) = h->fragsize;
    break;


  case SNDCTL_DSP_STEREO:
     DEBUG(fprintf(stderr,"STEREO %d",*((int*)data)));
    if (*((int*)data))
	h->ichans = h->ochans = 2;
    else
	h->ichans = h->ochans = 1;
    break;
    
  case SNDCTL_DSP_CHANNELS:
    DEBUG(fprintf(stderr,"CHANNELS\n");)
    if (h->ichans) h->ichans = *((int*)data);
    if (h->ochans) h->ochans = *((int*)data);
    break;

  case SOUND_PCM_READ_CHANNELS:
    *((int*)data) = 8;
    break;
  
    UNIMPLEMENTED(SOUND_PCM_WRITE_FILTER);

  case SNDCTL_DSP_POST:
    DEBUG(fprintf(stderr,"POST\n");)
    break;

    UNIMPLEMENTED(SNDCTL_DSP_SUBDIVIDE);

  case SNDCTL_DSP_SETFRAGMENT:
    DEBUG(fprintf(stderr,"SETFRAGMENT wanted %d\n",*((int*)data));)
    {
      if (h->fragsize == 8192)
	*((int*)data) = (h->fragnum<<16) | 15; 
      if (h->fragsize == 4096)
	*((int*)data) = (h->fragnum<<16) | 14; 
      if (h->fragsize == 2048)
	*((int*)data) = (h->fragnum<<16) | 13; 
      if (h->fragsize == 1024)
	*((int*)data) = (h->fragnum<<16) | 12; 
      if (h->fragsize == 512)
	*((int*)data) = (h->fragnum<<16) | 11; 
      if (h->fragsize == 128)
	*((int*)data) = (h->fragnum<<16) | 10; 
      if (h->fragsize == 64)
	*((int*)data) = (h->fragnum<<16) | 9; 

      DEBUG(fprintf(stderr,"SETFRAGMENT %d %d\n",*(int*)data,h->fragsize);)
      break;
    }

  case SNDCTL_DSP_SETFMT:
  case SNDCTL_DSP_GETFMTS:
    DEBUG(fprintf(stderr,"SETFMT %d (wanted %d)\n",AFMT_S16_NE,*((int*)data));)
    *((int*)data) = AFMT_S16_NE;
    break;
    
  case SOUND_PCM_READ_BITS:
    *((int*)data) = 16;
    break;
    
  case SNDCTL_DSP_GETOSPACE:
    DEBUG(fprintf(stderr,"GETOSPACE\n"));
    bufinfo->bytes = fifo_empty(&h->ofifo)*(h->ochans*SIZE_INT16);
    bufinfo->fragsize = h->fragsize*h->ochans*SIZE_INT16;
    bufinfo->fragstotal = h->fragnum;
    bufinfo->fragments = (bufinfo->bytes>>1)/(h->fragsize*h->ochans*SIZE_INT16);    
    DEBUG(fprintf(stderr,"GETOSPACE %d\n",bufinfo->bytes));
    break;

  case SNDCTL_DSP_GETISPACE:
    DEBUG(fprintf(stderr,"GETISPACE\n"));
    bufinfo->bytes = fifo_filled(&h->ofifo)*(h->ochans*SIZE_INT16);
    bufinfo->fragsize = h->fragsize*h->ochans*SIZE_INT16;
    bufinfo->fragstotal = h->fragnum;
    bufinfo->fragments = (bufinfo->bytes>>1)/(h->fragsize*h->ochans*SIZE_INT16);    
    DEBUG(fprintf(stderr,"GETISPACE %d\n",bufinfo->bytes));
    break;


    UNIMPLEMENTED(SNDCTL_DSP_NONBLOCK	);

  case SNDCTL_DSP_GETCAPS:
    DEBUG(fprintf(stderr,"GETCAPS\n"));

    *((int*)data) = DSP_CAP_DUPLEX|0xff;
    break;

    UNIMPLEMENTED( SNDCTL_DSP_GETTRIGGER);
    UNIMPLEMENTED( SNDCTL_DSP_SETTRIGGER);

  case SNDCTL_DSP_GETIPTR:
    DEBUG(fprintf(stderr,"GETIPTR \n"));

    cinfo->ptr = h->ififo.writep;
    cinfo->bytes = h->ififo.framecount*h->ochans*sizeof(short);
    cinfo->blocks = h->ififo.framecount;
    DEBUG(fprintf(stderr,"GETIPTR %d\n",cinfo->bytes));
    break;

  case SNDCTL_DSP_GETOPTR:
    DEBUG(fprintf(stderr,"GETOPTR \n"));

    cinfo->ptr = h->ofifo.writep;
    cinfo->bytes = h->ofifo.framecount*h->ochans*sizeof(short);
    cinfo->blocks = h->ofifo.framecount;
    DEBUG(fprintf(stderr,"GETOPTR %d\n",cinfo->bytes));
    break;

  case SNDCTL_DSP_GETODELAY:
    DEBUG(fprintf(stderr,"GETODELAY\n"));
    *((int*)data) = fifo_filled(&h->ofifo)*h->ochans*sizeof(short);
    break;

    UNIMPLEMENTED( SNDCTL_DSP_MAPINBUF);
    UNIMPLEMENTED( SNDCTL_DSP_MAPOUTBUF);
    UNIMPLEMENTED( SNDCTL_DSP_SETSYNCRO);
    UNIMPLEMENTED( SNDCTL_DSP_SETDUPLEX);
    UNIMPLEMENTED( SNDCTL_DSP_GETCHANNELMASK);
    UNIMPLEMENTED( SNDCTL_DSP_BIND_CHANNEL);
    UNIMPLEMENTED( SNDCTL_DSP_SETSPDIF);
    UNIMPLEMENTED( SNDCTL_DSP_GETSPDIF);

  default:
    fprintf(stderr,"unknown ioctl\n");
  }
  DEBUG(fprintf(stderr,"%s \n",__FUNCTION__));
  return 0;
}


/*************************************************************
 *  Implementation of the write() function
 */

ssize_t jackoss_write(int handle, const void* buf, size_t len)
{
  static int (*func) (long, const void*, int) = NULL;

  if (!func)
    func = (int (*) (long, const void*, int)) dlsym (REAL_LIBC, "write");
  if (!check_state(handle)) return (*func)(handle,buf,len); 

  DEBUG2(fprintf(stderr,"%s buf=0x%p len = %d\n",__FUNCTION__,buf,len));

  if (!buf) fprintf(stderr,"%s FATAL: NULL buffer arg\n",__FUNCTION__);

  return virdev_output16i(global_oss_dev->virdev,buf,len);
}


/*************************************************************
 *  Implementation of the read() function
 */

ssize_t jackoss_read(int handle, void* buf, size_t len)
{
  static int (*func) (long, void*, int) = NULL;

  if (!func)
    func = (int (*) (long, void*, int)) dlsym (REAL_LIBC, "read");
  
  if (!check_state(handle)) return (*func)(handle,buf,len);
  DEBUG2(fprintf(stderr,"%s \n",__FUNCTION__));

  if (!buf) fprintf(stderr,"%s FATAL: NULL buffer arg\n",__FUNCTION__);

  /* Do that later */
  return virdev_input16i(global_oss_dev->virdev,buf,len);
}


/*************************************************************
 *  Implementation of the close() function
 */

int jackoss_close(int handle) 
{
  static int (*func) (long) = NULL;

  if (!func)
    func = (int (*) (long)) dlsym (REAL_LIBC, "close");

  if (!check_state(handle)) return (*func)(handle);
  DEBUG(fprintf(stderr,"%s \n",__FUNCTION__));

  global_oss_dev->open_count--;
  //  if (global_oss_dev->open_count == 0) virdev_shutdown(h);

  return 0;
}


#if 0  // this doesnt help with the shutdown problem
void jackoss_exit(int status) {
  void (*func) (int) = (void (*)(int))dlsym (REAL_LIBC, "_exit");;
  DEBUG(fprintf(stderr,"exiting ....\n"));
  (*func)(status);
  virdev_shutdown(global_oss_dev->virdev);
}
#endif
/*************************************************************
 *  Implementation of the select() function
 */

/* just fake select if through is 1 (testing purpose) REMOVE ME */

#define through 0

int jackoss_select(int  n,  fd_set  *readfds,  
		       fd_set *writefds, 
		       fd_set *exceptfds, struct timeval *timeout)
{
  virdev* h;
  int wait = 1;
  int ret;
  struct timeval tv;
  int w_sec = 0;
  int w_usec = 0;
  int testread = 0;
  int testwrite = 0;
  fd_set oldw;
  fd_set oldr;
  static int (*func) (int,  fd_set*,fd_set*,fd_set*,struct timeval*) = NULL;


  if (!func)
    func = (int (*) (int,  fd_set*,fd_set*,fd_set*,struct timeval*)) dlsym (REAL_LIBC, "select");

  /* 
   * Call the libc select if needed 
   */

  if (!global_oss_dev || !global_oss_dev->virdev) 
    return (*func)(n,readfds,writefds,exceptfds,timeout);

  DEBUG2(fprintf(stderr,"%s \n",__FUNCTION__));

  h = global_oss_dev->virdev;
  if (readfds) {
    testread = FD_ISSET(global_oss_dev->oss_fd,readfds);
    FD_CLR(global_oss_dev->oss_fd,readfds);
  }
  if (writefds) {
    testwrite = FD_ISSET(global_oss_dev->oss_fd,writefds);
    FD_CLR(global_oss_dev->oss_fd,writefds);
  }

  tv.tv_sec = 0;
  tv.tv_usec = 0;

  /* backup */
  if (writefds) oldw = *writefds;
  if (readfds) oldr = *readfds;

  while (wait) {

    /* check standard filedescriptors */

    if ((ret = (*func)(n,readfds,writefds,exceptfds,&tv))) {
      wait  = 0;
    }


    /* check jack file descriptors */

    if (testwrite) 
      if  (through || fifo_empty(&h->ofifo) >= 64) {
	FD_SET(global_oss_dev->oss_fd,writefds);
	ret++;
	wait = 0;
      }

    if (testread)
      if  (through || fifo_filled(&h->ififo) >= 64) {
	FD_SET(global_oss_dev->oss_fd,readfds);
	ret++;
	wait = 0;
      }

    /* check timeout */

    w_usec+=10000;
    w_sec+=w_usec/1000000;
    w_usec%=1000000;

    if (timeout)
      if (w_sec >= timeout->tv_sec && w_usec >= timeout->tv_sec)
	wait = 0;

    if (!wait) break;

    /* revert to old state, do another select */

    if (writefds) *writefds = oldw;
    if (readfds) *readfds = oldr;

    tv.tv_sec = 0;
    tv.tv_usec = 10000;
  }
  DEBUG2(fprintf(stderr,"%s returned %d\n",__FUNCTION__,ret));
  return ret;
}


/*************************************************************
 *  Implementation of the poll() function
 */

int jackoss_poll (struct pollfd * ufds, nfds_t nfds, int timeout)
{
  virdev* h;
  int space;
  int changed = 0;
  int i = 0;
  static int (*func)(struct pollfd *, nfds_t, int) = NULL;

  if (!func)
    func = (int (*) (struct pollfd *, nfds_t, int)) dlsym (REAL_LIBC, "poll");

  if (!global_oss_dev || !global_oss_dev->virdev) 
    return (*func)(ufds,nfds,timeout);

  h = global_oss_dev->virdev;

  space = fifo_empty(&h->ofifo);
  DEBUG2(fprintf(stderr,"%s space %d timeout %d nfds %ld\n",__FUNCTION__,space,timeout,nfds));

  while (i<nfds && (ufds[i].fd != global_oss_dev->oss_fd)) i++;

  if (i<nfds) {
    if (space > 64) {
      ufds[i].revents = ufds[i].events & (POLLOUT|POLLIN);
      changed++;
    }
    return changed;
  }

  changed += (*func)(ufds,nfds,timeout);
  return changed;
}




/*
 *
 * And all of this for sox
 *
 */


FILE * jackoss_fopen (const char *pathname, const char *mode)
{
  int fd;
  static FILE *(*func) (const char *, const char *) = NULL;


  if (!func) 
    func = (FILE *(*) (const char *, const char *)) dlsym (REAL_LIBC, "fopen");
  
  if (strncmp(pathname,"/dev/dsp",8) || getenv("JACKASYN_USE_OSS")) 
    return (*func) (pathname, mode);
  
  fd = open(pathname,O_RDWR);
  return fdopen(fd,mode);
}


size_t  jackoss_fread(void *ptr, size_t size, size_t nmemb, FILE*stream) 
{
  int fn;
  static size_t (*func) (void *, size_t, size_t, FILE*) = NULL;

  fn = fileno(stream);

  if (!func) 
    func = (size_t (*)(void *, size_t, size_t, FILE*) ) dlsym (REAL_LIBC, "fread");

  if (!check_state(fn)) return (*func)(ptr,size,nmemb,stream);

  return jackoss_read(fn,ptr,size*nmemb)/size;
}

size_t jackoss_fwrite( const void *ptr, size_t size, size_t nmemb,FILE *stream)
{
  int fn;
  static size_t (*func) (const void *, size_t, size_t, FILE*) = NULL;
  
  fn = fileno(stream);
  
  if (!func) 
    func = (size_t (*)(const void *, size_t, size_t, FILE*) ) dlsym (REAL_LIBC, "fwrite");
  
  if (!check_state(fn)) return (*func)(ptr,size,nmemb,stream);

  return write(fn,ptr,size*nmemb)/size;
}

#ifdef LD_PRELOAD

# define strong_alias(name, aliasname) \
  extern __typeof (name) aliasname __attribute__ ((alias (#name)));
strong_alias(open, __open);
strong_alias(close, __close);
strong_alias(write, __write);
strong_alias(read, __read);
strong_alias(ioctl, __ioctl);
strong_alias(fcntl, __fcntl);
strong_alias(poll, __poll);
strong_alias(select, __select);

strong_alias(open, _open);
strong_alias(close, _close);
strong_alias(write, _write);
strong_alias(read, _read);
strong_alias(ioctl, _ioctl);
strong_alias(fcntl, _fcntl);
strong_alias(poll, _poll);
strong_alias(select, _select);

#if 0
static void initialize() __attribute__ ((constructor));

static void initialize()
{
        _open = dlsym(RTLD_NEXT, "open");
        _close = dlsym(RTLD_NEXT, "close");
        _write = dlsym(RTLD_NEXT, "write");
        _read = dlsym(RTLD_NEXT, "read");
        _ioctl = dlsym(RTLD_NEXT, "ioctl");
        _fcntl = dlsym(RTLD_NEXT, "fcntl");
        _select = dlsym(RTLD_NEXT, "select");
        _poll = dlsym(RTLD_NEXT, "poll");
}
#endif

#endif
