package jdepend.framework;

import java.io.*;
import java.util.*;

/**
 * The <code>JarFileParserTest</code> is a <code>TestCase</code> 
 * for the <code>JarFileParser</code> class.
 * 
 * @author <b>Mike Clark</b> (mike@clarkware.com)
 * @author Clarkware Consulting, Inc.
 */
 
public class JarFileParserTest extends JDependTestCase {

	private File jarFile;
    private File zipFile;
	
	
	public JarFileParserTest(String name) {
		super(name);
	}

	protected void setUp() {
		super.setUp();
		
        jarFile = new File(getTestDataDir() + "test.jar"); 
        zipFile = new File(getTestDataDir() + "test.zip"); 
	}

	protected void tearDown() {
	    super.tearDown();
	}
    
    public void testParseInvalidJarFile() throws IOException {
        
        JavaClassBuilder builder = new JavaClassBuilder();
        File bogusFile = new File(getTestDataDir() + "bogus.jar");
        
        try {
            
            builder.buildClasses(bogusFile);
            fail("Should raise IOException");
        
        } catch(IOException success) {
        }
    }
    
    public void testParseInvalidZipFile() throws IOException {
        
        JavaClassBuilder builder = new JavaClassBuilder();
        File bogusFile = new File(getTestDataDir() + "bogus.zip");
        
        try {
            
            builder.buildClasses(bogusFile);
            fail("Should raise IOException");
        
        } catch(IOException success) {
        }
    }

    public void testParseJarFile() throws IOException {
        
        JavaClassBuilder builder = new JavaClassBuilder();

        Collection classes = builder.buildClasses(jarFile);
        assertEquals(4, classes.size());
        
        assertClassesExist(classes);    
        assertTrue(classes.contains(new JavaClass("jdepend.framework." +
            "MockConcreteClass$MockInnerClass")));  
    }
    
    public void testParseJarFileWithoutInnerClasses() throws IOException {
        
        FileManager fm = new FileManager();
        fm.acceptInnerClasses(false);
        
        JavaClassBuilder builder = new JavaClassBuilder(fm);

        Collection classes = builder.buildClasses(jarFile);
        assertEquals(3, classes.size());
        
        assertClassesExist(classes);  
    }
    
    public void testParseZipFile() throws IOException {
        
        JavaClassBuilder builder = new JavaClassBuilder();

        Collection classes = builder.buildClasses(zipFile);
        assertEquals(4, classes.size());
        
        assertClassesExist(classes);
        assertTrue(classes.contains(new JavaClass("jdepend.framework." +
            "MockConcreteClass$MockInnerClass"))); 
    }
    
    public void testParseZipFileWithoutInnerClasses() throws IOException {
        
        FileManager fm = new FileManager();
        fm.acceptInnerClasses(false);
        
        JavaClassBuilder builder = new JavaClassBuilder(fm);

        Collection classes = builder.buildClasses(zipFile);
        assertEquals(3, classes.size());
        
        assertClassesExist(classes);  
    }
    
    private void assertClassesExist(Collection classes) {
        assertTrue(classes.contains(new JavaClass("jdepend.framework.MockAbstractClass")));
        assertTrue(classes.contains(new JavaClass("jdepend.framework.MockInterface")));
        assertTrue(classes.contains(new JavaClass("jdepend.framework.MockConcreteClass")));
    }
    
    public void testCountClasses() throws IOException {
        
        JDepend jdepend = new JDepend();
        jdepend.addDirectory(getTestDataDir());
        
        jdepend.analyzeInnerClasses(true);
        assertEquals(8, jdepend.countClasses());
        
        jdepend.analyzeInnerClasses(false);
        assertEquals(6, jdepend.countClasses());
    }
            
	public static void main(String args[]) {
		junit.textui.TestRunner.run(JarFileParserTest.class);
	}
}
