/**
 * Copyright (c) 2006, www.jempbox.org
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of pdfbox; nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * http://www.jempbox.org
 *
 */
package org.jempbox.xmp;

import java.io.IOException;
import java.util.Calendar;
import java.util.List;

import org.w3c.dom.Element;

/**
 * Define XMP properties that are common to all schemas.
 * 
 * @author Ben Litchfield (ben@benlitchfield.com)
 * @version $Revision: 1.1 $
 */
public class XMPSchemaBasic extends XMPSchema
{
    /**
     * The namespace of this schema.
     */
    public static final String NAMESPACE = "http://ns.adobe.com/xap/1.0/";
    
    /**
     * Construct a new blank PDF schema.
     *
     * @param parent The parent metadata schema that this will be part of.
     */
    public XMPSchemaBasic( XMPMetadata parent )
    {
        super( parent, "xmp", NAMESPACE );
        schema.setAttributeNS( 
            "http://ns.adobe.com/xap/1.0/g/img/", 
            "xmlns:xapGImg", 
            "http://ns.adobe.com/xap/1.0/g/img/" );
    }
    
    /**
     * Constructor from existing XML element.
     * 
     * @param element The existing element.
     */
    public XMPSchemaBasic( Element element )
    {
        super( element );
        if( schema.getAttribute( "xmlns:xapGImg" ) == null )
        {
            schema.setAttributeNS( 
                "http://ns.adobe.com/xap/1.0/g/img/", 
                "xmlns:xapGImg", 
                "http://ns.adobe.com/xap/1.0/g/img/" );
        }
    }
    
    /**
     * Remove an Advisory xpath expression.
     *
     * @param advisory An xpath expression specifying properties that
     * were edited outside of the authoring application.
     */
    public void removeAdvisory( String advisory )
    {
        removeBagValue( "xmp:Advisory", advisory );
    }
    
    /**
     * Add an advisory to the list.
     *
     * @param advisory The new advisory xpath expression.
     */
    public void addAdvisory( String advisory )
    {
        addBagValue( "xmp:Advisory", advisory );
    }
    
    /**
     * Get the complete list of advisories.
     *
     * @return The list of advisories.
     */
    public List getAdvisories()
    {
        return getBagList( "xmp:Advisory" );
    }
    
    /**
     * The base URL of the resource, for relative URLs in the document.
     *
     * @param url The base URL.
     */
    public void setBaseURL( String url )
    {
        setTextProperty( "xmp:BaseURL", url );
    }
    
    /**
     * Get the base URL of the resource.
     *
     * @return The base URL.
     */
    public String getBaseURL()
    {
        return getTextProperty( "xmp:BaseURL" );
    }
    
    /**
     * Set the creation date of the resource.
     *
     * @param date The creation date of the resource.
     */
    public void setCreateDate( Calendar date )
    {
        setDateProperty( "xmp:CreateDate", date );
    }
    
    /**
     * Get the creation date of the resource.
     *
     * @return The creation date of the resource.
     * 
     * @throws IOException If there is an error while converting this property to
     * a date.
     */
    public Calendar getCreateDate() throws IOException
    {
        return getDateProperty( "xmp:CreateDate" );
    }
    
    /**
     * The creator tool for the resource.  In the form of "vendor app version", ie
     * "Adobe Acrobat Distiller 5.0"
     *
     * @param creator The tool that was used to create the resource.
     */
    public void setCreateTool( String creator )
    {
        setTextProperty( "xmp:CreateTool", creator );
    }
    
    /**
     * Get the tool that created this resource, in the form of "vendor app version", ie
     * "Adobe Acrobat Distiller 5.0"
     *
     * @return The creator tool.
     */
    public String getCreateTool()
    {
        return getTextProperty( "xmp:CreateTool" );
    }
    
    /**
     * Remove an identifier to this resource.
     *
     * @param id An identifier to this resource.
     */
    public void removeIdentifier( String id )
    {
        removeBagValue( "xmp:Identifier", id );
    }
    
    /**
     * Add a new identifier for this resource.
     *
     * @param id A new identifier for this resource.
     */
    public void addIdentifier( String id )
    {
        addBagValue( "xmp:Identifier", id );
    }
    
    /**
     * Get the complete list of identifiers.
     *
     * @return The list of identifiers.
     */
    public List getIdentifiers()
    {
        return getBagList( "xmp:Identifier" );
    }
    
    /**
     * Set a short phrase that identifies this resource.
     *
     * @param label A short description of this resource.
     */
    public void setLabel( String label )
    {
        setTextProperty( "xmp:Label", label );
    }
    
    /**
     * Get the short phrase that describes this resource.
     *
     * @return The label for this resource.
     */
    public String getLabel()
    {
        return getTextProperty( "xmp:Label" );
    }
    
    /**
     * Set the date that any metadata was updated.
     *
     * @param date The metadata change date for this resource.
     */
    public void setMetadataDate( Calendar date )
    {
        setDateProperty( "xmp:MetadataDate", date );
    }
    
    /**
     * Get the metadata change date for this resource.
     *
     * @return The metadata change date of the resource.
     * 
     * @throws IOException If there is an error while converting this property to
     * a date.
     */
    public Calendar getMetadataDate() throws IOException
    {
        return getDateProperty( "xmp:MetadataDate" );
    }
    
    /**
     * Set the date that the resource was last modified.
     *
     * @param date The modify date for this resource.
     */
    public void setModifyDate( Calendar date )
    {
        setDateProperty( "xmp:ModifyDate", date );
    }
    
    /**
     * Get the date the resource was last modified.
     *
     * @return The modify date of the resource.
     * 
     * @throws IOException If there is an error while converting this property to
     * a date.
     */
    public Calendar getModifyDate() throws IOException
    {
        return getDateProperty( "xmp:ModifyDate" );
    }
    
    /**
     * Set a short informal name for the resource.
     *
     * @param nickname A short name of this resource.
     */
    public void setNickname( String nickname )
    {
        setTextProperty( "xmp:Nickname", nickname );
    }
    
    /**
     * Get the short informal name for this resource.
     *
     * @return The short name for this resource.
     */
    public String getNickname()
    {
        return getTextProperty( "xmp:Nickname" );
    }
    
    /**
     * Get a number that indicates the documents status.
     * 
     * @return The rating of the document.
     */
    public Integer getRating()
    {
        return getIntegerProperty( "xmp:Rating" );
    }
    
    /**
     * Set the document status.
     * 
     * @param rating A number indicating status relative to other documents.
     */
    public void setRating( Integer rating )
    {
        setIntegerProperty( "xmp:Rating", rating );
    }
    
    /**
     * Set the default value for the thumbnail.
     *
     * @param thumbnail The thumbnail of this resource.
     */
    public void setThumbnail( Thumbnail thumbnail )
    {
        setThumbnailProperty( "xmp:Thumbnails", null, thumbnail );
    }
    
    /**
     * Get the default value for the thumbnail.
     *
     * @return The thumbnail of this resource.
     */
    public Thumbnail getThumbnail()
    {
        return getThumbnailProperty( "xml:Thumnails", null );
    }
    
    /**
     * Set the thumbnail of this resource in a specific language.
     *
     * @param language The language code.
     * @param thumbnail The thumbnail in a specific language.
     */
    public void setThumbnail( String language, Thumbnail thumbnail )
    {
        setThumbnailProperty( "xmp:Thumbnails", language, thumbnail );
    }
    
    /**
     * Get the thumbnail in a specific language.
     *
     * @param language The language code to get the description for.
     *
     * @return The thumbnail in the specified language or null if it does not exist.
     */
    public Thumbnail getThumbnail( String language )
    {
        return getThumbnailProperty( "xmp:Thumbnails", language );
    }
    
    /**
     * Get a list of all languages that a thumbnail exists for.
     *
     * @return A non-null list of languages, potentially an empty list.
     */
    public List getThumbnailLanguages()
    {
        return getLanguagePropertyLanguages( "xml:Thumbnails" );
    }
}