/* Certificate.java -- SSL Certificate message.
   Copyright (C) 2003  Casey Marshall <rsdio@metastatic.org>

This file is a part of Jessie.

Jessie is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

Jessie is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with Jessie; if not, write to the

   Free Software Foundation, Inc.,
   59 Temple Place, Suite 330,
   Boston, MA  02111-1307
   USA  */


package org.metastatic.jessie.provider;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.EOFException;
import java.io.InputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.StringReader;
import java.io.StringWriter;

import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;

import java.util.LinkedList;

import javax.net.ssl.SSLProtocolException;

final class Certificate implements Handshake.Body
{

  // Fields.
  // -------------------------------------------------------------------------

  private final X509Certificate[] certs;

  // Constructors.
  // -------------------------------------------------------------------------

  Certificate(X509Certificate[] certs)
  {
    if (certs == null)
      {
        throw new NullPointerException();
      }
    this.certs = certs;
  }

  // Class methods.
  // -------------------------------------------------------------------------

  static Certificate read(InputStream in) throws IOException
  {
    int len = (in.read() & 0xFF) << 16 | (in.read() & 0xFF) << 8
            | (in.read() & 0xFF);
    byte[] buf = new byte[len];
    int count = 0;
    while (count < len)
      {
        int l = in.read(buf, count, len - count);
        if (l == -1)
          {
            throw new EOFException("unexpected end of stream");
          }
        count += l;
      }
    try
      {
        LinkedList certs = new LinkedList();
        CertificateFactory fact = CertificateFactory.getInstance("X.509");
        ByteArrayInputStream bin = new ByteArrayInputStream(buf);
        count = 0;
        while (count < len)
          {
            int len2 = (bin.read() & 0xFF) << 16 | (bin.read() & 0xFF) << 8
                     | (bin.read() & 0xFF);
            certs.add(fact.generateCertificate(bin));
            count += len2 + 3;
          }
        return new Certificate((X509Certificate[])
          certs.toArray(new X509Certificate[certs.size()]));
      }
    catch (CertificateException ce)
      {
        throw new SSLProtocolException(ce.getMessage());
      }
  }

  // Instance methods.
  // -------------------------------------------------------------------------

  public void write(OutputStream out) throws IOException
  {
    ByteArrayOutputStream bout = new ByteArrayOutputStream();
    try
      {
        for (int i = 0; i < certs.length; i++)
          {
            byte[] enc = certs[i].getEncoded();
            bout.write((enc.length >>> 16) & 0xFF);
            bout.write((enc.length >>>  8) & 0xFF);
            bout.write( enc.length & 0xFF);
            bout.write(enc);
          }
      }
    catch (CertificateEncodingException cee)
      {
        throw new Error("cannot encode certificates");
      }
    catch (IOException ignored)
      {
      }
    out.write(bout.size() >>> 16 & 0xFF);
    out.write(bout.size() >>>  8 & 0xFF);
    out.write(bout.size() & 0xFF);
    bout.writeTo(out);
  }

  X509Certificate[] getCertificates()
  {
    return certs;
  }

  public String toString()
  {
    StringWriter str = new StringWriter();
    PrintWriter out = new PrintWriter(str);
    out.println("struct {");
    out.println("  certificateList =");
    for (int i = 0; i < certs.length; i++)
      {
        BufferedReader r =
          new BufferedReader(new StringReader(certs[i].toString()));
        String s;
        try
          {
            while ((s = r.readLine()) != null)
              {
                out.print("    ");
                out.println(s);
              }
          }
        catch (IOException ignored)
          {
          }
      }
    out.println("} Certificate;");
    return str.toString();
  }
}
