/* Finished.java -- SSL Finished message.
   Copyright (C) 2003  Casey Marshall <rsdio@metastatic.org>

This file is a part of Jessie.

Jessie is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

Jessie is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with Jessie; if not, write to the

   Free Software Foundation, Inc.,
   59 Temple Place, Suite 330,
   Boston, MA  02111-1307
   USA  */


package org.metastatic.jessie.provider;

import java.io.DataInputStream;
import java.io.InputStream;
import java.io.IOException;
import java.io.OutputStream;

final class Finished implements Handshake.Body
{

  // Fields.
  // -------------------------------------------------------------------------

  /** TLSv1.x verify data. */
  private final byte[] verifyData;

  /** SSLv3 message digest pair. */
  private final byte[] md5, sha;

  // Constructor.
  // -------------------------------------------------------------------------

  Finished(byte[] verifyData)
  {
    this.verifyData = verifyData;
    md5 = sha = null;
  }

  Finished(byte[] md5, byte[] sha)
  {
    this.md5 = md5;
    this.sha = sha;
    verifyData = null;
  }

  // Class methods.
  // -------------------------------------------------------------------------

  static Finished read(InputStream in, CipherSuite suite)
    throws IOException
  {
    DataInputStream din = new DataInputStream(in);
    if (suite.getVersion().equals(ProtocolVersion.SSL_3))
      {
        byte[] md5 = new byte[16];
        byte[] sha = new byte[20];
        din.readFully(md5);
        din.readFully(sha);
        return new Finished(md5, sha);
      }
    else
      {
        byte[] buf = new byte[12];
        din.readFully(buf);
        return new Finished(buf);
      }
  }

  // Instance methods.
  // -------------------------------------------------------------------------

  public void write(OutputStream out) throws IOException
  {
    if (verifyData != null)
      out.write(verifyData);
    else
      {
        out.write(md5);
        out.write(sha);
      }
  }

  byte[] getVerifyData()
  {
    return verifyData;
  }

  byte[] getMD5Hash()
  {
    return md5;
  }

  byte[] getSHAHash()
  {
    return sha;
  }

  public String toString()
  {
    String nl = System.getProperty("line.separator");
    if (verifyData != null)
      {
        return "struct {" + nl +
          "  verifyData = " + Util.toHexString(verifyData, ':') + ";" + nl +
          "} Finished;" + nl;
      }
    else
      {
        return "struct {" + nl +
          "  md5Hash = " + Util.toHexString(md5, ':') + ";" + nl +
          "  shaHash = " + Util.toHexString(sha, ':') + ";" + nl +
          "} Finished;" + nl;
      }
  }
}
