/*  This file is part of LingoTeach, the Language Teaching program 
 *  Copyright (C) 2001-2003 The LingoTeach Team
 *
 *  This program is free software; you can redistribute it and/or modify 
 *  it under the terms of the GNU General Public License as published by 
 *  the Free Software Foundation; either version 2 of the License, or 
 *  (at your option) any later version.  
 *
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *  GNU General Public License for more details.  
 *
 *  You should have received a copy of the GNU General Public License 
 *  along with this program; if not, write to the Free Software 
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 */

#include <libxml/xpath.h>
#include <string.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "lingoteach.h"
#include "conf.h"
#include "lesson.h"
#include "language.h"


#define SEARCH "/%s/lang/@id"

/*********************
 * private functions *
 *********************/


/********************
 * public functions *
 ********************/

/**
 * Gets a NULL terminated list of available languages (based on the language
 * file set in the lingConfig). The return value has to be freed by the user.
 *
 * \param settings The lingConfig to use for looking up the languages.
 * \return a NULL terminated list of the languages
 */
lingchar**
ling_lang_get_languages (lingConfig *settings)
{
  xmlXPathObjectPtr   ptr;
  xmlXPathContextPtr  ctxPtr;
  xmlDocPtr           doc;
  lingchar  *this;
  lingchar **retval;
  lingchar  *search;
  int          i;
  unsigned int s;
  
  if (settings->langfile == NULL || settings->appname == NULL)
     return NULL;

  doc = xmlParseFile (settings->langfile);
  if (doc == NULL)
    return NULL;

  ctxPtr = lesson_get_xpath (doc);
  if (ctxPtr == NULL)
    {
      xmlFreeDoc (doc);
      return NULL;
    }

  search = malloc (strlen (SEARCH) + strlen (settings->appname));
  if (search == NULL)
    {
      xmlXPathFreeContext (ctxPtr);
      xmlFreeDoc (doc);
      return NULL;
    }

  sprintf (search, SEARCH, settings->appname);

  /* use the defined language file */
  ptr = xmlXPathEval (search, ctxPtr);
  if (ptr == NULL)
    {
      xmlXPathFreeContext (ctxPtr);
      xmlFreeDoc (doc);
      free (search);
      return NULL;
    }

  /* allocate space for the language array */
  retval = malloc (sizeof (char *) * ptr->nodesetval->nodeNr);
  if (retval == NULL)
    {
      xmlXPathFreeObject (ptr);
      xmlXPathFreeContext (ctxPtr);
      xmlFreeDoc (doc);
      free (search);
      return NULL;
    }

  /* fill the language array */
  for (i = 0; i < ptr->nodesetval->nodeNr; i++)
    {
      this = xmlXPathCastNodeToString (ptr->nodesetval->nodeTab[i]);
      s = strlen (this) + 1;
      
      retval[i] = malloc (s);
      if (retval[i] == NULL)
        ling_lang_free_languages (retval, i - 1);
	
      strncpy (retval[i], this, s);
    }

  xmlXPathFreeObject (ptr);
  xmlXPathFreeContext (ctxPtr);
  xmlFreeDoc (doc);
  free (search);

  return retval;
}

/**
 * Gets the count of languages contained in a lingConfig.
 *
 * \return An integer showing the maximum count of the languages.
 */
int
ling_lang_get_max_languages (lingConfig *settings)
{
  xmlXPathObjectPtr   ptr;
  xmlXPathContextPtr  ctxPtr;
  xmlDocPtr           doc;
  int       i;
  lingchar *search;
  
  if (settings == NULL || settings->langfile == NULL)
    return -1;

  doc = xmlParseFile (settings->langfile);
  if (doc == NULL)
    return -1;

  ctxPtr = lesson_get_xpath (doc);
  if (ctxPtr == NULL)
    {
      xmlFreeDoc(doc);
      return -1;
    }

  search = malloc (strlen (SEARCH) + strlen (settings->appname));
  if (search == NULL)
    {
      xmlXPathFreeContext (ctxPtr);
      xmlFreeDoc (doc);
      return -1;
    }
  
  sprintf (search, SEARCH, settings->appname);

  /* use the defined language file */  
  if ((ptr = xmlXPathEval (search, ctxPtr)) == NULL)
    {
      xmlXPathFreeContext (ctxPtr);
      xmlFreeDoc (doc);
      free (search);
      return -1;
    }
  
  i = ptr->nodesetval->nodeNr;

  xmlXPathFreeObject (ptr);
  xmlXPathFreeContext (ctxPtr);
  xmlFreeDoc (doc);
  free (search);

  return i;
}

/**
 * Frees a previously allocated array of languages.
 *
 * \param languages The array of languages
 * \param size The size of the array
 */
void 
ling_lang_free_languages (lingchar **languages, int size)
{
  int i;
  for (i = 0; i < size; i++)
    free (languages[i]);
  free (languages);    
}
