/*  This file is part of LingoTeach, the Language Teaching program 
 *  Copyright (C) 2001-2003 The LingoTeach Team
 *
 *  This program is free software; you can redistribute it and/or modify 
 *  it under the terms of the GNU General Public License as published by 
 *  the Free Software Foundation; either version 2 of the License, or 
 *  (at your option) any later version.  
 *
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *  GNU General Public License for more details.  
 *
 *  You should have received a copy of the GNU General Public License 
 *  along with this program; if not, write to the Free Software 
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 */

#ifndef LINGOTEACH_H
#define LINGOTEACH_H 1

#include "lingdefs.h"

/* 
 *  This file defines all functions that can be used to access
 *  data. No XML filenames should be ever visible outside this module.
 */

/**
 * Creates a new, empty lingConfig for usage.
 * The lingConfig has to be freed by the user.
 *
 * \return A new lingConfig.
 */
lingConfig*
ling_conf_init_config (void);

/**
 * Sets the application name in a lingConfig.
 *
 * \param settings The settings to modify.
 * \param name The name of the application, which uses the library.
 * \return the modified settings, if the name could be applied to them.
 */
lingConfig*
ling_conf_set_app_name (lingConfig *settings, lingchar *name);

/**
 * Sets the full qualified path to the language file in a lingConfig.
 *
 * \param settings The settings to modify.
 * \param lfile The file, which contains the languages.
 * \return The modifed settings, if the language file could be applied to them.
 */
lingConfig*
ling_conf_set_lang_file (lingConfig *settings, lingchar *lfile);

/**
 * Creates a new lingMeaning and returns it. 
 * The lingMeaning has to be freed by the user.
 *
 * \return A new, empty lingMeaning.
 */
lingMeaning*
ling_meaning_get_new (void);

/**
 * Searches for a lingMeaning, that is the same type as the one, that is passed
 * to it. This helps to make sure, that multiple choice answers are not 
 * obvious. (Currently not functional!).
 * The lingMeaning has to be freed by the user.
 *
 * \param meaning A lingMeaning, for which a similar one should be found.
 * \return A meaning that is the same type of thing as the one that
 * is passed to it.
 */
/*
  lingMeaning*
  ling_meaning_get_similar (lingMeaning *meaning);
*/

/**
 * Gets another lingMeaning from a lesson file.
 * The lingMeaning has to be freed by the user.
 *
 * \param lesson The lesson to fetch the meaning from.
 * \param type The method to use for getting the lingMeaning.
 * \param language The language, which should be used.
 * \return A meaning pointer, using the method given as argument 
 * to the function.
 */
lingMeaning*
ling_meaning_get_another_meaning (lingLesson *lesson, Method type, 
				  char *language);

/**
 * Gets a specific lingMeaning from the given lesson.
 * The lingMeaning has to be freed by the user.
 * 
 * \param lesson The lesson to fetch the meaning from.
 * \param id The id, which should be searched for.
 * \param language The language, which should be used.
 * \return a lingMeaning containing the meaning, which has the given id.
 * If none is found with the given language, the function returns NULL.
 */
lingMeaning*
ling_meaning_get_by_word_id (lingLesson *lesson, int id, char *language);

/**
 * Frees the memory used by a list of lingMeaning and the lingMeanings itself
 *
 * \param meaning The meaning list to free.
 */
void
ling_meaning_free_meaning (lingMeaning *meaning);

/**
 * Frees the memory used by a lingMeaning and the lingMeaning itself
 *
 * \param tree The list of meanings, in which the meanings is.
 * \param node The meaning, which should be freed.
 * return The new list without the freed meaning.
 */
lingMeaning*
ling_meaning_free_meaning_1 (lingMeaning *tree, lingMeaning *node);

/**
 * Saves a list of meanings into a given lesson file. The format is 
 * the standard lingoteach lesson format.
 * 
 * \param meaning The list of meanings to save.
 * \param filename The file the meanings should be saved in.
 * \param settings The settings to use for the file.
 * \return TRUE on succesful saving, else FALSE.
 */ 
lingbool
ling_meaning_save_meanings (lingMeaning *meaning, char *filename, 
			    lingConfig *settings);

/**
 * Adds a new meaning at the end of the given list.
 *
 * \param tree The meaning list to which the meaning should be added.
 * \param meaning The meaning to add to the tree.
 * \return The new, modified tree.
 */
lingMeaning*
ling_meaning_add_meaning (lingMeaning *tree, lingMeaning *meaning);

/**
 * Inserts a meaning after specific meaning into a meaning list.
 *
 * \param tree The meaning list to which the meaning should be added.
 * \param parent The parent meaning, after whihc the child should be added.
 * \param child The meaning to add.
 * \return The new, modified tree.
 */
lingMeaning*
ling_meaning_insert_after_meaning (lingMeaning *tree, lingMeaning *parent,
				   lingMeaning *child);

/**
 * Modifies a meaning of the given list of meanings
 *
 * \param tree The list of meanings in which the meaning exists.
 * \param id The id of the meaning, which should be modified.
 * \param meaning The modified meaning .
 * \return The tree with the modified meaning.
 */
lingMeaning*
ling_meaning_modify_meaning (lingMeaning *tree, int id, lingMeaning *meaning);

/**
 * Returns the path to the sound snippet for the given meaning.
 *
 * \param soundpath The full qualified path to the sound files.
 * \param meaning The lingMeaning the sound snippet has to be found for.
 * \return The full qualified path to the sound snippet of the meaning.
 */
char*
ling_meaning_find_sound (char *soundpath, lingMeaning *meaning);

/**
 * Allocates a chunk of memory for usage
 *
 * \param bytes The count of bytes to allocate
 * \return A pointer to the newly allocated space
 */
void*
ling_malloc (size_t bytes);

/**
 * Frees the memory hold by a pointer, which was previously allocated
 * using ling_malloc().
 *
 * \param ptr The pointer to free.
 */
void
ling_free (void *ptr);

/**
 * Prepares and adds a lesson file to an existing list of lessons.
 * 
 * \param lesson The lesson list, the new lesson should be added to.
 * If lesson is NULL, a new list will be created and returned.
 * \param filename The full qualified file path of the file.
 * \param settings The settings to use for the lesson.
 * \return The new lesson list.
 */
lingLesson*
ling_lesson_add_lesson (lingLesson *lesson, char *filename, 
			lingConfig *settings);

/**
 * Removes a lesson from the list of used ones and frees all its 
 * internal references.
 *
 * \param lesson The lesson list to look for the lesson.
 * \param node The lesson to free.
 * \return The modified lesson list.
 */
lingLesson*
ling_lesson_remove_lesson (lingLesson *lesson, lingLesson *node);

/**
 * Returns the full qualified file path of the matching lesson entry.
 *
 * \param lesson The lesson for which the path should be returned.
 * \return The lesson path of the lesson, else NULL.
 */
char*
ling_lesson_return_path (lingLesson *lesson);

/**
 * Modifies the usage flag of a lesson, so that it will be automatically used
 *  by the different meaning access functions.
 *
 * \param lesson The lesson to use
 * \param use A boolean statement for usage (TRUE or FALSE).
 */
void 
ling_lesson_use_lesson (lingLesson *lesson, lingbool use);

/**
 * Returns the status of the use-flag of a lesson.
 *
 * \param lesson The lesson, the usage flag should be returned for.
 * \return TRUE, if the lesson  is currently marked as used
 * in the internal list, else FALSE.
 */
lingbool
ling_lesson_return_used (lingLesson *lesson);

/**
 * Returns the last meaning id of the given lesson.
 * 
 * \param lesson The lesson, for which the max. meaning id should be returned.
 * \return the maximum (last!) meaning id of a lesson file (can be 0, if it 
 * fails).
 */
int
ling_lesson_get_max_meaning (lingLesson *lesson);

/**
 * Saves a lesson into the passed file.
 * If the file does not exist, it will be automatically created, else
 * its contents will be completely overwritten.
 *
 * \param lesson The lesson to save.
 * \param filename The lesson file for saving the lesson.
 * \return TRUE, if the lesson could be saved, else FALSE.
 */
lingbool
ling_lesson_save_lesson (lingLesson *lesson, char *filename);

/**
 * Creates a new template lesson with optional empty meanings.
 *
 * \param filename The lesson file to create.
 * \param meanings The amount of meaning templates to create.
 * \return The filename on success or NULL
 */
char*
ling_lesson_create_new (char *filename, int meanings);

/**
 * Creates a linked list of meanings, which are available in the given file.
 * 
 * \param lesson The lesson, from which the tree should be created.
 * \return A linked list of meanings of the lesson.
 */
lingMeaning*
ling_lesson_create_tree (lingLesson *lesson);

/**
 * Returns a random lesson of the lesson list, which is passed as argument. 
 * Lessons which are not used are ignored. The function tries to determine 
 * an used lesson twenty times. If none is found, NULL will be returned.
 *
 * \param lesson The lesson list to use for the random lesson.
 * \return A random lesson or NULL if none found after twenty rounds.
 */
lingLesson*
ling_lesson_return_rand_lesson (lingLesson *lesson);

/**
 * Gets a NULL terminated list of available languages (based on the language
 * file set in the lingConfig). The return value should to be freed using
 * ling_lang_free_languages().
 *
 * \param settings The lingConfig to use for looking up the languages.
 * \return a NULL terminated list of the languages
 */
lingchar**
ling_lang_get_languages (lingConfig *settings);

/**
 * Gets the count of languages contained in a lingConfig.
 *
 * \param settings The lingConfig to use for looking up the languages.
 * \return An integer showing the maximum count of the languages or 
 * -1 on failure.
 */
int
ling_lang_get_max_languages (lingConfig *settings);

/**
 * Frees a previously allocated array of languages.
 *
 * \param languages The array of languages
 * \param size The size of the array
 */
void 
ling_lang_free_languages (lingchar **languages, int size);

#endif /* LINGOTEACH_H */
