/*!
 * \file mei_node.c
 *
 * \brief Nodal structure of the interpreter
 */

/*
  This file is part of the "Mathematical Expression Interpreter" library.

  Copyright (C) 2008-2009  EDF

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*----------------------------------------------------------------------------
 * Standard C library headers
 *----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

/*----------------------------------------------------------------------------
 * BFT library headers
 *----------------------------------------------------------------------------*/

#include <bft_mem.h>
#include <bft_error.h>

/*----------------------------------------------------------------------------
 * Header for the current file
 *----------------------------------------------------------------------------*/

#include "mei_node.h"

/*----------------------------------------------------------------------------
 * External global variable declarations for the parser
 *----------------------------------------------------------------------------*/

#include "mei_parser_glob.h"

/*----------------------------------------------------------------------------*/

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*============================================================================
 * Public function definitions
 *============================================================================*/

/*----------------------------------------------------------------------------
 *
 *----------------------------------------------------------------------------*/

mei_node_t*
mei_const_node(const double value)
{
    mei_node_t *node = NULL;

    BFT_MALLOC(node, 1, mei_node_t);

    BFT_MALLOC(node->type, sizeof(const_node_t), node_type_t);

    node->flag = CONSTANT;
    node->ht = NULL;
    node->type->con.value = value;

    return node;
}

/*----------------------------------------------------------------------------
 *
 *----------------------------------------------------------------------------*/

mei_node_t*
mei_id_node(char *const variable)
{
    mei_node_t *node = NULL;
    size_t length;

    BFT_MALLOC(node, 1, mei_node_t);

    BFT_MALLOC(node->type, sizeof(id_node_t), node_type_t);

    length = strlen(variable)+1;

    BFT_MALLOC(node->type->id.i, length, char);

    node->flag = ID;
    node->ht = NULL;
    strncpy(node->type->id.i, variable, length);
    node->type->id.c = mei_glob_column-length+1;
    node->type->id.l = mei_glob_line;

    return node;
}

/*----------------------------------------------------------------------------
 *
 *----------------------------------------------------------------------------*/

mei_node_t*
mei_func_node(char *const function, mei_node_t *const expr)
{
    mei_node_t *node = NULL;
    size_t length;
    size_t nodeSize;

    nodeSize = sizeof(func_node_t) + sizeof(mei_node_t);

    BFT_MALLOC(node, 1, mei_node_t);

    BFT_MALLOC(node->type, nodeSize, node_type_t);

    length = strlen(function)+1;
    BFT_MALLOC(node->type->func.name, length, char);

    node->flag = FUNC1;
    node->ht = NULL;
    strncpy(node->type->func.name, function, length);
    node->type->func.op = expr;
    node->type->func.c  = mei_glob_column-length+1;
    node->type->func.l  = mei_glob_line;

    return node;
}

/*----------------------------------------------------------------------------
 *
 *----------------------------------------------------------------------------*/

mei_node_t*
mei_funcx_node(char *const function, const int nops, ...)
{
    va_list ap;
    mei_node_t *node = NULL;
    size_t length;
    size_t nodeSize;
    int i;

    nodeSize = sizeof(func2_node_t) + nops*sizeof(mei_node_t);

    BFT_MALLOC(node, 1, mei_node_t);

    BFT_MALLOC(node->type, nodeSize, node_type_t);

    length = strlen(function)+1;
    BFT_MALLOC(node->type->funcx.name, length, char);
    strncpy(node->type->funcx.name, function, length);

    if (nops == 2)
    {
        node->flag = FUNC2;
    }
    else if (nops == 3)
    {
        node->flag = FUNC3;
    }
    else if (nops == 4)
    {
        node->flag = FUNC4;
    }
    else
        bft_error(__FILE__, __LINE__, 0,
        "Error: number of arguments for the function is to long\n");


    node->ht = NULL;

    node->type->funcx.nops = nops;
    va_start(ap, nops);
    for (i = 0; i < nops; i++)
        node->type->funcx.op[i] = va_arg(ap, mei_node_t*);
    va_end(ap);

    return node;
}

/*----------------------------------------------------------------------------
 *
 *----------------------------------------------------------------------------*/

mei_node_t*
mei_opr_node(const int oper, const int nops, ...)
{
    va_list ap;
    mei_node_t *node = NULL;
    size_t nodeSize;
    int i;

    nodeSize = sizeof(opr_node_t) + nops*sizeof(mei_node_t);

    BFT_MALLOC(node, 1, mei_node_t);

    BFT_MALLOC(node->type, nodeSize, node_type_t);

    node->flag = OPR;
    node->ht = NULL;
    node->type->opr.oper = oper;
    node->type->opr.nops = nops;
    va_start(ap, nops);
    for (i = 0; i < nops; i++)
        node->type->opr.op[i] = va_arg(ap, mei_node_t*);
    va_end(ap);

    return node;
}

/*!
 * \brief Return a label for a node.
 *
 * \param [in] n node
 */

char *
mei_label_node(mei_node_t *n)
{
    char *buff;

    if (n->flag == CONSTANT)
    {
        BFT_MALLOC(buff, 256, char);
        sprintf(buff, "%f", n->type->con.value);
        return buff;
    }
    else if (n->flag == ID)
    {
        return n->type->id.i;
    }
    else if (n->flag == FUNC1)
    {
        return n->type->func.name;
    }
    else if (n->flag == FUNC2 || n->flag == FUNC3 || n->flag == FUNC4)
    {
        return n->type->funcx.name;
    }
    else if (n->flag == OPR)
    {
        BFT_MALLOC(buff, 256, char);
        sprintf(buff, "operator number: %d", n->type->opr.nops);
        return buff;
    }

    BFT_MALLOC(buff, 256, char);
    sprintf(buff, "%s", " ");
    return buff;
}

/*!
 * \brief Free memory.
 *
 * \param [in] n node
 */

void
mei_free_node(mei_node_t *n)
{
    int i;

    if (!n) return;

    if (n->flag == ID)
    {
        BFT_FREE(n->type->id.i);
    }
    else if (n->flag == FUNC1)
    {
        BFT_FREE(n->type->func.name);
        mei_free_node(n->type->func.op);
    }
    else if (n->flag == FUNC2 || n->flag == FUNC3 || n->flag == FUNC4)
    {
        BFT_FREE(n->type->funcx.name);
        for (i = 0; i < n->type->funcx.nops; i++)
            mei_free_node(n->type->funcx.op[i]);
    }
    else if (n->flag == OPR)
    {
        for (i = 0; i < n->type->opr.nops; i++)
            mei_free_node(n->type->opr.op[i]);
    }
    BFT_FREE(n->type);
    BFT_FREE(n);
}

/*----------------------------------------------------------------------------*/

#ifdef __cplusplus
}
#endif /* __cplusplus */
